/**
 ****************************************************************************
 * @file     mcuip_drv.h
 * @brief    PMD and VE driver
 * @version  V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2021
 *****************************************************************************
 */
#ifndef _MCUIP_DRV_H
#define _MCUIP_DRV_H

/*===================================================================*
	  Register bit position Definition
 *===================================================================*/
/* VExFMODE */
#define	B_SPWMEN				1U
#define	B_C2PEN					0U

/* VExMCTLF */
#define	B_LAVF					0U
#define	B_LAVFM					1U
#define	B_PLSLF					4U

/* PMDxEMGSTA */
#define	B_EMGST					0U
#define	B_EMGI					1U

/* PMDxEMGCR */
#define	B_EMGEN					0U
#define	B_EMGRS					1U

/* PMDxOVVCR */
#define	B_OVVEN					0U

/* PMDxMDPOT */
#define	B_POLL					2U
#define	B_POLH					3U


/*===================================================================*
	  Constant Data Definition
 *===================================================================*/
/* PMD Trigger Interrupt Select Registers */
#define	cPINTS_A				0U
#define	cPINTS_B				1U

/* PMD CH */
#define	cPMD0					0U
#define	cPMD1					1U

/** For PMD,VE ******************************************************/

/* Define the constant data for MODESEL */
#define WBUF_BUS				0								/* W-BUF data from Bus (by software) */
#define WBUF_VE 				1								/* W-BUF data from VE (automatic)	 */


/****************************************************************************************/
/* Define the constant data for MDCR */
/****************************************************************************************/
/* PWM carrier waveform */
#define	PWMMD_SAWTOOTH			0U								/* edge-aligned PWM, sawtooth wave */
#define	PWMMD_TRIANGULAR		1U								/* center-aligned PWM, triangular wave */

/* PWM interrupt period */
#define	INTPRD_HALF				0U								/* Interrupt request at every 0.5 PWM period (<PWMMD>= "1" only) */
#define	INTPRD_1				1U								/* Interrupt request at every PWM period */
#define	INTPRD_2				2U								/* Interrupt request at every 2 PWM periods */
#define	INTPRD_4				3U								/* Interrupt request at every 4 PWM periods */

/* PWM interrupt timing */
#define	PINT_BOTTOM				0U								/* Interrupt request when PWM counter PMDxMDCNT<MDCNT[15:0]> = 0x0001 */
#define	PINT_TOP				1U								/* Interrupt request when PWM counter PMDxMDCNT<MDCNT[15:0]> = <MDPRD[15:0]> */

/* Duty mode */
#define	DTYMD_COMMON			0U								/* 0: 3-phase common mode */
#define	DTYMD_INDEPENDENT		1U								/* 1: 3-phase independent mode */

/* Port output mode */
#define	SYNTMD_0				0U
#define	SYNTMD_1				1U

/* PWM period extension mode */
#define	PWMCK_NORMAL			0U								/* Normal period */
#define	PWMCK_4FOLD				1U								/* 4x period */

/*--------------------------------------------------------------------------------------*/
#define	cPWMEXCLK				PWMCK_NORMAL					/* Divide PWM clock */
#define	cPWMSYNT				SYNTMD_0						/* Special PWM mode */
#define	cPWMSPCFY				DTYMD_INDEPENDENT				/* Duty mode */
#define	cPWMINT					PINT_BOTTOM						/* INTPWM timing */
#define	cPWMINTPRD				INTPRD_1						/* INTPWM period */
#define	cPWMWAVE				PWMMD_TRIANGULAR				/* PWM mode */
/*--------------------------------------------------------------------------------------*/
#define cMDCR_INI				( (cPWMEXCLK << 6 ) | (cPWMSYNT   << 5 ) | (cPWMSPCFY << 4) \
								| (cPWMINT	 << 3 ) | (cPWMINTPRD << 1 ) | (cPWMWAVE  << 0))
/****************************************************************************************/


/****************************************************************************************/
/* Define the constant data for MDOUT */
/****************************************************************************************/
#define cMDOUT_OFF				BIT16(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0)	/* OUTPUT all L */
#define cMDOUT_ON				BIT16(0,0,0,0,0,1,1,1,0,0,1,1,1,1,1,1)	/* OUTPUT all PWM/~PWM */
								/* 0  | | | | | | | | | | | | | | | +--	UOC X out 0:OFF 1:ON	*/
								/* 1  | | | | | | | | | | | | | | +----		U out 0:OFF 1:ON	*/
								/* 2  | | | | | | | | | | | | | +------	VOC Y out 0:OFF 1:ON	*/
								/* 3  | | | | | | | | | | | +--------		V out 0:OFF 1:ON	*/
								/* 4  | | | | | | | | | | | +----------	WOC Z out 0:OFF 1:ON	*/
								/* 5  | | | | | | | | | | +------------		W out 0:OFF 1:ON	*/
								/* 6  | | | | | | | | | +--------------	0						*/
								/* 7  | | | | | | | | +----------------	0						*/
								/* 8  | | | | | | | +------------------	UPWM U out 0:H/L 1:PWM	*/
								/* 9  | | | | | | +--------------------	UPWM U out 0:H/L 1:PWM	*/
								/*10  | | | | | +----------------------	UPWM U out 0:H/L 1:PWM	*/
								/*11  | | | | +------------------------	0						*/
								/*12  | | | +--------------------------	0						*/
								/*13  | | +----------------------------	0						*/
								/*14  | +------------------------------	0						*/
								/*15  +--------------------------------	0						*/

/****************************************************************************************/

/****************************************************************************************/
/* Define the constant data for MDPOT */
/****************************************************************************************/
/* Selects PMDxMDOUT(VExOUTCR) transfer timing (PWM synchronous setting). */
#define	PSYNCS_WRITE			0U								/* Reflect when write */
#define	PSYNCS_BOTTOM			1U								/* Reflect when PWM counter MDCNT="1"(minimum) */
#define	PSYNCS_TOP				2U								/* Reflect when PWM counter MDCNT=PMDxMDPRD<MDPRD>(maximum) */
#define	PSYNCS_BOTTOM_TOP		3U								/* Reflect when PWM counter MDCNT="1"(minimum) or PMDxMDPRD<MDPRD>(maximum) */

/* Port polarity */
#define	POLH_LACT				0U
#define	POLH_HACT				1U
#define	POLL_LACT				0U
#define	POLL_HACT				1U

/*--------------------------------------------------------------------------------------*/
#define	cPSYNCS					(PSYNCS_TOP)
/*--------------------------------------------------------------------------------------*/
#define cMDPOT_INI				((cPSYNCS << 0))
/****************************************************************************************/


/****************************************************************************************/
/* Define the constant data for PORTMD */
/****************************************************************************************/
/* Port control settings at tool break. */
#define	PORTMD_ALLHIZ			0U								/* Upper phases = High-z / lower phases = High-z */
#define	PORTMD_UHIZ_LON			1U								/* Upper phases = High-z / ower phases = PMD output */
#define	PORTMD_UON_LHIZ			2U								/* Upper phases = PMD output / lower phases = High-z */
#define	PORTMD_ALLON			3U								/* Upper phases = PMD output / lower phases = PMD output */

/*--------------------------------------------------------------------------------------*/
#define	cPORTMD					PORTMD_ALLHIZ
/*--------------------------------------------------------------------------------------*/
#define	cPORTMD_INI				((cPORTMD) << 0)
/****************************************************************************************/

/****************************************************************************************/
/* Define the constant data for TRGCR */
/****************************************************************************************/
#define	TRGMD_DIS				0U								/* Trigger output disabled */
#define	TRGMD_DOWN				1U								/* Trigger output at down-count match */
#define	TRGMD_UP				2U								/* Trigger output at up-count match */
#define	TRGMD_UPDOWM			3U								/* Trigger output at up-/down-count match */
#define	TRGMD_PEAK				4U								/* Trigger output at PWM carrier peak */
#define	TRGMD_BOTTOM			5U								/* Trigger output at PWM carrier bottom */
#define	TRGMD_PEAKBOTTOM		6U								/* Trigger output at PWM carrier peak/bottom */
#define	TRGMD_DIS7				7U								/* Trigger output disabled */

#define	TRGBE_SYNC				0U								/* sync */
#define	TRGBE_ASYNC				1U								/* The value written to PMDTRGx is immediately reflected.) */


/*--------------------------------------------------------------------------------------*/
#define	cTRGMD_3SHT				TRGMD_PEAK						/* Trigger timing setting for 3shunt */
#define	cTRGMD_1SHT				TRGMD_UP						/* Trigger timing setting for 1shunt */
#define	cBUFSYNC				TRGBE_SYNC						/* Triger Buffer update timing */
/*--------------------------------------------------------------------------------------*/
/* triger setting for 3shunt */
#define cTRG0_3SHUNTSETTING		((cBUFSYNC << 3) | (cTRGMD_3SHT))
#define cTRG1_3SHUNTSETTING		((cBUFSYNC << 3) | (TRGMD_DIS))
#define cTRG2_3SHUNTSETTING		((cBUFSYNC << 3) | (TRGMD_DIS))
#define cTRG3_3SHUNTSETTING		((cBUFSYNC << 3) | (TRGMD_DIS))

/* triger setting for 1shunt */
#define cTRG0_1SHUNTSETTING		((cBUFSYNC << 3) | (cTRGMD_1SHT))
#define cTRG1_1SHUNTSETTING		((cBUFSYNC << 3) | (cTRGMD_1SHT))
#define cTRG2_1SHUNTSETTING		((cBUFSYNC << 3) | (TRGMD_PEAK))
#define cTRG3_1SHUNTSETTING		((cBUFSYNC << 3) | (TRGMD_DIS))
/*--------------------------------------------------------------------------------------*/
#define cTRG_3SHUNT 	 		((cTRG3_3SHUNTSETTING << 12) | (cTRG2_3SHUNTSETTING << 8) | (cTRG1_3SHUNTSETTING << 4) | (cTRG0_3SHUNTSETTING << 0))
#define cTRG_1SHUNT 			((cTRG3_1SHUNTSETTING << 12) | (cTRG2_1SHUNTSETTING << 8) | (cTRG1_1SHUNTSETTING << 4) | (cTRG0_1SHUNTSETTING << 0))
/****************************************************************************************/


/****************************************************************************************/
/* Define the constant data for TRGMD */
/****************************************************************************************/
/* Trigger output mode */
#define	TRGOUT_FIX				0U								/* Fixed trigger output */
#define	TRGOUT_SEL				1U								/* Variable trigger output */

/* Output enable in EMG protection state */
#define	EMGTGE_DISABLE			0U								/* Disable trigger output in the protection state */
#define	EMGTGE_ENABLE			1U								/* Enable trigger output in the protection state */

/*--------------------------------------------------------------------------------------*/
#define	cEMGTGE					EMGTGE_ENABLE
/*--------------------------------------------------------------------------------------*/
#define	cTRGMD_3SHUNT			( (TRGOUT_SEL << 1) | (cEMGTGE << 0) )
#define	cTRGMD_1SHUNT			( (TRGOUT_FIX << 1) | (cEMGTGE << 0) )
/****************************************************************************************/


/****************************************************************************************/
/* Define the constant data for EMGCR */
/****************************************************************************************/
/* Tool break enable/disable */
#define	INHEN_DISABLE			0U
#define	INHEN_ENABLE			1U

/* EMG protection mode select */
#define	EMGMD_ALLHIZ			0U								/* PWM output control disabled / Port output = All phases High-Z */
#define	EMGMD_UON_LHIZ			1U								/* All upper phases ON, all lower phases OFF / Port output = Lower phases High-Z */
#define	EMGMD_UHIZ_LON			2U								/* All upper phases OFF, all lower phases ON / Port output = Upper phases High-Z */
#define	EMGMD_ALLHIZ2			3U								/* All phases OFF / Port output = All phases High-Z */

/* EMG protection release */
#define	EMGRS_NOP				0U
#define	EMGRS_RELEASE			1U

/* EMG protection circuit enable/disable */
#define	EMGEN_DISABLE			0U
#define	EMGEN_ENABLE			1U

/*--------------------------------------------------------------------------------------*/
#define	cEMGCNT					5U								/* Detection counter unit.16/fsys  ex.200ns@80MHz */
#define	cINHEN					INHEN_ENABLE					/* Tool Break */
#define	cEMGMD					EMGMD_ALLHIZ					/* Stop mode */
/*--------------------------------------------------------------------------------------*/
#define cEMG_ENA				((cEMGCNT << 8) | (cINHEN << 5) | (cEMGMD << 3) | (EMGEN_ENABLE << 0))
#define cEMG_DIS				((cEMGCNT << 8) | (cINHEN << 5) | (cEMGMD << 3) | (EMGEN_DISABLE << 0) )
#define	cEMG_Release			((cEMGCNT << 8) | (cINHEN << 5) | (cEMGMD << 3) | (EMGRS_RELEASE << 1) | (EMGEN_ENABLE << 0))
/****************************************************************************************/


/****************************************************************************************/
/* Define the constant data for OVVCR */
/****************************************************************************************/
/* ADC B monitor interrupt input enable */
#define	ADIN1EN_DISABLE			0U
#define	ADIN1EN_ENABLE			1U

/* ADC A monitor interrupt input enable */
#define	ADIN0EN_DISABLE			0U
#define	ADIN0EN_ENABLE			1U

/* OVV protection mode */
#define	OVVMD_NOCON				0U
#define	OVVMD_UON_LOFF			1U
#define	OVVMD_UOFF_LON			2U
#define	OVVMD_ALLOFF			3U


/* OVV input select */
#define	OVVISEL_PORT			0U
#define	OVVISEL_ADC				1U

/* OVV protection release */
#define	OVVRS_NORMAL			0U
#define	OVVRS_AUTO				1U

/* OVV protection circuit enable/disable */
#define	OVVEN_DISABLE			0U
#define	OVVEN_ENABLE			1U

/*--------------------------------------------------------------------------------------*/
#define	cOVVCNT					5U								/* Detection counter unit.16/fsys  ex.200ns@80MHz */
#define	cADIN1EN				ADIN1EN_DISABLE					/* AD1 watch input */
#define	cADIN0EN				ADIN0EN_DISABLE					/* AD0 watch input */
#define	cOVVMD					OVVMD_ALLOFF					/* Stop mode */
#define	cOVVISEL				OVVISEL_PORT					/* Sourse of OVV */
#define	cOVVRS					OVVRS_NORMAL					/* Auto release */
/*--------------------------------------------------------------------------------------*/
#define cOVV_ENA				( (cOVVCNT << 8) | (cADIN1EN << 6) | (cADIN0EN << 5) | \
								  (cOVVMD << 3) | (cOVVISEL << 2) | (cOVVRS << 1) | (OVVEN_ENABLE << 0) )

#define cOVV_DIS				( (cOVVCNT << 8) | (cADIN1EN << 6) | (cADIN0EN << 5) | \
								  (cOVVMD << 3) | (cOVVISEL << 2) | (cOVVRS << 1) | (OVVEN_DISABLE << 0) )
/****************************************************************************************/


/** For VE ******************************************************/

#define	cTADC					(2e-6)							/* [s] AD convert time for 1shunt shift pwm */

/****************************************************************************************/
/* Define the constant data for FMODE */
/****************************************************************************************/
/* Keep the previous value of SIN/COS/SECTOR */
#define	MREGDIS_EFFECTIVE		0U
#define	MREGDIS_NOEFFECTIVE		1U

/* Current detection mode */
#define IDMODE_3SHUNT			0U								/* 3-shunt */
#define IDMODE_2SENSOR			1U								/* 2-sensor */
#define IDMODE_1SHUNT_UP		2U								/* 1-shunt (for up count PMDTRG) */
#define IDMODE_1SHUNT_DOWN		3U								/* 1-shunt (for down count PMDTRG) */

/* ADC unit */
#define	ADCSEL_UNITA			0U
#define	ADCSEL_UNITB			1U
#define	ADCSEL_UNITAB			2U
#define	ADCSEL_UNITAB1			3U

/* PMD channel */
#define	PMDSEL_CH0				0U
#define	PMDSEL_CH1				1U

/*--------------------------------------------------------------------------------------*/
#define	cMREGDIS				(MREGDIS_EFFECTIVE)
#define cADCSEL1				(ADCSEL_UNITB)
#define cADCSEL0				(ADCSEL_UNITA)

/*--------------------------------------------------------------------------------------*/
#define	cFMODE1_INI_3SHUNT		( (cMREGDIS << 9) | (cADCSEL1 << 6) | (PMDSEL_CH1 << 4) | (IDMODE_3SHUNT    << 2) )
#define	cFMODE1_INI_1SHUNT		( (cMREGDIS << 9) | (cADCSEL1 << 6) | (PMDSEL_CH1 << 4) | (IDMODE_1SHUNT_UP << 2) )
#define	cFMODE0_INI_3SHUNT		( (cMREGDIS << 9) | (cADCSEL0 << 6) | (PMDSEL_CH0 << 4) | (IDMODE_3SHUNT    << 2) )
#define	cFMODE0_INI_1SHUNT		( (cMREGDIS << 9) | (cADCSEL0 << 6) | (PMDSEL_CH0 << 4) | (IDMODE_1SHUNT_UP << 2) )
/****************************************************************************************/


/****************************************************************************************/


/****************************************************************************************/
#define	TAPP_OUT_CON			0U			/* Output control 1 */
#define	TAPP_TRG_GEN			1U			/* Trigger generation */
#define	TAPP_IN_PRC				2U			/* Input process 1 */
#define	TAPP_IN_PHSCNV			3U			/* Input phase transformation */
#define	TAPP_IN_AXISCNV			4U			/* Input coordinate axis transformation */
#define	TAPP_CRT_CON			5U			/* Current control */
#define	TAPP_SINCOS				6U			/* SIN/COS calculation */
#define	TAPP_OUT_AXISCNV		7U			/* Output coordinate axis transformation */
#define	TAPP_OUT_PHSCNV			8U			/* Output phase transformation 1[SVM] */

#define	SCH_1					1U
#define	SCH_4					4U
#define	SCH_9					9U
/****************************************************************************************/


/****************************************************************************************/
/* VETRGMODE */
/****************************************************************************************/
#define	TRGMODE_IGNORE			0U
#define	TRGMODE_UNITA			1U
#define	TRGMODE_UNITB			2U
#define TRGMODE_UNITAB			3U
/*--------------------------------------------------------------------------------------*/
#define	cTRGMODE1_INI			(TRGMODE_UNITB)
#define	cTRGMODE0_INI			(TRGMODE_UNITA)
/****************************************************************************************/


/****************************************************************************************/
/* VEERRINTEN */
/****************************************************************************************/
/* Controls an interrupt at error detection. */
#define	VERREN_DISABLE			0U
#define	VERREN_ENABLE			1U
/*--------------------------------------------------------------------------------------*/
#define	cVERREN_INI				(VERREN_DISABLE)
/****************************************************************************************/

/****************************************************************************************/
/* VECOMPEND */
/****************************************************************************************/
#define	VCEND_TERMINATE			1U
/*--------------------------------------------------------------------------------------*/
#define	VE_FORCE_STOP_CH0		(VCEND_TERMINATE << 0U)
#define	VE_FORCE_STOP_CH1		(VCEND_TERMINATE << 1U)
/****************************************************************************************/

/****************************************************************************************/
/* VEMODE */
/****************************************************************************************/
/* Output control */
#define	OCRMD_OUT_OFF			0U
#define	OCRMD_OUT_ON			1U
#define	OCRMD_OUT_OFF_EMGRET	3U

/* Zero current detection */
#define	ZIEN_DISABLE			0U
#define	ZIEN_ENABLE				1U

/* Phase interpolation */
#define	PVIEN_DISABLE			0U
#define	PVIEN_ENABLE			1U

/*--------------------------------------------------------------------------------------*/
#define	cZIEN					(ZIEN_DISABLE)					/* Zero current detection */
#define	cPVIEN					(PVIEN_DISABLE)					/* Phase interpolation */
/*--------------------------------------------------------------------------------------*/
#define	VEMODE_STOP				( (OCRMD_OUT_OFF << 2) | (cZIEN << 1) | (cPVIEN << 0) )
#define	VEMODE_BOOT				( (OCRMD_OUT_ON << 2)  | (cZIEN << 1) | (cPVIEN << 0) )
#define	VEMODE_INITPOS			( (OCRMD_OUT_ON << 2)  | (cZIEN << 1) | (cPVIEN << 0) )
#define	VEMODE_FORCE			( (OCRMD_OUT_ON << 2)  | (cZIEN << 1) | (cPVIEN << 0) )
#define	VEMODE_CHGUP			( (OCRMD_OUT_ON << 2)  | (cZIEN << 1) | (cPVIEN << 0) )
#define	VEMODE_STEADY			( (OCRMD_OUT_ON << 2)  | (cZIEN << 1) | (cPVIEN << 0) )
#define	VEMODE_EMG				( (OCRMD_OUT_OFF << 2) | (cZIEN << 1) | (cPVIEN << 0) )
/****************************************************************************************/

/****************************************************************************************/
/* VECPURUNTRG */
/****************************************************************************************/
#define	VCPURT_START			1U
/*--------------------------------------------------------------------------------------*/
#define	VE_START_CH0			(VCPURT_START << 0U)
#define VE_START_CH1			(VCPURT_START << 1U)
/****************************************************************************************/


#define	SET_TAPP_CH0(x)			(x << 0U)
#define	SET_TAPP_CH1(x)			(x << 4U)
#define	MASK_TAPP_CH0			((uint32_t)0xFFFFFFF0)
#define	MASK_TAPP_CH1			((uint32_t)0xFFFFFF0F)

#define	SET_ACTSCH_CH0(x)		(x << 0U)
#define	SET_ACTSCH_CH1(x)		(x << 4U)
#define	MASK_ACTSCH_CH0			((uint32_t)0xFFFFFFF0)
#define	MASK_ACTSCH_CH1			((uint32_t)0xFFFFFF0F)

#define	SET_REPTIME_CH0(x)		(x << 0U)
#define	SET_REPTIME_CH1(x)		(x << 4U)

#define	SET_TRGMODE_CH0(x)		(x << 0U)
#define	SET_TRGMODE_CH1(x)		(x << 2U)

#define	SET_ERRINTEN_CH0(x)		(x << 0U)
#define	SET_ERRINTEN_CH1(x)		(x << 1U)


/** For Encoder ******************************************************/
#define	cENCMULTIPLE			(4)


/** For AMP **********************************************************/
#define	AMPGAIN_FOLD_1_5		0U				/* *1.5 */
#define	AMPGAIN_FOLD_2_5		1U				/* *2.5 */
#define	AMPGAIN_FOLD_3_0		2U				/* *3.0 */
#define	AMPGAIN_FOLD_3_5		3U				/* *3.5 */
#define	AMPGAIN_FOLD_4_0		4U				/* *4.0 */
#define	AMPGAIN_FOLD_6_0		5U				/* *6.0 */
#define	AMPGAIN_FOLD_8_0		6U				/* *8.0 */
#define	AMPGAIN_FOLD_10_0		7U				/* *10.0 */
/*--------------------------------------------------------------------------------------*/
#define	AMP_ENABLE(fold)		((uint8_t)((fold << 1) | 0x01U))
#define	AMP_DISABLE(fold)		((uint8_t)(fold << 1))
/****************************************************************************************/


/** For CMP **********************************************************/
#define	CMPSEL_RAW				0U
#define	CMPSEL_OPAMP			1U
/*--------------------------------------------------------------------------------------*/
#define	CMP_ENABLE(sel)			((uint8_t)((sel << 1) | 0x01U))
#define	CMP_DISABLE(sel)		((uint8_t)(sel << 1))
/****************************************************************************************/


/*===================================================================*
	  Typedef Definition
 *===================================================================*/
typedef enum
{
	cNormal		 	 = 0,
	cEMGProtected	 = 1
} emg_status_e;

/**
  * @brief	PMD Init Structure definition
  */
typedef struct
{
	uint8_t		shunt;
	uint8_t		poll;
	uint8_t		polh;
	uint16_t	pwmfreq;
	uint16_t	deadtime;
} PMD_InitTypeDef;

/**
  * @brief	VE Init Structure definition
  */
typedef struct
{
	uint8_t		ve_ch;
	uint8_t		shunt;
	uint16_t	pwmfreq;

	uint16_t	reptime;
	uint16_t	trgmode;
	uint16_t	tpwm;

	uint16_t	idkp;
	uint16_t	idki;
	uint16_t	iqkp;
	uint16_t	iqki;

	uint16_t	zerooffset;

} VE_InitTypeDef;


/**
  * @brief	Encoder Init Structure definition
  */
typedef struct
{
	uint16_t	pulsenum;
} EN_InitTypeDef;

/**
  * @brief	Encoder Init Structure definition
  */
typedef struct
{
	uint8_t		shunt;
	uint8_t		iuch;
	uint8_t		ivch;
	uint8_t		iwch;
	uint8_t		idcch;
	uint8_t		vdcch;
	uint8_t		pmd_ch;
	uint8_t		pints;
} AD_InitTypeDef;

/*===================================================================*
	  Proto Type Definition
 *===================================================================*/
emg_status_e	PMD_GetEMG_Status(const ipdrv_t* const _ipdrv);
void	PMD_ReleaseEMG_Protection(const ipdrv_t* const _ipdrv);
void	VE_Start(const ipdrv_t* const _ipdrv);
void	VE_GetPhaseCurrent(const ipdrv_t* const _ipdrv, q15_t* _ia, q15_t* _ib, q15_t* _ic);
void	VE_GetCurrentAdcData(const ipdrv_t* const _ipdrv, uint32_t* _adc_ia, uint32_t* _adc_ib, uint32_t* _adc_ic);
void	VE_GetdataFromVEreg(const ipdrv_t* const _ipdrv, vector_t* const _motor);
uint32_t	VE_GetPWM_Period(const ipdrv_t* const _ipdrv);
uint32_t	VE_GetPWM_DutyU(const ipdrv_t* const _ipdrv);
uint32_t	VE_GetPWM_DutyV(const ipdrv_t* const _ipdrv);
uint32_t	VE_GetPWM_DutyW(const ipdrv_t* const _ipdrv);
uint32_t	VE_GetPWM_DutyMed(const ipdrv_t* const _ipdrv);
int		VE_GetPWM_Modulation(const ipdrv_t* const _ipdrv);
uint32_t	VE_GetSector(const ipdrv_t* const _ipdrv);
int		VE_GetShiftPWMState(const ipdrv_t* const _ipdrv);
int		VE_GetOutputMode(const ipdrv_t* const _ipdrv);
void	VE_SetdataToVEreg_Stop(const ipdrv_t* const _ipdrv, const vector_t* const _motor);
void	VE_SetdataToVEreg_Bootstrap(const ipdrv_t* const _ipdrv, const vector_t* const _motor);
void	VE_SetdataToVEreg_Initposition_i(const ipdrv_t* const _ipdrv, const vector_t* const _motor);
void	VE_SetdataToVEreg_Initposition_v(const ipdrv_t* const _ipdrv, const vector_t* const _motor);
void	VE_SetdataToVEreg_Force_i(const ipdrv_t* const _ipdrv, const vector_t* const _motor);
void	VE_SetdataToVEreg_Force_v(const ipdrv_t* const _ipdrv, const vector_t* const _motor);
void	VE_SetdataToVEreg_Change_up(const ipdrv_t* const _ipdrv, const vector_t* const _motor);
void	VE_SetdataToVEreg_Steady_A(const ipdrv_t* const _ipdrv, const vector_t* const _motor);
void	VE_SetdataToVEreg_Emergency(const ipdrv_t* const _ipdrv, const vector_t* const _motor);
void	VE_SetZeroCurrentData(const ipdrv_t* const _ipdrv, uint32_t _z_ia, uint32_t _z_ib, uint32_t _z_ic);
void	VE_SetVDCreg(const ipdrv_t* const _ipdrv, q15_t _dat);
void	VE_SetSpwmMode(const ipdrv_t* const _ipdrv, uint8_t _dat);
void	VE_SetModulType(const ipdrv_t* const _ipdrv, uint8_t _dat);
void	IP_PMD_init(TSB_PMD_TypeDef* const PMDx, PMD_InitTypeDef* const _initdata);
void	IP_VE_init(TSB_VE_TypeDef* const VEx, VE_InitTypeDef* const _initdata);
void	IP_ADC_init(TSB_AD_TypeDef* const ADx, AD_InitTypeDef* const _initdata);
void	IP_AMP_init(void);

#endif /* _MCUIP_DRV_H */

/*********************************** END OF FILE ******************************/


