/**
 ****************************************************************************
 * @file     B_User.c
 * @brief    Motor control for user Source File
 * @version  V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2021
 *****************************************************************************
 */
#include <stdlib.h>

#include "ipdefine.h"
#include "mcuip_drv.h"
#include "usercon.h"

#define DEFINE_APP
#include "B_User.h"
#undef  DEFINE_APP


/*===================================================================*
	  Local Variable Definition
 *===================================================================*/
int32_t	target_spd1 = cSPEED_ACT_CH1;

/*===================================================================*
	Function
 *===================================================================*/
/**
 *********************************************************************************************
 * @brief		Initialize at Motor control
 *
 * @return		none
 *********************************************************************************************
 */
void	B_Motor_Init(void)
{
	/***** Data Initialize for ch1 ********************************************/

	/***** Setting Motor type  *****/
	Motor_ch1.shunt_type = (shunt_type_e)cSHUNT_TYPE_CH1;
	Motor_ch1.boot_type = cBOOT_TYPE_CH1;

	/***** Initialize variable *****/
	Motor_ch1.stage.main = cStop;
	Motor_ch1.stage.sub = cStep0;

	Motor_ch1.drv.Iao_ave.word = Motor_ch1.drv.Ibo_ave.word = Motor_ch1.drv.Ico_ave.word = ADC_V2HEX(cSHUNT_ZERO_OFFSET_CH1) << 16;

	/***** user data setting *****/
	/* Start Current (Iq) command */
	Motor_ch1.usr.Iq_st_user = (q15_t)(cIQ_ST_USER_ACT_CH1 * FIX_15 / cA_MAX_CH1);

	/* Start Current (Id) command */
	Motor_ch1.usr.Id_st_user = (q15_t)(cID_ST_USER_ACT_CH1 * FIX_15 / cA_MAX_CH1);

	/* Set initial position */
	Motor_ch1.usr.lambda_user = ELE_DEG(cINITIAL_POSITION_CH1);

	/* Set Phase 3phase:0 2phase:1 */
#if (cSHUNT_TYPE_CH1 == 1)
	Motor_ch1.usr.com_user.modul = c2PHASE;
#else
	Motor_ch1.usr.com_user.modul = c3PHASE;
#endif
	/* Set Shift-PWM ON:1 OFF:0 for 1shunt */
	Motor_ch1.usr.com_user.spwm = 1;

	/***** Set parameter *****/
	/* Motor parameter */
	Motor_ch1.para.motor.r = (q31_t)(FIX_15 * cMOTOR_R_CH1 * cA_MAX_CH1 / cV_MAX_CH1);
	Motor_ch1.para.motor.Lq = (q31_t)(FIX_15 * PAI2 * cA_MAX_CH1 * cHZ_MAX_CH1 * cMOTOR_LQ_CH1 / 1000 / cV_MAX_CH1);
	Motor_ch1.para.motor.Ld = (q31_t)(FIX_15 * PAI2 * cA_MAX_CH1 * cHZ_MAX_CH1 * cMOTOR_LD_CH1 / 1000 / cV_MAX_CH1);

	/* Set Shift-PWM threshold for 1shunt */
	Motor_ch1.para.spwm_threshold = (q15_t)(cHZ_SPWM_CH1 * FIX_15 / cHZ_MAX_CH1);

	/* Set check pulse width */
#if defined(__A_PMD)
 #if (cSHUNT_TYPE_CH1 == 1)
   /* for 1shunt VE */
	Motor_ch1.para.chkpls = (uint16_t)(FIX_15 * cMINPLS_CH1 / cPWMPRD_CH1) * 2;
 #else /* cSHUNT_TYPE_CH1 == 1 */
   /* for 3shunt VE */
	/* Convert On width to Off width for VE Register. */
	Motor_ch1.para.chkpls = (uint16_t)(FIX_15 - (FIX_15 * cMAXPLS_CH1 / cPWMPRD_CH1));
 #endif /* cSHUNT_TYPE_CH1 == 1 */
#else /* defined(__A_PMD) */
 #if (cSHUNT_TYPE_CH1 == 1)
	Motor_ch1.para.chkpls = (uint16_t)(cMINPLS_CH1 * cIMCLK_FRQ);
 #else
	Motor_ch1.para.chkpls = (uint16_t)(cMAXPLS_CH1 * cIMCLK_FRQ);
 #endif
#endif /* defined(__A_PMD) */

	Motor_ch1.para.vd_pos = (q31_t)(FIX_31 * cVD_POS_CH1 / cV_MAX_CH1);
	Motor_ch1.para.spd_coef = (q31_t)(FIX_15 * cSPD_COEF_CH1);

	/* Acceleration setting */
	Motor_ch1.para.sp_ud_lim_f.word = (q31_t)(FIX_31 * cMAINLOOP_PRD * cFCD_UD_LIM_CH1 / cHZ_MAX_CH1);	/* Speed up/down limit at Force */
	Motor_ch1.para.sp_up_lim_s.word = (q31_t)(FIX_31 * cMAINLOOP_PRD * cSTD_UP_LIM_CH1 / cHZ_MAX_CH1);	/* Speed up limit at Steady */
	Motor_ch1.para.sp_dn_lim_s.word = (q31_t)(FIX_31 * cMAINLOOP_PRD * cSTD_DW_LIM_CH1 / cHZ_MAX_CH1);	/* Speed down limit at Steady */

	/* Time setting */
	Motor_ch1.para.time.bootstp = (uint16_t)(cBOOT_LEN_CH1 / cMAINLOOP_PRD); 							/* Time of Bootstrap */
	Motor_ch1.para.time.initpos = (uint16_t)(cINIT_LEN_CH1 / cMAINLOOP_PRD);							/* Time of Positioning */
	Motor_ch1.para.time.initpos2 = (uint16_t)(cINIT_WAIT_LEN_CH1 / cMAINLOOP_PRD);						/* Time of Positioning wait */
	Motor_ch1.para.time.go_up = (uint16_t)(cGOUP_DELAY_LEN_CH1 / cMAINLOOP_PRD);						/* Time of Change-up */

	/* Omega setting */
	Motor_ch1.para.omega_min = (q15_t)(FIX_15 * cHZ_MIN_CH1 / cHZ_MAX_CH1);
	Motor_ch1.para.omega_v2i = (q15_t)(FIX_15 * cHZ_V2I_CH1 / cHZ_MAX_CH1);
	Motor_ch1.para.delta_lambda = ((q31_t)(FIX_15 * cHZ_MIN_CH1 / cHZ_MAX_CH1) * (q31_t)(FIX_16 * cHZ_MAX_CH1 * cMAINLOOP_PRD)) << 1;

	/* Current con parameter */
#if defined(__CURRENT_DKI_RANGE)
	Motor_ch1.para.crt.dki = (q31_t)(FIX_12 * cID_KI_CH1 * cCTRL_PRD_CH1 * cA_MAX_CH1 / cV_MAX_CH1);
#else
	Motor_ch1.para.crt.dki = (q31_t)(FIX_15 * cID_KI_CH1 * cCTRL_PRD_CH1 * cA_MAX_CH1 / cV_MAX_CH1);
#endif	
#if defined(__CURRENT_DKP_RANGE)
	Motor_ch1.para.crt.dkp = (q31_t)(FIX_12 * cID_KP_CH1 * cA_MAX_CH1 / cV_MAX_CH1);
#else
	Motor_ch1.para.crt.dkp = (q31_t)(FIX_15 * cID_KP_CH1 * cA_MAX_CH1 / cV_MAX_CH1);
#endif
#if defined(__CURRENT_QKI_RANGE)
	Motor_ch1.para.crt.qki = (q31_t)(FIX_12 * cIQ_KI_CH1 * cCTRL_PRD_CH1 * cA_MAX_CH1 / cV_MAX_CH1);
#else
	Motor_ch1.para.crt.qki = (q31_t)(FIX_15 * cIQ_KI_CH1 * cCTRL_PRD_CH1 * cA_MAX_CH1 / cV_MAX_CH1);
#endif
#if defined(__CURRENT_QKP_RANGE)
	Motor_ch1.para.crt.qkp = (q31_t)(FIX_12 * cIQ_KP_CH1 * cA_MAX_CH1 / cV_MAX_CH1);
#else
	Motor_ch1.para.crt.qkp = (q31_t)(FIX_15 * cIQ_KP_CH1 * cA_MAX_CH1 / cV_MAX_CH1);
#endif	

	/* Position est parameter */
#if defined(__POSITION_KI_RANGE)
	Motor_ch1.para.pos.ki = (q31_t)(FIX_12 * cPOSITION_KI_CH1 * cCTRL_PRD_CH1 * cV_MAX_CH1 / cHZ_MAX_CH1);
#else
	Motor_ch1.para.pos.ki = (q31_t)(FIX_15 * cPOSITION_KI_CH1 * cCTRL_PRD_CH1 * cV_MAX_CH1 / cHZ_MAX_CH1);
#endif

#if defined(__POSITION_KP_RANGE)
	Motor_ch1.para.pos.kp = (q31_t)(FIX_12 * cPOSITION_KP_CH1 * cV_MAX_CH1 / cHZ_MAX_CH1);
#else
	Motor_ch1.para.pos.kp = (q31_t)(FIX_15 * cPOSITION_KP_CH1 * cV_MAX_CH1 / cHZ_MAX_CH1);
#endif
	Motor_ch1.para.pos.ctrlprd = (uint32_t)(FIX_16 * cHZ_MAX_CH1 * cCTRL_PRD_CH1);

	/* Speed con parameter */
#if defined(__SPEED_KI_RANGE)
	Motor_ch1.para.spd.ki = (q31_t)(FIX_12 * cSPEED_KI_CH1 * cSPD_CTRL_PRD_CH1 * cHZ_MAX_CH1 / cA_MAX_CH1);
#else
	Motor_ch1.para.spd.ki = (q31_t)(FIX_15 * cSPEED_KI_CH1 * cSPD_CTRL_PRD_CH1 * cHZ_MAX_CH1 / cA_MAX_CH1);
#endif

#if defined(__SPEED_KP_RANGE)
	Motor_ch1.para.spd.kp = (q31_t)(FIX_12 * cSPEED_KP_CH1 * cHZ_MAX_CH1 / cA_MAX_CH1);
#else
	Motor_ch1.para.spd.kp = (q31_t)(FIX_15 * cSPEED_KP_CH1 * cHZ_MAX_CH1 / cA_MAX_CH1);
#endif

	/* current limit */
	Motor_ch1.para.iq_lim = (q31_t)(FIX_31 * cIQ_LIM_CH1 / cA_MAX_CH1);
	Motor_ch1.para.id_lim = (q31_t)(FIX_31 * cID_LIM_CH1 / cA_MAX_CH1);
	Motor_ch1.para.err_ovc = (q15_t)(FIX_15 * cOVC_CH1 / cA_MAX_CH1);

	/* Field-weaking parameter */
#if defined(__USE_FIELD_WEAKING)
	{
		FldWk_InitTypeDef	fldwk;
		Spdlim_InitTypeDef	spdlim;

		fldwk.factor = (q15_t)(FIX_15 * cFLDWK_MODU_FACTOR_CH1 / 1);
		fldwk.id_min = (q31_t)(FIX_31 * cFLDWK_ID_MIN_CH1 / cA_MAX_CH1);
		fldwk.id_max = (q31_t)(FIX_31 * cFLDWK_ID_MAX_CH1 / cA_MAX_CH1);
		fldwk.fld_ki = (q31_t)(FIX_15 * cFLDWK_GAIN_KI_CH1 * cV_MAX_CH1 * cMAINLOOP_PRD / cA_MAX_CH1);

		spdlim.factor = (q15_t)(FIX_15 * cSPDLIM_MODU_FACTOR_CH1 / 1);
		spdlim.id_switch = (q15_t)(FIX_15 * cSPDLIM_SWITCH_VALUE_CH1 / cA_MAX_CH1);
		spdlim.spd_ki = (q31_t)(FIX_15 * cSPDLIM_GAIN_KI_CH1 * cV_MAX_CH1 * cMAINLOOP_PRD / cHZ_MAX_CH1);

		init_fldwk(&Motor_ch1.Fldwk, &fldwk);
		init_spdlim(&Motor_ch1.Spdlim, &spdlim);
	}
#endif /* defined(__USE_FIELD_WEAKING) */

	/* decoupling control parameter */
#if defined(__USE_DECOUPLING)
	{
		Decouple_InitTypeDef	decouple;

		decouple.Ke = (q15_t)(FIX_15 * cDECOUPLE_KE_CH1 * cHZ_MAX_CH1 / cV_MAX_CH1);

		init_decouple(&Motor_ch1.Decouple, &decouple);
	}
#endif /* defined(__USE_DECOUPLING) */

	/* deadtime compensation parameter  */
#if defined(__USE_DEADTIME)
	{
		Deadtime_InitTypeDef	deadtime;
		float32_t				work;

		deadtime.filter = cDEADTIME_FILTER_CH1;
		deadtime.current = (q15_t)(FIX_15 * cDEADTIME_CUR_MAX_CH1 / cA_MAX_CH1);

		work = cDEADTIME_DTIME_CH1 + cDEADTIME_FET_RT_CH1 - cDEADTIME_FET_FT_CH1;
		if (work < 0)
		{
			work = 0;
		}
#if defined(__A_PMD)
		deadtime.max_value = (q15_t)(FIX_15 * work / cPWMPRD_CH1);
#else
		deadtime.max_value = PWMPeriod(work);
#endif /* defined(__A_PMD) */

		init_deadtime(&Motor_ch1.Deadtime, &deadtime);
	}
#endif /* defined(__USE_DEADTIME) */

}


/**
 *********************************************************************************************
 * @brief		Set MotorControl data for user.
 *
 * @return		none
 *********************************************************************************************
 */
void	B_User_MotorControl(void)
{
	vector_t* motor_ch;
	const ipdrv_t* ipdrv;

	/***** user data setting for ch1 *****/
	motor_ch = &Motor_ch1;
	ipdrv = &Motor_ch1_IP;

	/* Get EMG state */
	if (PMD_GetEMG_Status(ipdrv) == cNormal)
	{
		motor_ch->drv.state.flg.emg_H = CLEAR;
	}
	else
	{
		motor_ch->drv.state.flg.emg_H = SET;
	}

	/* Set Speed */
#if (__TGTSPD_UNIT == 0)
	/* Hz of Electrical angle */
	motor_ch->usr.omega_user.word = (q31_t)((q15_t)(target_spd1 * FIX_15 / cHZ_MAX_CH1) << 16);
#elif (__TGTSPD_UNIT == 1)
	/* Hz of mechanical angle */
	motor_ch->usr.omega_user.word = (q31_t)((q15_t)((target_spd1 * (cPOLE_CH1 / 2) * FIX_15) / cHZ_MAX_CH1) << 16);
#elif (__TGTSPD_UNIT == 2)
	/* RPM of mechanical angle */
	motor_ch->usr.omega_user.word = (q31_t)((q15_t)(((target_spd1 * (cPOLE_CH1 / 2) * FIX_15) / 60) / cHZ_MAX_CH1) << 16);
#endif


	/* Motor ON/OFF */
	if (target_spd1 != 0)
	{
		motor_ch->usr.com_user.onoff = SET;
	}
	else
	{
		motor_ch->usr.com_user.onoff = CLEAR;
		motor_ch->drv.state.all = 0;									/* Clear error status. */

		if (PMD_GetEMG_Status(ipdrv) == cEMGProtected)
		{
			PMD_ReleaseEMG_Protection(ipdrv);
		}
	}

}

/*********************************** END OF FILE ******************************/
