/**
 ****************************************************************************
 * @file     interrupt.c
 * @brief    Interrupt Functions for VE
 * @version  V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2021
 *****************************************************************************
 */
#include <stdlib.h>

#include "ipdefine.h"
#include "mcuip_drv.h"
#include "D_Driver.h"
#include "usercon.h"
#include "interrupt.h"

/* for debug */
#if	defined(__MOTOR_DBGOUT_IDET)

	#define	DEBUG_IDET_ERROR(x)		debug_idet_error_ve(x)
	void	debug_idet_error_ve(TSB_VE_TypeDef* const VEx)
	{
		if ((VEx->MCTLF & 0x0010) != 0)
		{
			TP_IDET_HIGH;
		}
		else
		{
			TP_IDET_LOW;
		}
	}

#else
	#define	DEBUG_IDET_ERROR(x)
#endif	/* defined(__MOTOR_DBGOUT_IDET) */


/**
 *********************************************************************************************
 * @brief		INT_interval_4kHz
 *
 * @return		none
 *********************************************************************************************
 */
void INT_interval_4kHz(void)
{
	M_Main_Counter++;									/* Count for main routine Period */

}

/**
 ********************************************************************************************
 * @brief		Vector Control Interruption by VE for ch1.
 *
 * @param[in]	vector_t* const			_motor	.
 * @param[in]	const ipdrv_t* const 	_ipdrv	.
 *
 * @return		none
 ********************************************************************************************
 */
void INT_VectorControlByVE_CH1(vector_t* const _motor, const ipdrv_t* const _ipdrv)
{
	DEBUG_IDET_ERROR(_ipdrv->VEx);

#if (__FIXED_VDC_CH1 == 1)								/* 1: Vdc from constant data */
	/***** Set Vdc data to VE *****/
	VE_SetVDCreg(_ipdrv, (q15_t)(FIX_15 * cVDC_CH1 / cV_MAX_CH1));
#endif

	/***** Get data from VE register *****/
	VE_GetdataFromVEreg(_ipdrv, _motor);

	/*******/
	_motor->drv.Vd_ave.word = calc_ave(_motor->drv.Vd_ave.word , _motor->drv.Vd.word , cVdVq_Average_shift_cnt_CH1);
	_motor->drv.Vq_ave.word = calc_ave(_motor->drv.Vq_ave.word , _motor->drv.Vq.word , cVdVq_Average_shift_cnt_CH1);

	/***** Check over current *****/
	VE_GetPhaseCurrent(_ipdrv, &_motor->drv.Ia, &_motor->drv.Ib, &_motor->drv.Ic);
	if (D_Check_OverCurrent(_motor->para.err_ovc, _motor->drv.Ia, _motor->drv.Ib, _motor->drv.Ic))
	{
		_motor->drv.state.flg.emg_S = SET;
		_motor->stage.itr = ciEmergency;
	}

	/***** Check error voltage *****/
	if ((_motor->drv.Vdc < (q15_t)(FIX_15 * cVDC_MINLIM_CH1 / cV_MAX_CH1)) || ((q15_t)(FIX_15 * cVDC_MAXLIM_CH1 / cV_MAX_CH1) < _motor->drv.Vdc))
	{
		_motor->drv.state.flg.emg_DC = SET;
		_motor->stage.itr = ciEmergency;
	}

	if (_motor->drv.state.all == 0)						/* No error? */
	{
		/***** Detect Rotor Position *****/
		D_Detect_Rotor_Position(_motor);

		/***** Control Speed *****/
		_motor->drv.spdprd++;
		if (_motor->drv.spdprd >= cSPD_PI_PRD_CH1)
		{
			_motor->drv.spdprd = 0;
			D_Control_Speed(_motor);
		}
	}

	/***** Check Current detection error  *****/
#if (cSHUNT_TYPE_CH1 == 1)
	if ( (VE_GetOutputMode(_ipdrv) == OCRMD_OUT_ON)
	   && (D_Check_DetectCurrentError_1shunt(  VE_GetPWM_DutyU(_ipdrv),
												VE_GetPWM_DutyV(_ipdrv),
												VE_GetPWM_DutyW(_ipdrv),
												_motor->drv.chkpls,
												VE_GetShiftPWMState(_ipdrv),
												VE_GetPWM_Modulation(_ipdrv),
												VE_GetSector(_ipdrv),
												VE_GetPWM_Period(_ipdrv)) != 0)
	)
	{
		_motor->drv.idetect_error = 1;
	}
	else
	{
		_motor->drv.idetect_error = 0;
	}

#elif (cSHUNT_TYPE_CH1 == 3)
	if (  (VE_GetOutputMode(_ipdrv) == OCRMD_OUT_ON)
	   && (D_Check_DetectCurrentError_3shunt( VE_GetPWM_DutyMed(_ipdrv), _motor->drv.chkpls) != 0)
	)
	{
		_motor->drv.idetect_error = 1;
	}
	else
	{
		_motor->drv.idetect_error = 0;
	}
#endif /* cSHUNT_TYPE_CH1 */

	/**** Set data to VE register *****/
	/* Set Shift pwm change speed. for 1shunt */
	VE_SetSpwmMode(_ipdrv, _motor->drv.command.spwm);

	/* Set modulation type */
	VE_SetModulType(_ipdrv, _motor->drv.command.modul);

	switch (_motor->stage.itr)
	{
			/*--- Stop Stage -----------------------------*/
		case ciStop:
			VE_SetdataToVEreg_Stop(_ipdrv, _motor);
			break;

			/*--- Bootstrap Stage ------------------------*/
		case ciBootstrap:
			VE_SetdataToVEreg_Bootstrap(_ipdrv, _motor);
			break;

			/*--- Position Stage (Current type) ----------*/
		case ciInitposition_i:
			VE_SetdataToVEreg_Initposition_i(_ipdrv, _motor);
			break;

			/*--- Position Stage (Voltage type) ----------*/
		case ciInitposition_v:
			VE_SetdataToVEreg_Initposition_v(_ipdrv, _motor);
			break;

			/*--- Force Stage (Current type) -------------*/
		case ciForce_i:
			VE_SetdataToVEreg_Force_i(_ipdrv, _motor);
			break;

			/*--- Force Stage (Voltage type) -------------*/
		case ciForce_v:
			VE_SetdataToVEreg_Force_v(_ipdrv, _motor);
			break;

			/*--- ChangeUp Stage -------------------------*/
		case ciChange_up:
			VE_SetdataToVEreg_Change_up(_ipdrv, _motor);
			break;

			/*--- Steady_A Stage ---------------------------*/
		case ciSteady_A:
			VE_SetdataToVEreg_Steady_A(_ipdrv, _motor);
			break;

			/*--- Emergency Stage ------------------------*/
		case ciEmergency:
			VE_SetdataToVEreg_Emergency(_ipdrv, _motor);
			break;
			/*--- Fail-safe ------------------------*/
		default:
			VE_SetdataToVEreg_Stop(_ipdrv, _motor);
			break;
	}

#if defined(__USE_DAC)
	UiOutDataStartByVE(1);									/* Dac output for debug */
#endif /* __USE_DAC */

#if (cSHUNT_TYPE_CH1 == 3)
	VE_Start(_ipdrv);
#endif /* cSHUNT_TYPE_CH1 == 3 */

}


/**
 *********************************************************************************************
 * @brief		INT_PWM1
 *
 * @return		none
 *********************************************************************************************
 */
void	INT_PWM1(const ipdrv_t* const _ipdrv)
{
#if (cSHUNT_TYPE_CH1 == 1)

	/* VE Start/ */
	VE_Start(_ipdrv);

#endif /* cSHUNT_TYPE_CH1 == 1 */
}


/*********************************** END OF FILE ******************************/
