// *************** USER FUNCTIONS ***********************
// boolean cmdDetect()                    // Sends command to detect USB-I2C controller // return true: USB-I2C device detected
// void cmdSetId(int slaveId)            // Sends command to set I2C SlaveID // slaveID:I2C_slave_ID
// void cmdWriteByte(int add, int data) // Sends command to write one byte via I2C // add:register_address, data:data_to_write
// byte cmdReadByte(int add)             // Sends command to read one byte via I2C // add:register_address // returns:read_data
// USB-I2C Converter Flags (updated from in serialControl tab)
//
////Status - from USB-I2C Converter (stores the current status of Converter and I2C communication)
// int statusConverter = 0; //0:Normal_Operation(Correct_Response) 1:No_Response_From_USB-I2C_Converter
// int statusI2C = 0;    //0:success, 1:tooLongData, 2:NACKonAddress, 3:NACKonData, 4:other, 5:timeout,

////////////////////////////////////////////////////////////////////////////////
import processing.serial.*; //used for serial communication

// Serial Port
Serial serialPort;                  //stores current serail port
int serialBaudrate = serialBaudrateValue;          //baudrate setting

////Command response - from USB-I2C Converter (used in cmd*** functions, which communicate with converter)
byte[] cmdResponse;         //stores cmdResponse received after sending command
boolean cmdResponseFlag = false; //true:cmdResponse received

//************************** Command Functions ******************************
boolean cmdDetect(){ // Sends command to detect USB-I2C controller
  boolean detected = false; // true: USB-I2C device detected.
  //SetID Command Format// [0]:'d'(Command), [1]:noData , [2]:noData , [3]:'\n'(End_of_command)
  byte[] cmd = {'d','0','0','\n'};   //prepare command
  serialPort.clear();         //clears serial port data
  serialPort.write(cmd);      //send command
  for(int i=0; i<100; i++){
    if(cmdResponseFlag == true){    //response received
      cmdResponseFlag = false;      //reset flag
      statusConverter = 0; //0:Normal_Operation 1:No_Response_From_USB-I2C_Converter
      //Response Format// [0]:'#'(Start_of_response), [1]:'d', [2]:'\n'(End_of_response)
      char response = char(cmdResponse[1]); //Check received data// d:USB-I2C converter is detected
      if(response == 'd'){
        println("cmdDetect() (Success: USB-I2C converter detected)");
        detected = true;
      }
      break;
    }
    delay(5);
  }
  if(detected == false){
    println("cmdDetect() (Error: Wrong COM Port: Connected device is not USB-I2C converter)");
  }
  return detected;
}

void cmdSetId(int slaveId){ // Sends command to set I2C SlaveID // slaveID:I2C_slave_ID
  //SetID Command Format// [0]:'i'(Command), [1]:SlaveID, [2]:noData , [3]:'\n'(End_of_command)
  byte[] cmd = {'i', byte(slaveId),'0', '\n'};   //prepare command
  serialPort.clear();                  //clears serial port data
  serialPort.write(cmd);               //send command
  for(int i=0; i<40; i++){
    if(cmdResponseFlag == true){    //response received
      cmdResponseFlag = false;      //reset flag
      statusConverter = 0; //0:Normal_Operation 1:No_Response_From_USB-I2C_Converter
      //Response Format// [0]:'#'(Start_of_response), [1]:'i', [2]'\n'(End_of_response)
      char response = char(cmdResponse[1]);
      if(response=='i'){    // response i indicates that ID is set successfully
        println("cmdSetId() I2CslaveId:0x" + hex(slaveId,2) + " (Success)");
      }else{
        println("cmdSetId() I2CslaveId:0x" + hex(slaveId,2) + " (Error)");
        displayMessageBoxError("ERROR","Error in setting slave ID");
      }
      break;
    }else if(i==39){
      println("cmdSetId() (Error:No_Response_from_Converter)");
      statusConverter = 1; //0:Normal_Operation 1:No_Response_From_USB-I2C_Converter
      statusI2C = 6;    //0:success, 1:tooLongData, 2:NACKonAddress, 3:NACKonData, 4:other, 5:timeout, 6:unknown(when statusConver is 1)
      displayMessageBoxError("Error : No Response","No response from USB-I2C converter");
    }
    delay(5);
  }
}

void cmdWriteByte(int add, int data){ // Sends command to write one byte via I2C // add:register_address, data:data_to_write
  //Write Command Format// [0]:'w'(Command), [1]:registerAddress, [2]:registerData, [3]:'\n'(End_of_command)
  byte[] cmd = {'w', byte(add), byte(data), '\n'};   //prepare command
  //println(cmd);
  serialPort.clear();                    //clears serial port data
  serialPort.write(cmd);                 //send command
  for(int i=0; i<40; i++){
    if(cmdResponseFlag == true){      //response received
      cmdResponseFlag = false;        //reset flag
      statusConverter = 0; //0:Normal_Operation 1:No_Response_From_USB-I2C_Converter
      //Response Format// [0]:'#'(Start_of_response), [1]:I2C_communication_result, [2]:'\n'(End_of_response)
      statusI2C = int(cmdResponse[1]); //Update I2C status// 0:success, 1:tooLongData, 2:NACKonAddress, 3:NACKonData, 4:other, 5:timeout
      if(statusI2C == 0){
        println("cmdWriteByte() add:"+add+"(0x" + hex(add,2) + ") data:0x" + hex(data,2) + " statusI2C:" + statusI2C +  "(Success)");
      }else{
        println("cmdWriteByte() add:"+add+"(0x" + hex(add,2) + ") data:0x" + hex(data,2) +  " statusI2C:" + statusI2C + "(Error:I2C_Failure)");
        displayMessageBoxError("Error: I2C ("+statusI2C+")","I2C Failure. Check I2C connection."); 
      }
      break;
    }else if(i==39){
      println("cmdWriteByte() add:"+add+"(0x" + hex(add,2) + ") data:0x" + hex(data,2) + " (Error:No_Response_from_Converter)");
      statusConverter = 1; //0:Normal_Operation 1:No_Response_From_USB-I2C_Converter
      statusI2C = 6;    //0:success, 1:tooLongData, 2:NACKonAddress, 3:NACKonData, 4:other, 5:timeout, 6:unknown(when statusConver is 1)
      displayMessageBoxError("Error : No Response","No response from USB-I2C converter");
    }
    delay(5);
  }
}

int cmdReadByte(int add){ // Sends command to read one byte via I2C // add:register_address (int or hex) // returns:read_data
  serialPort.clear();                //clears serial port data
  int readData=0;
  //Write Command Format// [0]:'r'(Command), [1]:registerAddress, [2]:noData , [3]:'\n'(End_of_command)
  byte[] cmd = {'r', byte(add), '0', '\n'};     //prepare command
  //println(cmd);
  serialPort.write(cmd);             //send command
  for(int i=0; i<40; i++){
    if(cmdResponseFlag == true){  //response received
      cmdResponseFlag = false;    //reset flag
      statusConverter = 0; //0:Normal_Operation 1:No_Response_From_USB-I2C_Converter
      //Response Format// [0]:'#'(Start_of_response), [1]:I2C_communication_result, [2]:read_data, [3]'\n'(End_of_response)
      statusI2C = int(cmdResponse[1]); //Update I2C status// 0:success, 1:tooLongData, 2:NACKonAddress, 3:NACKonData, 4:other, 5:timeout
      if(statusI2C == 0){
        readData = int(cmdResponse[2]);  //get read register data
        if(displyI2CMessages==1){ //mainly used to hide successful I2C read messages occuring because of poling.
          println("cmdReadByte() add:"+add+"(0x" + hex(add,2) + ") readData:0x" + hex(readData,2) + " statusI2C:" + statusI2C + "(Success)");
        }
      }else{
        readData = 0;  //setting 0 because couldn't read I2C register
        println("cmdReadByte() add:"+add+"(0x" + hex(add,2) + ") readData:0x" + hex(readData,2) + " statusI2C:" + statusI2C + "(Error:I2C_Failure)");
        displayMessageBoxError("Error: I2C ("+statusI2C+")","I2C Failure. Check I2C connection."); 
      }
      break;
    }else if(i==39){
      println("cmdReadByte() add:"+add+"(0x" + hex(add,2) + ") readData:0x" + hex(readData,2) + " (Error:No_Response_from_Converter)");
      statusConverter = 1; //0:Normal_Operation 1:No_Response_From_USB-I2C_Converter
      statusI2C = 6;    //0:success, 1:tooLongData, 2:NACKonAddress, 3:NACKonData, 4:other, 5:timeout, 6:unknown(when statusConver is 1)
      displayMessageBoxError("Error : No Response","No response from USB-I2C converter");
    }
    delay(5);
  }
  return readData;
}



//************************************ Serial Port **********************************************
void serialPortInitialization(){
  boolean portSelected = false;    
  boolean directConnection = true; // if only one COM is available, this is used to perform direct connection only once
  String portName="";
  int portsAvailable = Serial.list().length;
  while(portSelected==false){
      if(portsAvailable==0){
        println("ERROR - No COM PORT available");
        while(Serial.list().length==0){
          displayMessageBoxError("ERROR", "No COM port available. Connect device."); // String title, String label
        }
        portsAvailable = Serial.list().length;
      }else if(portsAvailable==1 && directConnection==true){
        try{
          portName=Serial.list()[0];
          serialPort=new Serial(this,portName,serialBaudrate);
          portSelected=true;
        }catch(Exception e){
          portSelected=false;
          println("Error: Can't connect to available COM Port: "+portName);
          displayMessageBoxError("ERROR: COM Port Busy","Available COM Port:"+portName+" is busy");
          directConnection =false;
        }
      }else{
        println("Multiple ports available.");
        printArray(Serial.list());
        portName = displayMessageBoxSelection("Selct COM Port","Select COM Port of USB-I2C converter",Serial.list()); // title, label, choices
        if(portName!=null){
          try{
            serialPort=new Serial(this,portName,serialBaudrate);
            portSelected=true;
          }catch(Exception e){
            portSelected=false;
            println("Error: Can't connect to selected COM Port: "+portName);
            displayMessageBoxError("ERROR: COM Port","Cannot connect to COM Port:"+portName);
          }
        }else{
          displayMessageBoxError("ERROR", "Select a COM port."); // String title, String label
        }
      }
      if(portSelected == true){
        println("Connected to : "+portName);
        serialPort.clear();
        serialPort.bufferUntil('\n'); // Buffers serial port input until '\n'
        displayMessageBoxInformation("COM Port Connected", "Connected to "+portName); // String title, String label
        delay(700);  //Arduino Reset after COM port connection. This is time for arduino to startup. Note: Arduino Nano takes more time to startup than Arduino Nano EVERY. So this time is set with Arduino Nano.
        //CHECK IF CONNECTED COM PORT belongs to USB-I2C converter
        boolean detected = cmdDetect(); // Sends command to detect USB-I2C controller
        textComPort=portName;
        if(detected==false){
          displayMessageBoxError("ERROR: WRONG DEVICE","Connected device is not USB-I2C converter.\nCheck COM Port and try again.");
          directConnection=false; // user have to select com port even if only 1 com port is available
          portSelected=false;
          serialPort.stop();
        }
      }
    }
}

void serialEvent(Serial p){
   byte[] inputData=p.readBytes();
   String inputDataStr = new String(inputData);
   if(char(inputData[0])=='#'){        //Indicates command response
     //println("cmd_response: "+inputData);
     cmdResponse=inputData;
     cmdResponseFlag=true;
   }else if(char(inputData[0])=='!'){  //Indicates INVALID command response
     println("USB-I2C_response: INVALID COMMAND CODE");
     println(inputDataStr);
     displayMessageBoxError("Error: Invalid commad code","USB-I2C converter received Invalid Command Code");
   }else if(inputDataStr.length()>1){                              
     println("USB-I2C_message: " + inputDataStr);    //Message from USB-I2C converter
     println(inputDataStr.length());
   }
  serialPort.clear();                //clears serial port data
}
