// *************** USER FUNCTIONS ***********************
// ICData ic = new ICData();        // Define ICData class object
//
// int  getReg(int index)                 //returns registerData
// void setReg(int index, int data)       //set the registerData
// int  getVar(ICVariable vari)           //returns variableData
// void setVar(ICVariable vari, int data) //set the variableData in corresponding register
//
// void printRegAll()               //Print all registers
// void printReg(int index)         //Print a single register
// void printVar(ICVariable vari)   //Print a single variable


/////////////////////////////////////////////////////////////////////////
//NVM processing is not handled by this class
class ICData{ 
  //Registers of TC78B011FTG IC
  int[] register = new int[31]; // used to store register values corresponding to TC78B011FTG

  //Variable Information of TC78B011FTG IC
  //Register 0
  ICVariable cp_low     = new ICVariable("cp_low",     0,  5,  5 ); //Variable Info // NAME, Register, MSB, LSB (-ve means have bits in next register)
  ICVariable tsd        = new ICVariable("tsd",        0,  4,  4 );
  ICVariable isd        = new ICVariable("isd",        0,  3,  3 );
  ICVariable ov_spd     = new ICVariable("ov_spd",     0,  2,  2 );
  ICVariable ud_spd     = new ICVariable("ud_spd",     0,  1,  1 );
  ICVariable st_fail    = new ICVariable("st_fail",    0,  0,  0 );
  //Register 1
  ICVariable userid     = new ICVariable("userid",     1,  7,  0 );
  //Register 2
  ICVariable nostop     = new ICVariable("nostop",     2,  7,  7 );
  ICVariable stopduty   = new ICVariable("stopduty",   2,  6,  0 );
  //Register 3
  ICVariable startduty  = new ICVariable("startduty",  3,  7,  0 );
  //Register 4
  ICVariable changeduty = new ICVariable("changeduty", 4,  7,  0 ); //Variable Info // NAME, Register, MSB, LSB (-ve means have bits in next register)
  //Register 5
  ICVariable maxduty    = new ICVariable("maxduty",    5,  7,  0 );
  //Register 6
  ICVariable startrpm   = new ICVariable("startrpm",   6,  7, -4 );
  //Register 7
  ICVariable maxdutyhys = new ICVariable("maxdutyhys", 7,  3,  0 );
  //Register 8
  ICVariable speedslop  = new ICVariable("speedslop",  8,  7, -6 );
  //Register 9
  ICVariable maxopen    = new ICVariable("maxopen",    9,  1,  1 );
  ICVariable maxoff     = new ICVariable("maxoff",     9,  0,  0 );
  //Register 10
  ICVariable speedslop2 = new ICVariable("speedslop2", 10, 7, -6 ); //Variable Info // NAME, Register, MSB, LSB (-ve means have bits in next register)
  //Register 11
  ICVariable vcp_mask   = new ICVariable("vcp_mask",   11, 1,  1 );
  ICVariable openloop   = new ICVariable("openloop",   11, 0,  0 );
  //Register 12
  ICVariable kix        = new ICVariable("kix",        12, 7,  7 );
  ICVariable ki         = new ICVariable("ki",         12, 6,  0 );
  //Register 13
  ICVariable kpx        = new ICVariable("kpx",        13, 7,  7 );
  ICVariable kp         = new ICVariable("kp",         13, 6,  0 );
  //Register 14
  ICVariable stby_mode  = new ICVariable("stby_mode",  14, 7,  7 );
  ICVariable dir        = new ICVariable("dir",        14, 6,  6 );
  ICVariable polepair   = new ICVariable("polepair",   14, 5,  3 );
  ICVariable maxspeed   = new ICVariable("maxspeed",   14, 2,  1 );
  ICVariable fg_on      = new ICVariable("fg_on",      14, 0,  0 );
  //Register 15
  ICVariable fgsel      = new ICVariable("fgsel",      15, 7,  5 );  //Variable Info // NAME, Register, MSB, LSB (-ve means have bits in next register)
  ICVariable tspsel     = new ICVariable("tspsel",     15, 4,  4 );
  ICVariable spdinv     = new ICVariable("spdinv",     15, 3,  3 );
  ICVariable latch      = new ICVariable("latch",      15, 2,  2 );
  ICVariable ocpmask    = new ICVariable("ocpmask",    15, 1,  0 );
  //Register 16
  ICVariable lockdis         = new ICVariable("lockdis",         16, 7, 7 );
  ICVariable dutychglimit    = new ICVariable("dutychglimit",    16, 6, 4 );
  ICVariable startcurrent    = new ICVariable("startcurrent",    16, 3, 1 );
  ICVariable ocpdis          = new ICVariable("ocpdis",          16, 0, 0 );
  //Register 17
  ICVariable ss_add_sel      = new ICVariable("ss_add_sel",      17, 7, 6 );
  ICVariable ss_up_sel       = new ICVariable("ss_up_sel",       17, 5, 4 );
  ICVariable ss_dutychglimit = new ICVariable("ss_dutychglimit", 17, 3, 1 );
  ICVariable duty_up_time    = new ICVariable("duty_up_time",    17, 0, 0 );
  //Register 18
  ICVariable rpmlimit        = new ICVariable("rpmlimit",        18, 7, 5 );
  ICVariable brk_inv         = new ICVariable("brk_inv",         18, 4, 4 );
  ICVariable isd_mask        = new ICVariable("isd_mask",        18, 3, 3 );
  ICVariable rs_sel          = new ICVariable("rs_sel",          18, 2, 1 );
  ICVariable antithrough     = new ICVariable("antithrough",     18, 0, 0 );
  //Register 19
  ICVariable wait_time       = new ICVariable("wait_time",       19, 7, 5 );  //Variable Info // NAME, Register, MSB, LSB (-ve means have bits in next register)
  ICVariable wait_mode       = new ICVariable("wait_mode",       19, 4, 4 );
  ICVariable wait_con        = new ICVariable("wait_con",        19, 3, 3 );
  ICVariable lock_brk        = new ICVariable("lock_brk",        19, 2, 2 );
  ICVariable alertinv        = new ICVariable("alertinv",        19, 1, 1 );
  ICVariable tsd_mask        = new ICVariable("tsd_mask",        19, 0, 0 );
  //Register 20
  ICVariable tre        = new ICVariable("tre",        20, 7, 5 );
  ICVariable pretip     = new ICVariable("pretip",     20, 4, 3 );
  ICVariable tip        = new ICVariable("tip",        20, 2, 0 );
  //Register 21
  ICVariable la         = new ICVariable("la",         21, 7, 4 );
  ICVariable fmax       = new ICVariable("fmax",       21, 3, 2 );
  ICVariable fst        = new ICVariable("fst",        21, 1, 0 );
  //Register 22
  ICVariable fpwm       = new ICVariable("fpwm",       22, 4, 2 );  //Variable Info // NAME, Register, MSB, LSB (-ve means have bits in next register)
  ICVariable deadtime   = new ICVariable("deadtime",   22, 1, 0 );
  //Register 23
  ICVariable isd_lvl    = new ICVariable("isd_lvl",    23, 7, 7 );
  ICVariable ocp_lvl    = new ICVariable("ocp_lvl",    23, 6, 6 );
  ICVariable source     = new ICVariable("source",     23, 5, 3 );
  ICVariable sink       = new ICVariable("sink",       23, 2, 0 );
  //Register 24
  ICVariable comp_hys   = new ICVariable("comp_hys",   24, 7, 6 );
  //Register 25
  ICVariable slave_adrs = new ICVariable("slave_adrs", 25, 7, 1 );
  //Register 27 & 28
  ICVariable spd        = new ICVariable("spd",        27, 7, -2 );  //Variable Info // NAME, Register, MSB, LSB (-ve means have bits in next register)
  //Register 29 & 30
  ICVariable hz_cnt     = new ICVariable("hz_cnt",     29, 7, -8 );


  ICData(){ //Constructor of this class
    register[25] = 41; //Default value of register 25 (Slave_address) is 0x29 (41 in dec)
  }

  //Functions for general use
  int getReg(int index){ //returns registerData
    return register[index];
  }
  
  void setReg(int index, int data){ //set the registerData
    if(data < 256){ //check if input data is valid (fit in register)
      //Setting register data
      register[index] = data;
    } else {
      println("setReg(" + index + ", " + data + ") ERROR: input data larger than register size, Max limit: 255");
    }
  }
  
  int getVar(ICVariable vari){ //returns variableData
    //reading corresponding register data
    if(vari.LSB >= 0){  //this variable is stored in single register
      int data = 0;
      data = register[vari.REGISTER] & bitMask(vari.MSB, vari.LSB); //getting variable data from corresponding register
      data = data >> vari.LSB;  //shifting to get correct value
      return data;
    } else {          //this variable is stored in two registers
      int dataH, dataL;       // upper & lower portion of data
      int msbL = 7;           // msb of lower portion of data is always 7;
      int lsbL = 8 + vari.LSB; // lsb of lorwer portion of data
      dataH = register[vari.REGISTER] & bitMask(vari.MSB, 0);  //higher portion of data 
      dataL = register[vari.REGISTER + 1] & bitMask(msbL, lsbL);  //extracting lower portion of data
      dataL = dataL >> lsbL;  //shifting to correct position
      int varData = (dataH << (-1 * vari.LSB)) + dataL; //combining both portions of data
      return varData;      
    }
  }
  
  void setVar(ICVariable vari, int data){ //set the variableData in corresponding register
    if(data < (int)Math.pow(2,(vari.MSB - vari.LSB + 1 ))){  // checking if input data is less than max variable size
      // Updating corresponding register data
      if(vari.LSB >= 0){ //variable data is in single register
        int temp = register[vari.REGISTER];
        temp = temp & (~bitMask(vari.MSB,vari.LSB));  //clearing the space of variable in register
        temp = temp | (data << vari.LSB); //setting the varaible data in register.
        register[vari.REGISTER] = temp;
      } else { //var.LSB<0 :variable data is in two registers.  //LSB shows the amount of bits in lower register
        int lowerBits = -1 * vari.LSB;
        //High Side Register
        int dataH = data >> lowerBits; //extracting portion of input data for High Register (by discarding lower register data)
        int tempH = register[vari.REGISTER]; //High side data
        tempH = tempH & (~bitMask(vari.MSB, 0));  //clearing the space of variable in register
        tempH = tempH | dataH; //setting the varaible data in register.
        register[vari.REGISTER] = tempH;
        //Low Side Register
        int dataL = data & bitMask((lowerBits - 1), 0); //extracting portion of input data for Lower Register
        int tempL = register[vari.REGISTER+1]; //Low side data is in next register
        int msbL = 7;            //MSB of this variable in Lower register is always 7
        int lsbL = 8 - lowerBits;  //LSB of this variable in Lower register
        tempL = tempL & (~bitMask(msbL, lsbL));  //clearing the space of variable in register
        tempL = tempL | (dataL << lsbL); //setting the varaible data in register.
        register[vari.REGISTER+1] = tempL;
      }
    } else {
      println("setVar(" + vari.NAME + ", " + data + ") ERROR: input data larger than variable size, Max limit: " + ((int)Math.pow(2,(vari.MSB - vari.LSB + 1))-1));
    }
  }
  
  String printRegAllInString(){ //Returns all registers in String
    String str=""; //String for storing result
    str+=" Reg. Hex (Bin)\n";
    for(int index=0; index<register.length; index++){
          str+=" "+String.format("%2d",index);
          str+="  0x" + hex(register[index],2);
          //str+="    dec: " + String.format("%4d",register[index]);
          str+=" (" + intToBin(register[index],8)+")\n";
    }
    return str;
  }
  
  void printRegAll(){ //Print all registers
    println("All REGISTERS:");
    for(int i=0; i<register.length; i++){
      printReg(i);
    }
  }
  
  void printReg(int index){ //Print a single register
    print("Register "+ String.format("%2d",index) + " >");
    print("  hex: " + hex(register[index],2));
    print("    dec: " + String.format("%4d",register[index]));
    println("    bin: " + intToBin(register[index],8));
  }
  
  String printVarAllInString(){ //Returns all Variables in String (in alphabetical order)
    String str=""; //String for storing result
      str+=" Parameter    Data (Reg,MSB,Len)\n";
      str+=printVarInString(alertinv);
      str+=printVarInString(antithrough);
      str+=printVarInString(brk_inv);
      str+=printVarInString(changeduty);
      str+=printVarInString(comp_hys);
      str+=printVarInString(cp_low);
      str+=printVarInString(deadtime);
      str+=printVarInString(dir);
      str+=printVarInString(duty_up_time);
      str+=printVarInString(dutychglimit);
      str+=printVarInString(fg_on);
      str+=printVarInString(fgsel);
      str+=printVarInString(fmax);
      str+=printVarInString(fpwm);
      str+=printVarInString(fst);
      str+=printVarInString(hz_cnt);
      str+=printVarInString(isd);
      str+=printVarInString(isd_lvl);
      str+=printVarInString(isd_mask);
      str+=printVarInString(ki);
      str+=printVarInString(kix);
      str+=printVarInString(kp);
      str+=printVarInString(kpx);
      str+=printVarInString(la);
      str+=printVarInString(latch);
      str+=printVarInString(lock_brk);
      str+=printVarInString(lockdis);
      str+=printVarInString(maxduty);
      str+=printVarInString(maxdutyhys);
      str+=printVarInString(maxoff);
      str+=printVarInString(maxopen);
      str+=printVarInString(maxspeed);
      str+=printVarInString(nostop);
      str+=printVarInString(ocp_lvl);
      str+=printVarInString(ocpdis);
      str+=printVarInString(ocpmask);
      str+=printVarInString(openloop);
      str+=printVarInString(ov_spd);
      str+=printVarInString(polepair);
      str+=printVarInString(pretip);
      str+=printVarInString(rpmlimit);
      str+=printVarInString(rs_sel);
      str+=printVarInString(sink);
      str+=printVarInString(slave_adrs);
      str+=printVarInString(source);
      str+=printVarInString(spd);
      str+=printVarInString(spdinv);
      str+=printVarInString(speedslop);
      str+=printVarInString(speedslop2);
      str+=printVarInString(ss_add_sel);
      str+=printVarInString(ss_dutychglimit);
      str+=printVarInString(ss_up_sel);
      str+=printVarInString(st_fail);
      str+=printVarInString(startcurrent);
      str+=printVarInString(startduty);
      str+=printVarInString(startrpm);
      str+=printVarInString(stby_mode);
      str+=printVarInString(stopduty);
      str+=printVarInString(tip);
      str+=printVarInString(tre);
      str+=printVarInString(tsd);
      str+=printVarInString(tsd_mask);
      str+=printVarInString(tspsel);
      str+=printVarInString(ud_spd);
      str+=printVarInString(userid);
      str+=printVarInString(vcp_mask);
      str+=printVarInString(wait_con);
      str+=printVarInString(wait_mode);
      str+=printVarInString(wait_time);
    return str;
  }
  
  String printVarInString(ICVariable vari){ //Returns variable data in string  
    String str=" ";
    str+=String.format("%-12s",vari.NAME);
    int dataDec=getVar(vari);
    str+=String.format("%5d",dataDec)+" ("+String.format("%2d",vari.REGISTER)+","+vari.MSB+","+(vari.MSB-vari.LSB+1)+")\n";
    return str;
  }
  void printVar(ICVariable vari){
    int dataDec = getVar(vari);
    String dataHex = (vari.LSB >= 0) ? ("  " + hex(dataDec, 2)) : (hex(dataDec, 4));  //negative LSB indicates varaible data is in two registers
    String dataBin = intToBin(dataDec, (vari.MSB - vari.LSB + 1));
    //String dataBin = (var.LSB >= 0) ? (intToBin(dataDec, 8)) : (intToBin(dataDec, 16));  //negative LSB indicates varaible data is in two registers
    print(String.format("%-12s",vari.NAME));
    print("> Register:" + String.format("%2d",vari.REGISTER) + " MSB:" + vari.MSB + " LSB:" + String.format("%2d", vari.LSB));
    print("  Data>   hex: " + dataHex);
    print("    dec: " + String.format("%4d", dataDec));
    print("    bin: " + dataBin);
    println();
  }

  int bitMask(int msb, int lsb){ //returns the mask value with 1 from msb to lsb
    int mask=0;
    for(int i=lsb; i<=msb; i++){
      mask = mask + (int)Math.pow(2,i);
    }
    return mask;
  }
}

//////////////////////////////////////////////////////////////////////
//NVM processing is not handled by this class
class ICVariable { //Variables of TC78B011FTG IC
  //Constant parameters of each variable
  final String NAME;   //Name of string
  final int REGISTER;  //Register of this variable
  final int MSB;       //MSB of this variable in above register
  final int LSB;       //LSB of this variable in above register (negative value means that variable is larger than one register and some)
  
  //Constructor for initializing parameters of variables
  ICVariable(String NAME, int REGISTER, int MSB, int LSB){  //if variable doesn't have remaining bits and default value is 0
    this.NAME     = NAME;
    this.REGISTER = REGISTER;
    this.MSB      = MSB;
    this.LSB      = LSB;
  }
}

/////////// GENERAL FUNCTIONS /////////////////////
String intToBin(int input, int numOfBits){ //returns the binary value string of an int
  String bin = Integer.toBinaryString(input); // Convert to binary string
  bin = String.format("%"+String.valueOf(numOfBits)+"s",bin ).replaceAll(" ","0");  // change length of binary String bin to numOfBits length by adding 0 to front
  StringBuffer sb = new StringBuffer();
  for (int i = 0; i< bin .length(); i++ ) { //creating new string with a nibble
    sb.append(bin.charAt(i));
    if ((bin.length()-i-1)%4==0)  
      sb.append(" ");
  }
  sb.setLength(sb.length() - 1); //removing space at the end
  return sb.toString();
}
