//Import Library
import controlP5.*;
import java.util.*;

// ************************************* Declarations *************************************** 
// ControlP5
ControlP5 cp5; //create ControlP5 object
Canvas tt;     //Canvas for drawing tooltip
ICData ic = new ICData();        // Define ICData class object to Manage Register and Variable Data of IC
PFont guiTabFont, guiBaseFont, guiBaseFontSmallCase, guiHeadingFont;
final boolean debug = true;   // true: enables extra messages in console window of processing.

//GlobalTextArea
Textarea textArea_registerList, textArea_variableList;

// ************************************* Constants ******************************************
//Serial
final int serialBaudrateValue = 9600; // Serial Baudrate (Should be same as baudrate of USB-I2C converter)
//GUI Font
final String mainFontStyle = "Arial";
final String headingFontStyle = "Arial Bold";
final int guiTabFontSize = 15; //Font size for TABs of GUI
final int guiBaseFontSize = 16; //Font size for GUI controllers other than TABs 
final int graphFontSize = 15;  // Base Font size for text of graph: label=this, axisTitle=this+3, graphTitle=this+5
//GUI Colors
final int guiBackgroundColor = color(230);  // color of GUI background other than TAB
final int tabAreaBackgroundColor = color(210);  // color of TAB are background
final int tabActiveColor = color(30,90,200);  //Active Tab color
final int controllerActiveColor = color(0,116,217); //Active color of various controllers
final int graphColor = color(30,90,220);  //Color of Speed profile lines in graph
final int labelColor = color(10);  //Dark grey
final int colorDarkGrey = color(10);
final int colorGreen = color(130,240,130);
final int colorRed   = color(255,140,140);
final int colorYellow= color(237,226,0);
final int colorBlueActive = tabActiveColor;
//GUI Components
final int tabHeight = 30;
final int MaxCloseLoopMotorRPM =20000;
//Controller Data   //data meaning of variable data equal to index
//tab default (Configuration Control)
final String[] sourceCurrentData = {"10.0 mA","13.9 mA","19.3 mA","26.8 mA","37.3 mA","51.8 mA","72.0 mA","100.0 mA"}; //SOURCE 23[5:3]
final String[] sinkCurrentData = {"20.0 mA","27.8 mA","38.6 mA","53.7 mA","74.6 mA","103.6 mA","143.9 mA","200.0 mA"}; //SINK 23[5:3]
final String[] dutyChangeLimitData = {"64/8","2/8","3/8","4/8","6/8","10/8","20/8","56/8"}; //DUTYCHGLIMIT 16[6:4]
final String[] dutyChangeTimeData = {"2.7 ms","10.8 ms"}; //DUTY_UP_TIME 17[0]
final String[] deadTimeData = {"250 ns","500 ns","1000 ns","1500 ns"}; //DEADTIME 22[1:0]
final String[] restartTimeData = {"0 s","0.5 s","1 s","1.5 s","2 s","4 s","7 s","10 s"}; //TRE 22[1:0]
//tab default (Configuration Control)
final String[] firstDcExcitationData = {"0 s","0.2 s","0.5 s","1.0 s"}; //PreTIP 20[4:3]
final String[] secondDcExcitationData = {"1 s","0.2 s","0.4 s","0.6 s","0.8 s","1.0 s","1.5 s","2.0 s"}; //TIP 20[2:0]
final String[] forcedFreqData = {"1.6 Hz","3.2 Hz","6.4 Hz","12.8 Hz"}; //FST 21[1:0]
final String[] compHysData = {"None","±100 mV","±200 mV","±300 mV"}; //COMP_HYS 24[7:6]
final String[] startCurrentData = {"100% Voc","87.5% Voc","75.0% Voc","62.5% Voc","50.0% Voc","37.5% Voc","25.0% Voc","12.5% Voc"}; //STARTCURRENT 16[3:1]
final String[] targetCurrentData = {"+0% Voc","+30% Voc","+40% Voc","+50% Voc"}; //SS_ADD_SEL 17[7:6]
final String[] currentIncrData = {"+1% Voc","+2% Voc","+5% Voc","+10% Voc"}; //SS_UP_SEL 17[5:4]
final String[] sStartDutyData = {"64/8","2/8","3/8","4/8","6/8","10/8","20/8","56/8"}; //SS_DUTYCHGLIMIT 17[3:1]
final String[] waitTimeData = {"0 s","1 s","2 s","3 s","4 s","5 s","6 s","7 s"}; //WAIT_TIME 19[7:5]

// ************************************* Global Variables ******************************************
// Text are just for displaying information on GUI without Tooltip. (Thus same as Labels but without Tooltips)
String textI2CRegAddress = ""; //stores the I2C Address of MCD set via Register mode
String textComPort = ""; //Stores the last selected COM port
int timer=0;

// ************************************* Flags **********************************************
// USB-I2C Converter Flags (updated in serialControl functions)
////Status - from USB-I2C Converter (stores the current status of Converter and I2C communication)
int statusConverter = 1; //0:Normal_Operation(Correct_Response) 1:No_Response_From_USB-I2C_Converter
int statusI2C = 6;    //0:success, 1:tooLongData, 2:NACKonAddress, 3:NACKonData, 4:other, 5:timeout, 6:unknown(when statusConver is 1)
int displyI2CMessages = 1; //1:display in console of processing  , 0:don't display //mainly used to hide successful I2C read messages occuring because of poling.

//Flag for displaying error incase of invalid setting 
boolean startStopErrorDisplayed= false; 
boolean slopeErrorDisplayed= false; 
boolean slope2ErrorDisplayed= false;

//Misc Flag
boolean sendSlopeRegDataFlag=false; //if ture: sendSlopeData() function runs once

// ************************************** Setup *********************************************
void setup(){ //Run Once
  cp5 = new ControlP5(this);
  guiTabFont=createFont(mainFontStyle, guiTabFontSize);  //Font for TABs of GUI
  guiBaseFont=createFont(mainFontStyle, guiBaseFontSize); //Font for GUI controllers other than TABs (with all Large Case letters)
  guiBaseFontSmallCase=createFont(mainFontStyle, guiBaseFontSize+2); //Font for GUI controllers other than TABs (with small case letters)
  guiHeadingFont=createFont(headingFontStyle, guiBaseFontSize+2); //Font for Headings within a tab
  size(1200, 700);               //Window size, (width, height)
  messageBoxInitialization();    //messageBox style setting 
  tooltipInitialization();       //tooltip setting
  serialPortInitialization();    //Serial port checking, selection and setting
  guiStructureInitialization();  //Creates GUI strutrue and creates I2C address selector button
  boolean ret = I2CAddressSelection();         //Asking I2C address of MCD and checking if its Valid or not  //return false:ConnectionError, true:Success
  if(ret==true){
    readAllICRegisters();          //Read all MCD registers to intialize all the controllers in GUI with same setting as MCD
    guiInitialization();           //Add various controllers to GUI structure created via guiStructureInitialization()  
  }
}

// ********************************** Draw ****************************************************
void draw(){  //Infinite Loop
  background(guiBackgroundColor); //Background colour of window 
  //Tab Area Outline
  strokeWeight(5);
  stroke(tabActiveColor); // Outline Color : same as tab color
  fill(tabAreaBackgroundColor);  //Tab Area Background color
  rect(2,tabHeight+2,800,665); //x,y,with,height
  //Graph 
  drawGraph();
  drawI2CandComPortSection();  //Draws Non-Tooltip items of I2C & COM port section
  drawMcdStatusSection();      //Draws Non-Tooltip items of MCD Status section
  pollingMCD();                //Reads status and speed registers 0,29,30 of MCD every 1 second
  updateSlopeIfRequired();
  invalidSettingErrorDisplay(); //Displays error ONCE incase of invalid setting
  displayValidControllers();
  updateSliderDisplayLabel();
  //if(debug){
  //  textSize(14);
  //  fill(0);
  //  text("X: "+mouseX,880,695);  //For displaying Mouse Location
  //  text("Y: "+mouseY,930,695);
  //}
}



// ********************************** GUI Initialization *************************************************
void guiInitialization(){        //GUI generation
  int x,y,dx,dy,len;

// Adding GUI controllers
// Button  ======================================================================================================  
////Button in tab global (always visibale area)*************************
  cp5.addButton("button_comPortReset").setPosition(975,320).setSize(100,30).setFont(createFont(mainFontStyle, 13)).setLabel("Port Reset").moveTo("global");
  cp5.addButton("button_i2cTest").setPosition(1005,355).setSize(70,30).setFont(createFont(mainFontStyle, 13)).setLabel("Test I2C").moveTo("global");
  cp5.addButton("button_saveSettingFile").setPosition(820, 575).setSize(170, 50).setFont(guiBaseFont).setLabel("Save Setting").moveTo("global");     
  cp5.addButton("button_loadSettingFile").setPosition(1000, 575).setSize(170, 50).setFont(guiBaseFont).setLabel("Load Setting").moveTo("global");    
  cp5.addButton("button_reloadFromMcd").setPosition(1000, 640).setSize(170, 50).setFont(guiBaseFont).setLabel("Reload from MCD").moveTo("global");     

////Button in tab3 (Registers)*************************
  cp5.addButton("button_ReadReg").setPosition(280,615).setSize(80,30).setFont(guiBaseFont).setLabel("READ").moveTo("tab3");
  cp5.addButton("button_WriteReg").setPosition(280,655).setSize(80,30).setFont(guiBaseFont).setLabel("WRITE").moveTo("tab3");
  cp5.addButton("button_RegFieldClear").setPosition(380,655).setSize(80,30).setFont(guiBaseFont).setLabel("CLEAR").moveTo("tab3");

////Button in tab4 (NVM)*************************
  cp5.addButton("button_ReadNvm").setPosition(130,275).setSize(200,80).setFont(guiBaseFont).setLabel("READ NVM").moveTo("tab4");
  cp5.addButton("button_WriteNvm").setPosition(130,575).setSize(200,80).setFont(guiBaseFont).setLabel("WRITE NVM").moveTo("tab4").setVisible(false);


// Slider  ========================================================================================================
////Slider in tab default (Configuration Control)*************************
  x=290;y=160;dy=30;
  cp5.addTextlabel("label_SourceCurrentDisplay").setText(sourceCurrentData[ic.getVar(ic.source)]).setPosition(x-90,y).setFont(guiHeadingFont).setColor(colorBlueActive).moveTo("default"); //label without Tooltip for showing value correspoinding to slider
  cp5.addSlider("slider_SourceCurrent").setBroadcast(false).setPosition(x,y).setSize(460,20).setRange(0,sourceCurrentData.length-1).setNumberOfTickMarks(sourceCurrentData.length).changeValue(ic.getVar(ic.source)).setDecimalPrecision(0).setFont(guiBaseFont).moveTo("default").setTriggerEvent(Slider.RELEASE).setSliderMode(Slider.FLEXIBLE).showTickMarks(false).setHandleSize(40).getCaptionLabel().hide();
  cp5.getController("slider_SourceCurrent").setBroadcast(true);
  
  cp5.addTextlabel("label_SinkCurrentDisplay").setText(sinkCurrentData[ic.getVar(ic.sink)]).setPosition(x-90,y+dy).setFont(guiHeadingFont).setColor(colorBlueActive).moveTo("default"); //label without Tooltip for showing value correspoinding to slider
  cp5.addSlider("slider_SinkCurrent").setBroadcast(false).setPosition(x,y+dy).setSize(460,20).setRange(0,sinkCurrentData.length-1).setNumberOfTickMarks(sourceCurrentData.length).changeValue(ic.getVar(ic.sink)).setDecimalPrecision(0).setFont(guiBaseFont).moveTo("default").setTriggerEvent(Slider.RELEASE).setSliderMode(Slider.FLEXIBLE).showTickMarks(false).setHandleSize(40).getCaptionLabel().hide();
  cp5.getController("slider_SinkCurrent").setBroadcast(true);
  
  cp5.addTextlabel("label_DutyChnageLimitDisplay").setText(dutyChangeLimitData[ic.getVar(ic.dutychglimit)]).setPosition(x-90,y+2*dy).setFont(guiHeadingFont).setColor(colorBlueActive).moveTo("default"); //label without Tooltip for showing value correspoinding to slider
  cp5.addSlider("slider_DutyChangeLimit").setBroadcast(false).setPosition(x,y+2*dy).setSize(460,20).setRange(0,dutyChangeLimitData.length-1).setNumberOfTickMarks(dutyChangeLimitData.length).changeValue(ic.getVar(ic.dutychglimit)).setDecimalPrecision(0).setFont(guiBaseFont).moveTo("default").setTriggerEvent(Slider.RELEASE).setSliderMode(Slider.FLEXIBLE).showTickMarks(false).setHandleSize(40).getCaptionLabel().hide();
  cp5.getController("slider_DutyChangeLimit").setBroadcast(true);
  
  cp5.addTextlabel("label_DutyChangeTimeDisplay").setText(dutyChangeTimeData[ic.getVar(ic.duty_up_time)]).setPosition(x-90,y+3*dy).setFont(guiHeadingFont).setColor(colorBlueActive).moveTo("default"); //label without Tooltip for showing value correspoinding to slider
  cp5.addSlider("slider_DutyChangeTime").setBroadcast(false).setPosition(x,y+3*dy).setSize(83,20).setRange(0,dutyChangeTimeData.length-1).setNumberOfTickMarks(dutyChangeTimeData.length).changeValue(ic.getVar(ic.duty_up_time)).setDecimalPrecision(0).setFont(guiBaseFont).moveTo("default").setTriggerEvent(Slider.RELEASE).setSliderMode(Slider.FLEXIBLE).showTickMarks(false).setHandleSize(40).getCaptionLabel().hide();
  cp5.getController("slider_DutyChangeTime").setBroadcast(true);
  
  cp5.addTextlabel("label_DeadTimeDisplay").setText(deadTimeData[ic.getVar(ic.deadtime)]).setPosition(x-90,y+4*dy).setFont(guiHeadingFont).setColor(colorBlueActive).moveTo("default"); //label without Tooltip for showing value correspoinding to slider
  cp5.addSlider("slider_DeadTime").setBroadcast(false).setPosition(x,y+4*dy).setSize(210,20).setRange(0,deadTimeData.length-1).setNumberOfTickMarks(deadTimeData.length).changeValue(ic.getVar(ic.deadtime)).setDecimalPrecision(0).setFont(guiBaseFont).moveTo("default").setTriggerEvent(Slider.RELEASE).setSliderMode(Slider.FLEXIBLE).showTickMarks(false).setHandleSize(40).getCaptionLabel().hide();
  cp5.getController("slider_DeadTime").setBroadcast(true);

  cp5.addTextlabel("label_RestartTimeDisplay").setText(restartTimeData[ic.getVar(ic.tre)]).setPosition(190,650).setFont(guiHeadingFont).setColor(colorBlueActive).moveTo("default"); //label without Tooltip for showing value correspoinding to slider
  cp5.addSlider("slider_RestartTime").setBroadcast(false).setPosition(245,650).setSize(505,20).setRange(0,restartTimeData.length-1).setNumberOfTickMarks(restartTimeData.length).changeValue(ic.getVar(ic.tre)).setDecimalPrecision(0).setFont(guiBaseFont).moveTo("default").setTriggerEvent(Slider.RELEASE).setSliderMode(Slider.FLEXIBLE).showTickMarks(false).setHandleSize(40).getCaptionLabel().hide();
  cp5.getController("slider_RestartTime").setBroadcast(true);

////Slider in tab1 (Sequence Control)*************************
  x=290;y=192;dy=30;
  cp5.addTextlabel("label_FirstDcExcitationDisplay").setText(firstDcExcitationData[ic.getVar(ic.pretip)]).setPosition(x-95,y).setFont(guiHeadingFont).setColor(colorBlueActive).moveTo("tab1"); //label without Tooltip for showing value correspoinding to slider
  cp5.addSlider("slider_FirstDcExcitation").setBroadcast(false).setPosition(x,y).setSize(210,20).setRange(0,firstDcExcitationData.length-1).setNumberOfTickMarks(firstDcExcitationData.length).changeValue(ic.getVar(ic.pretip)).setDecimalPrecision(0).setFont(guiBaseFont).moveTo("tab1").setTriggerEvent(Slider.RELEASE).setSliderMode(Slider.FLEXIBLE).showTickMarks(false).setHandleSize(40).getCaptionLabel().hide();
  cp5.getController("slider_FirstDcExcitation").setBroadcast(true);
  
  cp5.addTextlabel("label_SecondDcExcitationDisplay").setText(secondDcExcitationData[ic.getVar(ic.tip)]).setPosition(x-95,y+dy).setFont(guiHeadingFont).setColor(colorBlueActive).moveTo("tab1"); //label without Tooltip for showing value correspoinding to slider
  cp5.addSlider("slider_SecondDcExcitation").setBroadcast(false).setPosition(x,y+dy).setSize(460,20).setRange(0,secondDcExcitationData.length-1).setNumberOfTickMarks(secondDcExcitationData.length).changeValue(ic.getVar(ic.tip)).setDecimalPrecision(0).setFont(guiBaseFont).moveTo("tab1").setTriggerEvent(Slider.RELEASE).setSliderMode(Slider.FLEXIBLE).showTickMarks(false).setHandleSize(40).getCaptionLabel().hide();
  cp5.getController("slider_SecondDcExcitation").setBroadcast(true);

  cp5.addTextlabel("label_ForcedFreqDisplay").setText(forcedFreqData[ic.getVar(ic.fst)]).setPosition(x-95,y+dy*2).setFont(guiHeadingFont).setColor(colorBlueActive).moveTo("tab1"); //label without Tooltip for showing value correspoinding to slider
  cp5.addSlider("slider_ForcedFreq").setBroadcast(false).setPosition(x,y+dy*2).setSize(210,20).setRange(0,forcedFreqData.length-1).setNumberOfTickMarks(forcedFreqData.length).changeValue(ic.getVar(ic.fst)).setDecimalPrecision(0).setFont(guiBaseFont).moveTo("tab1").setTriggerEvent(Slider.RELEASE).setSliderMode(Slider.FLEXIBLE).showTickMarks(false).setHandleSize(40).getCaptionLabel().hide();
  cp5.getController("slider_ForcedFreq").setBroadcast(true);

  cp5.addTextlabel("label_CompHysDisplay").setText(compHysData[ic.getVar(ic.comp_hys)]).setPosition(x-95,y+dy*3).setFont(guiHeadingFont).setColor(colorBlueActive).moveTo("tab1"); //label without Tooltip for showing value correspoinding to slider
  cp5.addSlider("slider_CompHys").setBroadcast(false).setPosition(x,y+dy*3).setSize(210,20).setRange(0,compHysData.length-1).setNumberOfTickMarks(compHysData.length).changeValue(ic.getVar(ic.comp_hys)).setDecimalPrecision(0).setFont(guiBaseFont).moveTo("tab1").setTriggerEvent(Slider.RELEASE).setSliderMode(Slider.FLEXIBLE).showTickMarks(false).setHandleSize(40).getCaptionLabel().hide();
  cp5.getController("slider_CompHys").setBroadcast(true);
  
  y+=190;
  cp5.addTextlabel("label_StartCurrentDisplay").setText(startCurrentData[ic.getVar(ic.startcurrent)]).setPosition(x-95,y).setFont(guiHeadingFont).setColor(colorBlueActive).moveTo("tab1"); //label without Tooltip for showing value correspoinding to slider
  cp5.addSlider("slider_StartCurrent").setBroadcast(false).setPosition(x,y).setSize(460,20).setRange(0,startCurrentData.length-1).setNumberOfTickMarks(startCurrentData.length).changeValue(ic.getVar(ic.startcurrent)).setDecimalPrecision(0).setFont(guiBaseFont).moveTo("tab1").setTriggerEvent(Slider.RELEASE).setSliderMode(Slider.FLEXIBLE).showTickMarks(false).setHandleSize(40).getCaptionLabel().hide();
  cp5.getController("slider_StartCurrent").setBroadcast(true);

  cp5.addTextlabel("label_TargetCurrentDisplay").setText(targetCurrentData[ic.getVar(ic.ss_add_sel)]).setPosition(x-95,y+dy).setFont(guiHeadingFont).setColor(colorBlueActive).moveTo("tab1"); //label without Tooltip for showing value correspoinding to slider
  cp5.addSlider("slider_TargetCurrent").setBroadcast(false).setPosition(x,y+dy).setSize(210,20).setRange(0,targetCurrentData.length-1).setNumberOfTickMarks(targetCurrentData.length).changeValue(ic.getVar(ic.ss_add_sel)).setDecimalPrecision(0).setFont(guiBaseFont).moveTo("tab1").setTriggerEvent(Slider.RELEASE).setSliderMode(Slider.FLEXIBLE).showTickMarks(false).setHandleSize(40).getCaptionLabel().hide();
  cp5.getController("slider_TargetCurrent").setBroadcast(true);

  cp5.addTextlabel("label_CurrentIncrDisplay").setText(currentIncrData[ic.getVar(ic.ss_up_sel)]).setPosition(x-95,y+dy*2).setFont(guiHeadingFont).setColor(colorBlueActive).moveTo("tab1"); //label without Tooltip for showing value correspoinding to slider
  cp5.addSlider("slider_CurrentIncr").setBroadcast(false).setPosition(x,y+dy*2).setSize(210,20).setRange(0,currentIncrData.length-1).setNumberOfTickMarks(currentIncrData.length).changeValue(ic.getVar(ic.ss_up_sel)).setDecimalPrecision(0).setFont(guiBaseFont).moveTo("tab1").setTriggerEvent(Slider.RELEASE).setSliderMode(Slider.FLEXIBLE).showTickMarks(false).setHandleSize(40).getCaptionLabel().hide();
  cp5.getController("slider_CurrentIncr").setBroadcast(true);

  cp5.addTextlabel("label_SoftStartDutyDisplay").setText(sStartDutyData[ic.getVar(ic.ss_dutychglimit)]).setPosition(x-95,y+dy*3).setFont(guiHeadingFont).setColor(colorBlueActive).moveTo("tab1"); //label without Tooltip for showing value correspoinding to slider
  cp5.addSlider("slider_SoftStartDuty").setBroadcast(false).setPosition(x,y+dy*3).setSize(460,20).setRange(0,sStartDutyData.length-1).setNumberOfTickMarks(sStartDutyData.length).changeValue(ic.getVar(ic.ss_dutychglimit)).setDecimalPrecision(0).setFont(guiBaseFont).moveTo("tab1").setTriggerEvent(Slider.RELEASE).setSliderMode(Slider.FLEXIBLE).showTickMarks(false).setHandleSize(40).getCaptionLabel().hide();
  cp5.getController("slider_SoftStartDuty").setBroadcast(true);
  
  y+=190;
  cp5.addTextlabel("label_WaitTimeDisplay").setText(waitTimeData[ic.getVar(ic.wait_time)]).setPosition(x-95,y).setFont(guiHeadingFont).setColor(colorBlueActive).moveTo("tab1"); //label without Tooltip for showing value correspoinding to slider
  cp5.addSlider("slider_WaitTime").setBroadcast(false).setPosition(x,y).setSize(460,20).setRange(0,waitTimeData.length-1).setNumberOfTickMarks(waitTimeData.length).changeValue(ic.getVar(ic.wait_time)).setDecimalPrecision(0).setFont(guiBaseFont).moveTo("tab1").setTriggerEvent(Slider.RELEASE).setSliderMode(Slider.FLEXIBLE).showTickMarks(false).setHandleSize(40).getCaptionLabel().hide();
  cp5.getController("slider_WaitTime").setBroadcast(true);

////Slider in tab2 (Motor Control)*************************
  x=180;
  y=225;
  int l=600;
  float _startDuty=float(ic.getVar(ic.startduty))*100/512;
  cp5.addSlider("slider_StartDuty").setBroadcast(false).setPosition(x,y).setSize(l,20).setRange(0,49.8).changeValue(_startDuty).setDecimalPrecision(1).setFont(guiBaseFont).moveTo("tab2").setTriggerEvent(Slider.RELEASE).setLabel("%").getCaptionLabel().alignX(ControlP5.RIGHT); //.getCaptionLabel().hide(); .getCaptionLabel().alignX(ControlP5.LEFT_OUTSIDE)
  cp5.getController("slider_StartDuty").setBroadcast(true);
  
  float _stopDuty=float(ic.getVar(ic.stopduty))*100/256;
  cp5.addSlider("slider_StopDuty").setBroadcast(false).setPosition(x,y+25).setSize(l,20).setRange(0,49.6).changeValue(_stopDuty).setDecimalPrecision(1).setFont(guiBaseFont).moveTo("tab2").setTriggerEvent(Slider.RELEASE).setLabel("%").getCaptionLabel().alignX(ControlP5.RIGHT);
  cp5.getController("slider_StopDuty").setBroadcast(true);
  
  float _changeDuty=float(ic.getVar(ic.changeduty))*100/256;
  cp5.addSlider("slider_ChangeDuty").setBroadcast(false).setPosition(x,y+225).setSize(l,20).setRange(0.4,99.6).changeValue(_changeDuty).setDecimalPrecision(1).setFont(guiBaseFont).moveTo("tab2").setTriggerEvent(Slider.RELEASE).setVisible(false).setLabel("%").getCaptionLabel().alignX(ControlP5.RIGHT);
  cp5.getController("slider_ChangeDuty").setBroadcast(true);
  
  float _maxDuty=float(ic.getVar(ic.maxduty)+257)*100/512;
  cp5.addSlider("slider_MaxDuty").setBroadcast(false).setPosition(x,y+100).setSize(l,20).setRange(50.2,100).changeValue(_maxDuty).setDecimalPrecision(1).setFont(guiBaseFont).moveTo("tab2").setTriggerEvent(Slider.RELEASE).setLabel("%").getCaptionLabel().alignX(ControlP5.RIGHT);
  cp5.getController("slider_MaxDuty").setBroadcast(true);
  
  float _startRpmOL=float(ic.getVar(ic.startrpm))/16*100/512;
  cp5.addSlider("slider_MinOutputOL").setBroadcast(false).setPosition(x,y+50).setSize(l,20).setRange(0,49.8).changeValue(_startRpmOL).setDecimalPrecision(1).setFont(guiBaseFont).moveTo("tab2").setTriggerEvent(Slider.RELEASE).setLabel("%").getCaptionLabel().alignX(ControlP5.RIGHT);
  cp5.getController("slider_MinOutputOL").setBroadcast(true);
  
  float _startRpmCL=ic.getVar(ic.startrpm);
  cp5.addSlider("slider_MinOutputCL").setBroadcast(false).setPosition(x,y+50).setSize(l,20).setRange(0,4095).changeValue(_startRpmCL).setDecimalPrecision(0).setFont(guiBaseFont).moveTo("tab2").setTriggerEvent(Slider.RELEASE).setLabel("RPM").getCaptionLabel().alignX(ControlP5.RIGHT);
  cp5.getController("slider_MinOutputCL").setBroadcast(true);
  
  float _slopeOL = float(ic.getVar(ic.speedslop))/1024;
  float _slopeCL = float(ic.getVar(ic.speedslop))*8/100;
  float _slope2OL = float(ic.getVar(ic.speedslop2))/1024;
  float _slope2CL = float(ic.getVar(ic.speedslop2))*8/100;
  
  float _changeOutputOL = _startRpmOL + _slopeOL * (_changeDuty-_startDuty);
  cp5.addSlider("slider_ChangeOutputOL").setBroadcast(false).setPosition(x,y+250).setSize(l,20).setRange(0,100).changeValue(_changeOutputOL).setDecimalPrecision(1).setFont(guiBaseFont).moveTo("tab2").setTriggerEvent(Slider.RELEASE).setVisible(false).setLabel("%").getCaptionLabel().alignX(ControlP5.RIGHT);
  cp5.getController("slider_ChangeOutputOL").setBroadcast(true);
  
  float _changeOutputCL = _startRpmCL + _slopeCL * (_changeDuty-_startDuty);
  cp5.addSlider("slider_ChangeOutputCL").setBroadcast(false).setPosition(x,y+250).setSize(l,20).setRange(0,MaxCloseLoopMotorRPM).changeValue(_changeOutputCL).setDecimalPrecision(0).setFont(guiBaseFont).moveTo("tab2").setTriggerEvent(Slider.RELEASE).setVisible(false).setLabel("RPM").getCaptionLabel().alignX(ControlP5.RIGHT);
  cp5.getController("slider_ChangeOutputCL").setBroadcast(true);
  float _maxOutputOL, _maxOutputCL;
  if(ic.getVar(ic.changeduty)==0){ //Two point profile
    _maxOutputOL = _startRpmOL + _slopeOL * (_maxDuty-_startDuty);
    _maxOutputCL = _startRpmCL + _slopeCL * (_maxDuty-_startDuty);
  }else{ //Three point profile
    _maxOutputOL = _changeOutputOL + _slope2OL * (_maxDuty-_changeDuty);
    _maxOutputCL = _changeOutputCL + _slope2CL * (_maxDuty-_changeDuty);
  }
  cp5.addSlider("slider_MaxOutputOL").setBroadcast(false).setPosition(x,y+125).setSize(l,20).setRange(0,100).changeValue(_maxOutputOL).setDecimalPrecision(1).setFont(guiBaseFont).moveTo("tab2").setTriggerEvent(Slider.RELEASE).setLabel("%").getCaptionLabel().alignX(ControlP5.RIGHT);
  cp5.getController("slider_MaxOutputOL").setBroadcast(true);
  cp5.addSlider("slider_MaxOutputCL").setBroadcast(false).setPosition(x,y+125).setSize(l,20).setRange(0,MaxCloseLoopMotorRPM).changeValue(_maxOutputCL).setDecimalPrecision(0).setFont(guiBaseFont).moveTo("tab2").setTriggerEvent(Slider.RELEASE).setLabel("RPM").getCaptionLabel().alignX(ControlP5.RIGHT);
  cp5.getController("slider_MaxOutputCL").setBroadcast(true);

  cp5.addSlider("slider_MaxDutyHysteresis").setBroadcast(false).setPosition(x,y+150).setSize(l/2,20).setRange(0.4,6.4).changeValue((ic.getVar(ic.maxdutyhys)+1)*0.4).setDecimalPrecision(1).setFont(guiBaseFont).moveTo("tab2").setTriggerEvent(Slider.RELEASE).setSliderMode(Slider.FLEXIBLE).setNumberOfTickMarks(16).showTickMarks(false).setHandleSize(40).setLabel("%").getCaptionLabel().alignX(ControlP5.RIGHT);
  cp5.getController("slider_MaxDutyHysteresis").setBroadcast(true);
  cp5.addSlider("slider_KP").setBroadcast(false).setPosition(x,y+300).setSize(l/2,20).setRange(0,127).changeValue(ic.getVar(ic.kp)).setDecimalPrecision(0).setFont(guiBaseFont).moveTo("tab2").setTriggerEvent(Slider.RELEASE).getCaptionLabel().hide();
  cp5.getController("slider_KP").setBroadcast(true);
  cp5.addSlider("slider_KI").setBroadcast(false).setPosition(x,y+325).setSize(l/2,20).setRange(0,127).changeValue(ic.getVar(ic.ki)).setDecimalPrecision(0).setFont(guiBaseFont).moveTo("tab2").setTriggerEvent(Slider.RELEASE).getCaptionLabel().hide();
  cp5.getController("slider_KI").setBroadcast(true);
  cp5.addSlider("slider_SPD").setBroadcast(false).setPosition(x,y+405).setSize(l,20).setRange(0,100).changeValue(float(ic.getVar(ic.spd))*100/512).setDecimalPrecision(1).setFont(guiBaseFont).moveTo("tab2").setTriggerEvent(Slider.RELEASE).setLabel("%").getCaptionLabel().alignX(ControlP5.RIGHT);
  cp5.getController("slider_SPD").setBroadcast(true);

// Textlabel  =====================================================================================
//// Labels in tab defalut (Control Configuration)***********************
  x=10;
  y=40;
  cp5.addTextlabel("label_DriveControl").setText("Driver Control:").setPosition(x,y).setFont(guiHeadingFont).setColor(labelColor).moveTo("default"); //label without Tooltip
  y+=40;
  // Labels with tooltip display, ID of following should match it's index, tooltip text is set in txtlabelTooltipText[] array.
  txtlabel_tabDefault[0] = cp5.addTextlabel("label_SpeedControl").setId(0).setText("Speed Control").setPosition(x+20,y).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  txtlabel_tabDefault[1] = cp5.addTextlabel("label_DirControl").setId(1).setText("Dir Control").setPosition(x+300,y).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  txtlabel_tabDefault[2] = cp5.addTextlabel("label_Brake").setId(2).setText("Brake").setPosition(x+570,y).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  y+=30;
  txtlabel_tabDefault[3] = cp5.addTextlabel("label_SpeedControlInverse").setId(3).setText("Speed Control Inverse").setPosition(x+20,y).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  txtlabel_tabDefault[4] = cp5.addTextlabel("label_LeadAngle").setId(4).setText("Lead Angle").setPosition(x+300,y).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  txtlabel_tabDefault[5] = cp5.addTextlabel("label_PwmFreq").setId(5).setText("PWM Freq.").setPosition(x+530,y).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  cp5.addTextlabel("label_OutPwmFreqUnit").setText("kHz").setPosition(x+740,y).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default"); //label without Tooltip
  y+=50;
  txtlabel_tabDefault[6] = cp5.addTextlabel("label_SourceCurrent").setId(6).setText("Source Current").setPosition(x+46,y).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  txtlabel_tabDefault[7] = cp5.addTextlabel("label_SinkCurrent").setId(7).setText("Sink Current").setPosition(x+68,y+30).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  txtlabel_tabDefault[8] = cp5.addTextlabel("label_DutyChangeLimit").setId(8).setText("Duty Change Limit").setPosition(x+20,y+60).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  txtlabel_tabDefault[9] = cp5.addTextlabel("label_DutyChangeTime").setId(9).setText("Duty Change Time").setPosition(x+20,y+90).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  txtlabel_tabDefault[10] = cp5.addTextlabel("label_DeadTime").setId(10).setText("Dead Time").setPosition(x+80,y+120).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  txtlabel_tabDefault[11] = cp5.addTextlabel("label_DisableAntiTrough").setId(11).setText("Disable Anti-Through").setPosition(x+540,y+120).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  y+=180;
  cp5.addTextlabel("label_ProtectionControl").setText("Protection Control:").setPosition(x,y).setFont(guiHeadingFont).setColor(labelColor).moveTo("default"); //label without Tooltip
  y+=50;
  txtlabel_tabDefault[12] = cp5.addTextlabel("label_OcpMask").setId(12).setText("OCP Mask").setPosition(x+77,y+30).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  txtlabel_tabDefault[13] = cp5.addTextlabel("label_IsdMask").setId(13).setText("ISD Mask").setPosition(x+85,y+60).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  txtlabel_tabDefault[14] = cp5.addTextlabel("label_TsdMask").setId(14).setText("TSD Mask").setPosition(x+79,y+90).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  txtlabel_tabDefault[15] = cp5.addTextlabel("label_LockMask").setId(15).setText("LOCK Mask").setPosition(x+68,y+120).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  txtlabel_tabDefault[16] = cp5.addTextlabel("label_VcpMask").setId(16).setText("VCP Mask").setPosition(x+79,y+150).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  txtlabel_tabDefault[17] = cp5.addTextlabel("label_OcpLevel").setId(17).setText("OCP Level").setPosition(x+250,y+30).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  txtlabel_tabDefault[18] = cp5.addTextlabel("label_RsaFilter").setId(18).setText("RSA Filter").setPosition(x+530,y+30).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  cp5.addTextlabel("label_RsaFilterUnit").setText("kHz").setPosition(x+745,y+30).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default"); //label without Tooltip
  txtlabel_tabDefault[19] = cp5.addTextlabel("label_IsdLevel").setId(19).setText("ISD Level").setPosition(x+260,y+70).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  cp5.addTextlabel("label_IsdUnit").setText("V").setPosition(x+470,y+70).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default"); //label without Tooltip
  txtlabel_tabDefault[20] = cp5.addTextlabel("label_MaskTime").setId(20).setText("Mask Time").setPosition(x+525,y+70).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  cp5.addTextlabel("label_MaskTimeUnit").setText("ns").setPosition(x+745,y+70).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default"); //label without Tooltip
  txtlabel_tabDefault[21] = cp5.addTextlabel("label_FMAX").setId(21).setText("FMAX").setPosition(x+293,y+110).setFont(guiBaseFont).setColor(labelColor).moveTo("default");
  cp5.addTextlabel("label_FMAXUnit").setText("kHz").setPosition(x+470,y+110).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default"); //label without Tooltip
  txtlabel_tabDefault[22] = cp5.addTextlabel("label_LockStatus").setId(22).setText("Lock Status").setPosition(x+519,y+110).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  txtlabel_tabDefault[23] = cp5.addTextlabel("label_AlertInverse").setId(23).setText("Alert Inverse").setPosition(x+65,y-10).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  y+=200;
  cp5.addTextlabel("label_RestartControl").setText("Restart Control:").setPosition(x,y).setFont(guiHeadingFont).setColor(labelColor).moveTo("default"); //label without Tooltip
  txtlabel_tabDefault[24] = cp5.addTextlabel("label_Latch").setId(24).setText("Recovery Mode").setPosition(x+38,y+30).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  txtlabel_tabDefault[25] = cp5.addTextlabel("label_RestartTime").setId(25).setText("Restart Time").setPosition(x+58,y+60).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("default");
  
//// Labels in tab1 (Sequence Control)************************
  x=10;
  y=40;
  cp5.addTextlabel("label_StandbyPin").setText("Standby Pin:").setPosition(x,y).setFont(guiHeadingFont).setColor(labelColor).moveTo("tab1"); //label without Tooltip
  // Labels with tooltip display, ID of following should match it's index, tooltip text is set in txtlabelTooltipText[] array.
  txtlabel_tab1[0] = cp5.addTextlabel("label_StandbyMode").setId(0).setText("Standby Mode").setPosition(x+38,y+50).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab1");
  y+=120;
  cp5.addTextlabel("label_StartSequence").setText("START Sequence:").setPosition(x,y-10).setFont(guiHeadingFont).setColor(labelColor).moveTo("tab1"); //label without Tooltip
  txtlabel_tab1[1] = cp5.addTextlabel("label_FirstDcExcitation").setId(1).setText("1st DC Excitation").setPosition(x+25,y+30).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab1");
  txtlabel_tab1[2] = cp5.addTextlabel("label_SecondDcExcitation").setId(2).setText("2nd DC Excitation").setPosition(x+20,y+60).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab1");
  txtlabel_tab1[3] = cp5.addTextlabel("label_ForcedFreq").setId(3).setText("Forced Freq.").setPosition(x+65,y+90).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab1");
  txtlabel_tab1[4] = cp5.addTextlabel("label_CompHys").setId(4).setText("Comp. Hys.").setPosition(x+72,y+120).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab1");
  y+=190;
  cp5.addTextlabel("label_SoftStartControl").setText("SOFT START Control:").setPosition(x,y-10).setFont(guiHeadingFont).setColor(labelColor).moveTo("tab1"); //label without Tooltip
  txtlabel_tab1[5] = cp5.addTextlabel("label_StartCurrent").setId(5).setText("Start Current").setPosition(x+60,y+30).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab1");
  txtlabel_tab1[6] = cp5.addTextlabel("label_TargetCurrent").setId(6).setText("Target Current").setPosition(x+48,y+60).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab1");
  txtlabel_tab1[7] = cp5.addTextlabel("label_CurrentIncr").setId(7).setText("Current Incr.").setPosition(x+65,y+90).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab1");
  txtlabel_tab1[8] = cp5.addTextlabel("label_SStartDuty").setId(8).setText("Soft Start Duty Chng.").setPosition(x+4,y+120).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab1");
  y+=190;
  cp5.addTextlabel("label_BrakeSequence").setText("BRAKE Sequence:").setPosition(x,y-10).setFont(guiHeadingFont).setColor(labelColor).moveTo("tab1"); //label without Tooltip
  txtlabel_tab1[9] = cp5.addTextlabel("label_WaitTime").setId(9).setText("Wait Time").setPosition(x+80,y+30).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab1");
  txtlabel_tab1[10] = cp5.addTextlabel("label_WaitMode").setId(10).setText("Wait Mode").setPosition(x+73,y+92).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab1");
  txtlabel_tab1[11] = cp5.addTextlabel("label_AfterWait").setId(11).setText("After Wait").setPosition(x+450,y+92).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab1");
  
//// Labels in tab2 (Motor Control)************************* 
  // Labels without tooltip display in tab2
  x=5;
  cp5.addTextlabel("label_MotorInfo").setText("Motor Info:").setPosition(x,50).setFont(guiHeadingFont).setColor(labelColor).moveTo("tab2");
  cp5.addTextlabel("label_CurveOptions").setText("Curve Options:").setPosition(x,90).setFont(guiHeadingFont).setColor(labelColor).moveTo("tab2");
  cp5.addTextlabel("label_StartPoint").setText("Start Point:").setPosition(x,200).setFont(guiHeadingFont).setColor(labelColor).moveTo("tab2");
  cp5.addTextlabel("label_MaxPoint").setText("Max Point:").setPosition(x,300).setFont(guiHeadingFont).setColor(labelColor).moveTo("tab2");
  cp5.addTextlabel("label_ChangePoint").setText("Change Point:").setPosition(x,405).setFont(guiHeadingFont).setColor(labelColor).moveTo("tab2");
  cp5.addTextlabel("label_ClosedLoop").setText("Closed Loop:").setPosition(x,505).setFont(guiHeadingFont).setColor(labelColor).moveTo("tab2");
  cp5.addTextlabel("label_Misc").setText("Misc:").setPosition(x,585).setFont(guiHeadingFont).setColor(labelColor).moveTo("tab2");
  // Labels with tooltip display, ID of following should match it's index, tooltip text is set in txtlabelTooltipText[] array.
  txtlabel_tab2[0] = cp5.addTextlabel("label_OpenCloseLoop").setId(0).setText("Speed Control").setPosition(45,120).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab2");
  txtlabel_tab2[1] = cp5.addTextlabel("label_NoStop").setId(1).setText("NOSTOP").setPosition(170,162).setFont(guiBaseFont).setColor(labelColor).moveTo("tab2");
  txtlabel_tab2[2] = cp5.addTextlabel("label_MaxOff").setId(2).setText("MAXOFF").setPosition(320,162).setFont(guiBaseFont).setColor(labelColor).moveTo("tab2");
  txtlabel_tab2[3] = cp5.addTextlabel("label_MaxOpen").setId(3).setText("MAXOPEN").setPosition(470,162).setFont(guiBaseFont).setColor(labelColor).moveTo("tab2");
  x=28;
  y=225;
  txtlabel_tab2[4] = cp5.addTextlabel("label_StartDuty").setId(4).setText("Start Duty (1)").setPosition(x+34,y).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab2");
  txtlabel_tab2[5] = cp5.addTextlabel("label_StopDuty").setId(5).setText("Stop Duty (2)").setPosition(x+35,y+25).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab2");
  txtlabel_tab2[6] = cp5.addTextlabel("label_MinOutput").setId(6).setText("Min Output (4)").setPosition(x+27,y+50).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab2");
  txtlabel_tab2[7] = cp5.addTextlabel("label_MaxDuty").setId(7).setText("Max Duty (3)").setPosition(x+38,y+100).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab2");
  txtlabel_tab2[8] = cp5.addTextlabel("label_MaxOutput").setId(8).setText("Max Output (5)").setPosition(x+22,y+125).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab2");
  txtlabel_tab2[9] = cp5.addTextlabel("label_MaxDutyHys").setId(9).setText("Max Duty Hys").setPosition(x+24,y+150).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab2");
  txtlabel_tab2[10] = cp5.addTextlabel("label_ChangePointEn").setId(10).setText("Enable").setPosition(x+80,y+200).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab2");
  txtlabel_tab2[11] = cp5.addTextlabel("label_ChangeDuty").setId(11).setText("Change Duty (8)").setPosition(x+11,y+225).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab2");
  txtlabel_tab2[12] = cp5.addTextlabel("label_ChangeOutput").setId(12).setText("Change Output (10)").setPosition(x-13,y+250).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab2");
  txtlabel_tab2[13] = cp5.addTextlabel("label_KP").setId(13).setText("KP").setPosition(x+117,y+300).setFont(guiBaseFont).setColor(labelColor).moveTo("tab2");
  txtlabel_tab2[14] = cp5.addTextlabel("label_KPX").setId(14).setText("KPX").setPosition(x+525,y+300).setFont(guiBaseFont).setColor(labelColor).moveTo("tab2");
  txtlabel_tab2[15] = cp5.addTextlabel("label_KI").setId(15).setText("KI").setPosition(x+119,y+325).setFont(guiBaseFont).setColor(labelColor).moveTo("tab2");
  txtlabel_tab2[16] = cp5.addTextlabel("label_KIX").setId(16).setText("KIX").setPosition(x+527,y+325).setFont(guiBaseFont).setColor(labelColor).moveTo("tab2");
  txtlabel_tab2[17] = cp5.addTextlabel("label_RPMLimit").setId(17).setText("RPM Limit").setPosition(x+478,y+370).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab2");
  txtlabel_tab2[18] = cp5.addTextlabel("label_FGSEL").setId(18).setText("FGSEL").setPosition(x+235,y+372).setFont(guiBaseFont).setColor(labelColor).moveTo("tab2");
  txtlabel_tab2[19] = cp5.addTextlabel("label_FG_ON").setId(19).setText("FG_ON").setPosition(x+87,y+372).setFont(guiBaseFont).setColor(labelColor).moveTo("tab2");
  txtlabel_tab2[20] = cp5.addTextlabel("label_SPD").setId(20).setText("SPD").setPosition(x+107,y+405).setFont(guiBaseFont).setColor(labelColor).moveTo("tab2");
  txtlabel_tab2[21] = cp5.addTextlabel("label_PolePairs").setId(21).setText("Poles").setPosition(165,52).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab2");
  txtlabel_tab2[22] = cp5.addTextlabel("label_MaxSpeed").setId(22).setText("Max Speed").setPosition(378,52).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab2");
  //Labels to display unavailable status of other controllers.
  cp5.addTextlabel("label_UA_changePoint1").setText("Unavailable (En Change Point)").setPosition(180,452).setFont(guiBaseFont).setColor(120).moveTo("tab2");
  cp5.addTextlabel("label_UA_changePoint2").setText("Unavailable (En Change Point)").setPosition(180,477).setFont(guiBaseFont).setColor(120).moveTo("tab2");
 
//// Labels in tab3 (Registers)************************
  // Labels without tooltip
  cp5.addTextlabel("label_RegisterList").setText("Register Data").setPosition(70,45).setFont(guiHeadingFont).setColor(labelColor).moveTo("tab3");
  cp5.addTextlabel("label_VariableList").setText("Parameter Data").setPosition(370,45).setFont(guiHeadingFont).setColor(labelColor).moveTo("tab3"); //(Variable in Code is same as Parameter in datasheet)
  cp5.addTextlabel("label_SingleRegReadWrite").setText("Single Register Read/Write (MCD)").setPosition(10,580).setFont(guiHeadingFont).setColor(labelColor).moveTo("tab3");
  cp5.addTextlabel("label_RegAdd").setText("Reg. Add.").setPosition(50,620).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab3");
  cp5.addTextlabel("label_RegAddUnit").setText("Dec").setPosition(190,620).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab3");
  cp5.addTextlabel("label_RegData").setText("Reg. Data").setPosition(50,660).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab3");
  cp5.addTextlabel("label_RegDataUnit").setText("Hex").setPosition(190,660).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab3");

//// Labels in tab4 (NVM)************************
  // Labels without tooltip
  x=10;y=70;
  cp5.addTextlabel("label_NvmInfo").setText("Register Settings of MCD can be stored in its Non-Volatile Memory (NVM).").setPosition(x+90,y).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab4");
  y+=150;
  cp5.addTextlabel("label_ReadNvm").setText("Read NVM: Load data from NVM to Registers.").setPosition(x,y).setFont(guiHeadingFont).setColor(labelColor).moveTo("tab4");
  y+=250;
  cp5.addTextlabel("label_WriteNvm").setText("Write NVM: Store data of Registers in NVM.").setPosition(x,y).setFont(guiHeadingFont).setColor(labelColor).moveTo("tab4");
  cp5.addTextlabel("label_WriteNvmConfirm").setText("I understand that NVM can be written only few times.").setPosition(x+150,y+42).setFont(guiBaseFontSmallCase).setColor(labelColor).moveTo("tab4");

// TextArea ========================================================================================================
//// TextArea in tab3 (Registers)************************
  textArea_registerList = cp5.addTextarea("textArea_registerList").setPosition(70,70).setSize(240,495).setFont(createFont("Monospaced",20)).setLineHeight(22).setText(ic.printRegAllInString()).setColorBackground(240).setColor(0).moveTo("tab3");
  textArea_variableList = cp5.addTextarea("textArea_variableList1").setPosition(370,70).setSize(350,495).setFont(createFont("Monospaced",20)).setLineHeight(22).setText(ic.printVarAllInString()).setColorBackground(240).setColor(0).moveTo("tab3");

// TextField ========================================================================================================
//// TextField in tab3 (Registers)************************
  cp5.addTextfield("textField_RegAdd").setPosition(145,615).setSize(40,28).setFont(guiHeadingFont).setColorCursor(240).setInputFilter(ControlP5.INTEGER).setText("0").moveTo("tab3").getCaptionLabel().hide();
  cp5.addTextfield("textField_RegData").setPosition(145,655).setSize(40,28).setFont(guiHeadingFont).setColorCursor(240).moveTo("tab3").getCaptionLabel().hide();

// Toggle ==========================================================================================================
////Toggle in tab default (Configuration Control)*************************
  cp5.addToggle("toggle_SpeedControl").setLabel("   PWM   Analog").changeValue(ic.getVar(ic.tspsel)).setPosition(165,80).setSize(110,20).setColorActive(controllerActiveColor).moveTo("default").setMode(ControlP5.SWITCH).setFont(createFont("Arial",14)).setColorLabel(labelColor).getCaptionLabel().alignX(ControlP5.CENTER).alignY(ControlP5.TOP_OUTSIDE);
  cp5.addToggle("toggle_DirControl").setLabel("H:CW    L:CW").changeValue(ic.getVar(ic.dir)).setPosition(408,80).setSize(110,20).setColorActive(controllerActiveColor).moveTo("default").setMode(ControlP5.SWITCH).setFont(createFont("Arial",14)).setColorLabel(labelColor).getCaptionLabel().alignX(ControlP5.CENTER).alignY(ControlP5.TOP_OUTSIDE);
  cp5.addToggle("toggle_Brake").setLabel("L:Brake  H:Brake").changeValue(ic.getVar(ic.brk_inv)).setPosition(640,80).setSize(110,20).setColorActive(controllerActiveColor).moveTo("default").setMode(ControlP5.SWITCH).setFont(createFont("Arial",14)).setColorLabel(labelColor).getCaptionLabel().alignX(ControlP5.CENTER).alignY(ControlP5.TOP_OUTSIDE);
  cp5.addToggle("toggle_SpeedControlInverse").changeValue(ic.getVar(ic.spdinv)).setPosition(220,110).setSize(20,20).setColorActive(controllerActiveColor).moveTo("default").getCaptionLabel().hide();
  cp5.addToggle("toggle_DisableAntiTrought").changeValue(ic.getVar(ic.antithrough)).setPosition(730,281).setSize(20,20).setColorActive(controllerActiveColor).moveTo("default").getCaptionLabel().hide();
  cp5.addToggle("toggle_AlertInverse").setLabel("L:Error    H:Error").changeValue(ic.getVar(ic.alertinv)).setPosition(190,382).setSize(150,20).setColorActive(controllerActiveColor).moveTo("default").setMode(ControlP5.SWITCH).setFont(createFont("Arial",14)).setColorLabel(labelColor).getCaptionLabel().alignX(ControlP5.CENTER).alignY(ControlP5.TOP_OUTSIDE);
  cp5.addToggle("toggle_OcpMask").changeValue(ic.getVar(ic.ocpdis)).setPosition(190,422).setSize(20,20).setColorActive(controllerActiveColor).moveTo("default").getCaptionLabel().hide();
  cp5.addToggle("toggle_IsdMask").changeValue(ic.getVar(ic.isd_mask)).setPosition(190,452).setSize(20,20).setColorActive(controllerActiveColor).moveTo("default").getCaptionLabel().hide();
  cp5.addToggle("toggle_TsdMask").changeValue(ic.getVar(ic.tsd_mask)).setPosition(190,482).setSize(20,20).setColorActive(controllerActiveColor).moveTo("default").getCaptionLabel().hide();
  cp5.addToggle("toggle_LockMask").changeValue(ic.getVar(ic.lockdis)).setPosition(190,512).setSize(20,20).setColorActive(controllerActiveColor).moveTo("default").getCaptionLabel().hide();
  cp5.addToggle("toggle_VcpMask").changeValue(ic.getVar(ic.vcp_mask)).setPosition(190,542).setSize(20,20).setColorActive(controllerActiveColor).moveTo("default").getCaptionLabel().hide();
  cp5.addToggle("toggle_RecoveryMode").setLabel("Latch    Auto").changeValue(ic.getVar(ic.latch)).setPosition(190,621).setSize(110,20).setColorActive(controllerActiveColor).moveTo("default").setMode(ControlP5.SWITCH).setFont(createFont("Arial",14)).setColorLabel(labelColor).getCaptionLabel().alignX(ControlP5.CENTER).alignY(ControlP5.TOP_OUTSIDE);

////Toggle in tab1 (Sequence Control)*************************
  cp5.addToggle("toggle_StandbyMode").setLabel("   STBY & SPD    STBY Pin only").changeValue(ic.getVar(ic.stby_mode)).setPosition(200,88).setSize(200,25).setColorActive(controllerActiveColor).moveTo("tab1").setMode(ControlP5.SWITCH).setFont(createFont("Arial",14)).setColorLabel(labelColor).getCaptionLabel().alignX(ControlP5.CENTER).alignY(ControlP5.TOP_OUTSIDE);
  cp5.addToggle("toggle_WaitMode").setLabel("Short Brake    OFF(Hi-Z)   ").changeValue(ic.getVar(ic.wait_mode)).setPosition(200,635).setSize(180,20).setColorActive(controllerActiveColor).moveTo("tab1").setMode(ControlP5.SWITCH).setFont(createFont("Arial",14)).setColorLabel(labelColor).getCaptionLabel().alignX(ControlP5.CENTER).alignY(ControlP5.TOP_OUTSIDE);
  cp5.addToggle("toggle_AfterWait").setLabel("Keep          Idle").changeValue(ic.getVar(ic.wait_con)).setPosition(560,635).setSize(180,20).setColorActive(controllerActiveColor).moveTo("tab1").setMode(ControlP5.SWITCH).setFont(createFont("Arial",14)).setColorLabel(labelColor).getCaptionLabel().alignX(ControlP5.CENTER).alignY(ControlP5.TOP_OUTSIDE);

////Toggle in tab2 (Motor Control)*************************
  cp5.addToggle("toggle_OpenCloseLoop").setLabel("Open Loop       Closed Loop").changeValue(ic.getVar(ic.openloop)).setPosition(170,120).setSize(250,25).setColorActive(controllerActiveColor).moveTo("tab2").setMode(ControlP5.SWITCH).setFont(guiBaseFont).setColorLabel(labelColor).getCaptionLabel().alignX(ControlP5.CENTER).alignY(ControlP5.TOP_OUTSIDE);
  cp5.addToggle("toggle_NoStop").changeValue(ic.getVar(ic.nostop)).setPosition(250,162).setSize(20,20).setColorActive(controllerActiveColor).moveTo("tab2").getCaptionLabel().hide();
  cp5.addToggle("toggle_MaxOff").changeValue(ic.getVar(ic.maxoff)).setPosition(400,162).setSize(20,20).setColorActive(controllerActiveColor).moveTo("tab2").getCaptionLabel().hide();
  cp5.addToggle("toggle_MaxOpen").changeValue(ic.getVar(ic.maxopen)).setPosition(560,162).setSize(20,20).setColorActive(controllerActiveColor).moveTo("tab2").getCaptionLabel().hide();
  cp5.addToggle("toggle_ChangePoint").changeValue(((ic.getVar(ic.changeduty))==0)?0:1).setPosition(180,425).setSize(20,20).setColorActive(controllerActiveColor).moveTo("tab2").getCaptionLabel().hide();
  cp5.addToggle("toggle_KPX").changeValue(ic.getVar(ic.kpx)).setPosition(600,525).setSize(20,20).setColorActive(controllerActiveColor).moveTo("tab2").getCaptionLabel().hide();
  cp5.addToggle("toggle_KIX").changeValue(ic.getVar(ic.kix)).setPosition(600,550).setSize(20,20).setColorActive(controllerActiveColor).moveTo("tab2").getCaptionLabel().hide();
  cp5.addToggle("toggle_FG_ON").changeValue(ic.getVar(ic.fg_on)).setPosition(180,595).setSize(20,20).setColorActive(controllerActiveColor).moveTo("tab2").getCaptionLabel().hide();
  
////Toggle in tab4 (NVM)*************************
  cp5.addToggle("toggle_WriteNvmConfirm").setPosition(130,510).setSize(25,25).setColorActive(controllerActiveColor).moveTo("tab4").getCaptionLabel().hide();

  
// ScrollableList (Dropdown)  ====================================================================================
////ScrollableList in tab default (Configuration Control)*************************
  cp5.addScrollableList("scrollableList_LeadAngle").addItems(Arrays.asList("0°","3.75°","7.5°","11.25°","15°","18.75°","22.5°","26.25°","30°","7.5° - 22.5° A","7.5° - 22.5° B","0° - 18.75°","3.75° - 22.5°","7.5° - 26.25°","11.25° - 30°","3.75° - 30°")).changeValue(ic.getVar(ic.la)).setPosition(410,110).setSize(110,250).setBarHeight(30).setItemHeight(30).setFont(guiBaseFont).close().moveTo("default");
  cp5.addScrollableList("scrollableList_Fpwm").addItems(Arrays.asList("23.4","46.9","93.7","187.5","46.9 - 187.5","23.4 - 93.7 A","23.4 - 93.7 B","23.4 - 187.5")).changeValue(ic.getVar(ic.fpwm)).setPosition(640,110).setSize(110,250).setBarHeight(30).setItemHeight(30).setFont(guiBaseFont).close().moveTo("default");
  x=360;y=420;dx=275;dy=40;len=120;
  cp5.addScrollableList("scrollableList_Fmax").addItems(Arrays.asList("0.75","1.5","3.0","No Limit")).changeValue(ic.getVar(ic.fmax)).setPosition(x,y+2*dy).setSize(len,250).setBarHeight(30).setItemHeight(30).setFont(guiBaseFont).close().moveTo("default");
  cp5.addScrollableList("scrollableList_LockStatus").addItems(Arrays.asList("OFF (Hi-Z)","Short Brake")).changeValue(ic.getVar(ic.lock_brk)).setPosition(x+dx,y+2*dy).setSize(len,250).setBarHeight(30).setItemHeight(30).setFont(guiBaseFont).close().moveTo("default");
  cp5.addScrollableList("scrollableList_Isdlevel").addItems(Arrays.asList("1.0","0.5")).changeValue(ic.getVar(ic.isd)).setPosition(x,y+dy).setSize(len,250).setBarHeight(30).setItemHeight(30).setFont(guiBaseFont).close().moveTo("default");
  cp5.addScrollableList("scrollableList_MaskTime").addItems(Arrays.asList("None / 83","500 / 583","666 / 750","750 / 833")).changeValue(ic.getVar(ic.ocpmask)).setPosition(x+dx,y+dy).setSize(len,250).setBarHeight(30).setItemHeight(30).setFont(guiBaseFont).close().moveTo("default");
  cp5.addScrollableList("scrollableList_OcpLevel").addItems(Arrays.asList("0.250V / 10x","0.125V / 20x")).changeValue(ic.getVar(ic.ocp_lvl)).setPosition(x,y).setSize(len,250).setBarHeight(30).setItemHeight(30).setFont(guiBaseFont).close().moveTo("default");
  cp5.addScrollableList("scrollableList_RsaFilter").addItems(Arrays.asList("No Filter","fc=200","fc=100","fc=50")).changeValue(ic.getVar(ic.rs_sel)).setPosition(x+dx,y).setSize(len,250).setBarHeight(30).setItemHeight(30).setFont(guiBaseFont).close().moveTo("default");
////ScrollableList in tab2 (Motor Control)*************************
  cp5.addScrollableList("scrollableList_PolePairs").addItems(Arrays.asList("2","4","6","8","10","12","14","16")).changeValue(ic.getVar(ic.polepair)).setPosition(230,50).setSize(65,250).setBarHeight(30).setItemHeight(30).setFont(guiBaseFont).close().moveTo("tab2");
  cp5.addScrollableList("scrollableList_MaxSpeed").addItems(Arrays.asList("4096","8192","16384","32768")).changeValue(ic.getVar(ic.maxspeed)).setPosition(480,50).setSize(65,150).setBarHeight(30).setItemHeight(30).setFont(guiBaseFont).close().moveTo("tab2");
  cp5.addScrollableList("scrollableList_FGSEL").addItems(Arrays.asList("1 ppr")).changeValue(ic.getVar(ic.fgsel)).setPosition(330,595).setSize(90,90).setBarHeight(30).setItemHeight(30).setFont(guiBaseFont).close().moveTo("tab2");
  cp5.addScrollableList("scrollableList_RPM_Limit").addItems(Arrays.asList("No Limit","512","2200","3800","5400","7000","8600","10240")).changeValue(ic.getVar(ic.rpmlimit)).setPosition(600,595).setSize(90,100).setBarHeight(30).setItemHeight(30).setFont(guiBaseFont).close().moveTo("tab2");
}

void guiStructureInitialization(){  //Creates GUI strutrue and creates I2C address selector button
  // Adding Tabs 
  // "global" is the always visible area
  cp5.getTab("default").setLabel("Configuration Control").setWidth(220).setHeight(tabHeight).setColorActive(tabActiveColor); // Tab setting   (Configuration Control)
  cp5.getTab("default").getCaptionLabel().setFont(guiTabFont).align(CENTER, CENTER);  // Tab FONT
  cp5.addTab("tab1").setLabel("Sequence Control").setWidth(180).setHeight(tabHeight).setColorActive(tabActiveColor); // Tab setting   (Sequence Control)
  cp5.getTab("tab1").getCaptionLabel().setFont(guiTabFont).align(CENTER, CENTER);  // Tab FONT
  cp5.addTab("tab2").setLabel("Motor Control").setWidth(160).setHeight(tabHeight).setColorActive(tabActiveColor); // Tab setting   (Motor Control)
  cp5.getTab("tab2").getCaptionLabel().setFont(guiTabFont).align(CENTER, CENTER);  // Tab FONT
  cp5.addTab("tab3").setLabel("Registers").setWidth(130).setHeight(tabHeight).setColorActive(tabActiveColor); // Tab setting   (Registers)
  cp5.getTab("tab3").getCaptionLabel().setFont(guiTabFont).align(CENTER, CENTER);  // Tab FONT
  cp5.addTab("tab4").setLabel("NVM").setWidth(80).setHeight(tabHeight).setColorActive(tabActiveColor); // Tab setting   (NVM)
  cp5.getTab("tab4").getCaptionLabel().setFont(guiTabFont).align(CENTER, CENTER);  // Tab FONT
  // Changing active tab at start
  cp5.getTab("default").setActive(false);
  cp5.getTab("tab2").setActive(true);
  
  // Labels with tooltip display, ID of following should match it's index, tooltip text is set in txtlabelTooltipText[] array.
  txtlabel_tabGlobal[0] = cp5.addTextlabel("label_mcdStatus").setId(0).setText("MCD STATUS").setPosition(817,407).setFont(guiBaseFont).setColor(labelColor).moveTo("global");  //Not required so change this// ID should match index
  int x=835;
  int y=435;
  txtlabel_tabGlobal[1] = cp5.addTextlabel("label_cp_low").setId(1).setText("CP_LOW").setPosition(x,y).setFont(guiBaseFont).setColor(labelColor).moveTo("global");
  txtlabel_tabGlobal[2] = cp5.addTextlabel("label_tsd").setId(2).setText("TSD")      .setPosition(x+119,y).setFont(guiBaseFont).setColor(labelColor).moveTo("global");
  txtlabel_tabGlobal[3] = cp5.addTextlabel("label_isd").setId(3).setText("ISD")      .setPosition(x+238,y).setFont(guiBaseFont).setColor(labelColor).moveTo("global");
  txtlabel_tabGlobal[4] = cp5.addTextlabel("label_ov_spd").setId(4).setText("OV_SPD").setPosition(x,y+30).setFont(guiBaseFont).setColor(labelColor).moveTo("global");
  txtlabel_tabGlobal[5] = cp5.addTextlabel("label_ud_spd").setId(5).setText("UD_SPD").setPosition(x+119,y+30).setFont(guiBaseFont).setColor(labelColor).moveTo("global");
  txtlabel_tabGlobal[6] = cp5.addTextlabel("label_st_fail").setId(6).setText("ST_FAIL").setPosition(x+238,y+30).setFont(guiBaseFont).setColor(labelColor).moveTo("global");
  txtlabel_tabGlobal[7] = cp5.addTextlabel("label_hz_count").setId(7).setText("HZ_CNT").setPosition(835,522).setFont(guiBaseFont).setColor(labelColor).moveTo("global");
  
  // Adding Dropdown (ScrollableList)  ====================================================================================
  cp5.addScrollableList("scrollableList_I2CAddressSelect").setFont(guiBaseFont).setPosition(891,356).setSize(65,200).setBarHeight(30).setItemHeight(30).addItems(Arrays.asList("Reg","0x29","0x2D","0x32")).changeValue(1).close().moveTo("global");
}

void updateControllerValuesFromRegData(){
// Slider  ========================================================================================================
////Slider in tab default (Configuration Control)*************************
  cp5.getController("slider_SourceCurrent").changeValue(ic.getVar(ic.source));
  cp5.getController("slider_SinkCurrent").changeValue(ic.getVar(ic.sink));
  cp5.getController("slider_DutyChangeLimit").changeValue(ic.getVar(ic.dutychglimit));
  cp5.getController("slider_DutyChangeTime").changeValue(ic.getVar(ic.duty_up_time));
  cp5.getController("slider_DeadTime").changeValue(ic.getVar(ic.deadtime));
  cp5.getController("slider_RestartTime").changeValue(ic.getVar(ic.tre));
////Slider in tab1 (Sequence Control)*************************
  cp5.getController("slider_FirstDcExcitation").changeValue(ic.getVar(ic.pretip));
  cp5.getController("slider_SecondDcExcitation").changeValue(ic.getVar(ic.tip));
  cp5.getController("slider_ForcedFreq").changeValue(ic.getVar(ic.fst));
  cp5.getController("slider_CompHys").changeValue(ic.getVar(ic.comp_hys));
  cp5.getController("slider_StartCurrent").changeValue(ic.getVar(ic.startcurrent));
  cp5.getController("slider_TargetCurrent").changeValue(ic.getVar(ic.ss_add_sel));
  cp5.getController("slider_CurrentIncr").changeValue(ic.getVar(ic.ss_up_sel));
  cp5.getController("slider_SoftStartDuty").changeValue(ic.getVar(ic.ss_dutychglimit));
  cp5.getController("slider_WaitTime").changeValue(ic.getVar(ic.wait_time));
////Slider in tab2 (Motor Control)*************************
  float _startDuty=float(ic.getVar(ic.startduty))*100/512;
  cp5.getController("slider_StartDuty").changeValue(_startDuty);
  float _stopDuty=float(ic.getVar(ic.stopduty))*100/256;
  cp5.getController("slider_StopDuty").changeValue(_stopDuty);
  float _changeDuty=float(ic.getVar(ic.changeduty))*100/256;
  cp5.getController("slider_ChangeDuty").changeValue(_changeDuty);
  float _maxDuty=float(ic.getVar(ic.maxduty)+257)*100/512;
  cp5.getController("slider_MaxDuty").changeValue(_maxDuty);
  float _startRpmOL=float(ic.getVar(ic.startrpm))/16*100/512;
  cp5.getController("slider_MinOutputOL").changeValue(_startRpmOL);
  float _startRpmCL=ic.getVar(ic.startrpm);
  cp5.getController("slider_MinOutputCL").changeValue(_startRpmCL);
  float _slopeOL = float(ic.getVar(ic.speedslop))/1024;
  float _slopeCL = float(ic.getVar(ic.speedslop))*8/100;
  float _slope2OL = float(ic.getVar(ic.speedslop2))/1024;
  float _slope2CL = float(ic.getVar(ic.speedslop2))*8/100;
  float _changeOutputOL = _startRpmOL + _slopeOL * (_changeDuty-_startDuty);
  cp5.getController("slider_ChangeOutputOL").changeValue(_changeOutputOL);
  float _changeOutputCL = _startRpmCL + _slopeCL * (_changeDuty-_startDuty);
  cp5.getController("slider_ChangeOutputCL").changeValue(_changeOutputCL);
  float _maxOutputOL, _maxOutputCL;
  if(ic.getVar(ic.changeduty)==0){ //Two point profile
    _maxOutputOL = _startRpmOL + _slopeOL * (_maxDuty-_startDuty);
    _maxOutputCL = _startRpmCL + _slopeCL * (_maxDuty-_startDuty);
  }else{ //Three point profile
    _maxOutputOL = _changeOutputOL + _slope2OL * (_maxDuty-_changeDuty);
    _maxOutputCL = _changeOutputCL + _slope2CL * (_maxDuty-_changeDuty);
  }
  cp5.getController("slider_MaxOutputOL").changeValue(_maxOutputOL);
  cp5.getController("slider_MaxOutputCL").changeValue(_maxOutputCL);
  cp5.getController("slider_MaxDutyHysteresis").changeValue((ic.getVar(ic.maxdutyhys)+1)*0.4);
  cp5.getController("slider_KP").changeValue(ic.getVar(ic.kp));
  cp5.getController("slider_KI").changeValue(ic.getVar(ic.ki));
  cp5.getController("slider_SPD").changeValue(float(ic.getVar(ic.spd))*100/512);
// Toggle ==========================================================================================================
////Toggle in tab default (Configuration Control)*************************
  cp5.getController("toggle_SpeedControl").changeValue(ic.getVar(ic.tspsel));
  cp5.getController("toggle_DirControl").changeValue(ic.getVar(ic.dir));
  cp5.getController("toggle_Brake").changeValue(ic.getVar(ic.brk_inv));
  cp5.getController("toggle_SpeedControlInverse").changeValue(ic.getVar(ic.spdinv));
  cp5.getController("toggle_DisableAntiTrought").changeValue(ic.getVar(ic.antithrough));
  cp5.getController("toggle_AlertInverse").changeValue(ic.getVar(ic.alertinv));
  cp5.getController("toggle_OcpMask").changeValue(ic.getVar(ic.ocpdis));
  cp5.getController("toggle_IsdMask").changeValue(ic.getVar(ic.isd_mask));
  cp5.getController("toggle_TsdMask").changeValue(ic.getVar(ic.tsd_mask));
  cp5.getController("toggle_LockMask").changeValue(ic.getVar(ic.lockdis));
  cp5.getController("toggle_VcpMask").changeValue(ic.getVar(ic.vcp_mask));
  cp5.getController("toggle_RecoveryMode").changeValue(ic.getVar(ic.latch));
////Toggle in tab1 (Sequence Control)*************************
  cp5.getController("toggle_StandbyMode").changeValue(ic.getVar(ic.stby_mode));
  cp5.getController("toggle_WaitMode").changeValue(ic.getVar(ic.wait_mode));
  cp5.getController("toggle_AfterWait").changeValue(ic.getVar(ic.wait_con));
////Toggle in tab2 (Motor Control)*************************
  cp5.getController("toggle_OpenCloseLoop").changeValue(ic.getVar(ic.openloop));
  cp5.getController("toggle_NoStop").changeValue(ic.getVar(ic.nostop));
  cp5.getController("toggle_MaxOff").changeValue(ic.getVar(ic.maxoff));
  cp5.getController("toggle_MaxOpen").changeValue(ic.getVar(ic.maxopen));
  cp5.getController("toggle_ChangePoint").changeValue(((ic.getVar(ic.changeduty))==0)?0:1);
  cp5.getController("toggle_KPX").changeValue(ic.getVar(ic.kpx));
  cp5.getController("toggle_KIX").changeValue(ic.getVar(ic.kix));
  cp5.getController("toggle_FG_ON").changeValue(ic.getVar(ic.fg_on));
  
// ScrollableList (Dropdown)  ========================================
////ScrollableList in tab default (Configuration Control)*************
  cp5.getController("scrollableList_LeadAngle").changeValue(ic.getVar(ic.la));
  cp5.getController("scrollableList_Fpwm").changeValue(ic.getVar(ic.fpwm));
  cp5.getController("scrollableList_Fmax").changeValue(ic.getVar(ic.fmax));
  cp5.getController("scrollableList_LockStatus").changeValue(ic.getVar(ic.lock_brk));
  cp5.getController("scrollableList_Isdlevel").changeValue(ic.getVar(ic.isd));
  cp5.getController("scrollableList_MaskTime").changeValue(ic.getVar(ic.ocpmask));
  cp5.getController("scrollableList_OcpLevel").changeValue(ic.getVar(ic.ocp_lvl));
  cp5.getController("scrollableList_RsaFilter").changeValue(ic.getVar(ic.rs_sel));
////ScrollableList in tab2 (Motor Control)****************************
  cp5.getController("scrollableList_PolePairs").changeValue(ic.getVar(ic.polepair));
  cp5.getController("scrollableList_MaxSpeed").changeValue(ic.getVar(ic.maxspeed));
  cp5.getController("scrollableList_FGSEL").changeValue(ic.getVar(ic.fgsel));
  cp5.getController("scrollableList_RPM_Limit").changeValue(ic.getVar(ic.rpmlimit));
}

void readMcdAndUpdateGui(){ //reads all registers from MCD and updates registers in GUI and updates all controllers of GUI accordingly
  int data;
  //Read all registers to update GUI
  for(int reg=0; reg<=30; reg++){
    data=cmdReadByte(reg);
    delay(50);
    ic.setReg(reg,data);
  }
  //updating all controllers
  updateControllerValuesFromRegData(); //Updates GUI controllers according to loaded data
  //Updating the text in Register Tab
  textArea_registerList.setText(ic.printRegAllInString());
  textArea_variableList.setText(ic.printVarAllInString());
}

void tooltipInitialization(){
  //Tooltip setup
  tt = new TooltipCanvas(); //create a tooltipCanvas
  tt.post();          // Displays tooltip on top
  cp5.addCanvas(tt);  //adds tooltip canvas to main form
}

void readAllICRegisters(){
  int result;
  println("Reading all registers..");
  for(int reg=0; reg<=30; reg++){
    result= cmdReadByte(reg);
    if(statusI2C==0){
      //println("__ReadData: 0x" + hex(result,2));
      ic.setReg(reg,result);
    }else{
      println("__I2C Failure");
    }
    delay(50); //delay between read cycles
  }
}

// ********************************** I2C address selection at Startup *************************************************
Boolean I2CAddressSelection(){
  String address = "0x29"; //I2C address in hex. (0x29 is the default value)
  boolean validAddress = false;
  //Test Defalut I2C address (0x29)
  cmdSetId(0x29);  //Sets defalut I2C Address 0x29 in USB-I2C Controller
  cmdReadByte(0x00); //Reading register1 of MCD i.e. USERID for testing the I2C connection
   if(statusI2C==0){
     println("I2CAddressSelection(): SUCCESS (MCD detected at defalut I2C address 0x29)");
     validAddress = true;
   }else{
     println("I2CAddressSelection(): ERROR (MCD not detected at defalut I2C address 0x29, select correct I2C address of MCD)");
   }
  //Ask user for the I2C address
  if(validAddress==false){
   println("Default I2C Connection Failed");
    //Check if MCD is not powered ON and STBY=H (if No power or STBY=L in MCD then No Response will come for cmdSetId(0x29))
    cmdSetId(0x29);  //Sets defalut I2C Address 0x29 in USB-I2C Controller
    println("statusConverter: "+statusConverter);
    if(statusConverter==0){   //MCD is powered ON and STBY=H (0:Normal_Operation) 
          while(validAddress==false){
                String[] addressOptions = {"Reg","0x29","0x2D","0x32"};
                address = displayMessageBoxSelection("Selct MCD I2C address","Select I2C address of MCD", addressOptions); // title, label, choices
                if(address!=null){
                  try{
                      if(address!="Reg"){ //I2C address is one of the predefined address
                          //Setting GUI based on selection
                          switch(address){
                            case "0x29":
                              cmdSetId(0x29);  //Sets I2C Address in USB-I2C Controller
                              cmdReadByte(0x00); //Reading register1 of MCD i.e. USERID for testing the I2C connection
                              if(statusI2C==0){
                                println("I2CAddressSelection(): SUCCESS (MCD detected at selected I2C address 0x29)");                  
                                cp5.getController("scrollableList_I2CAddressSelect").changeValue(1); //Set controller to "0x29"
                                validAddress = true;
                              }else{
                                println("I2CAddressSelection(): ERROR (MCD not detected at selected I2C address 0x29, select correct I2C address of MCD)");
                              }
                              break;
                            case "0x2D":
                              cmdSetId(0x2D);  //Sets I2C Address in USB-I2C Controller
                              cmdReadByte(0x00); //Reading register1 of MCD i.e. USERID for testing the I2C connection
                              if(statusI2C==0){
                                println("I2CAddressSelection(): SUCCESS (MCD detected at selected I2C address 0x2D)");                  
                                cp5.getController("scrollableList_I2CAddressSelect").changeValue(2); //Set controller to "0x29"
                                validAddress = true;
                              }else{
                                println("I2CAddressSelection(): ERROR (MCD not detected at selected I2C address 0x2D, select correct I2C address of MCD)");
                              }
                              break;
                            case "0x32":
                              cmdSetId(0x32);  //Sets I2C Address in USB-I2C Controller
                              cmdReadByte(0x00); //Reading register1 of MCD i.e. USERID for testing the I2C connection
                              if(statusI2C==0){
                                println("I2CAddressSelection(): SUCCESS (MCD detected at selected I2C address 0x32)");                  
                                cp5.getController("scrollableList_I2CAddressSelect").changeValue(3); //Set controller to "0x29"
                                validAddress = true;
                              }else{
                                println("I2CAddressSelection(): ERROR (MCD not detected at selected I2C address 0x32, select correct I2C address of MCD)");
                              }
                              break;
                          }
                      }else{ //If Reg is selected then ask used to enter custom I2C address of MCD
                            String i2cAddStr = displayMessageBoxInputString("I2C Address", "Enter MCD I2C address (00 to 7F)");
                              if(i2cAddStr!=null){
                                  try{
                                      if(i2cAddStr.length()==2){
                                        char l = i2cAddStr.charAt(0); //left char
                                        char r = i2cAddStr.charAt(1); //right char
                                        println("l:"+l+"    r:"+r);
                                        if((l>='0' && l<='7') && ((r>='0' && r<='9')||(r>='a' && r<='f')||(r>='A' && r<='F'))){
                                            cmdSetId(unhex(i2cAddStr));  //Sets I2C Address in USB-I2C Controller
                                            cmdReadByte(0x00); //Reading register1 of MCD i.e. USERID for testing the I2C connection
                                            if(statusI2C==0){
                                              println("I2CAddressSelection(): SUCCESS (MCD detected at selected I2C address "+i2cAddStr+")");                  
                                              cp5.getController("scrollableList_I2CAddressSelect").changeValue(0); //Set controller to "0x29"
                                              textI2CRegAddress="0x"+i2cAddStr;
                                              validAddress = true;
                                            }else{
                                              println("I2CAddressSelection(): ERROR (MCD not detected at selected I2C address "+i2cAddStr+", select correct I2C address of MCD)");
                                            }
                                        }else{
                                          println("Error: Entered I2C address is invalid: "+i2cAddStr);
                                          displayMessageBoxError("ERROR: Invalid I2C Address","Enter a valid I2C address (00 to 7F)");
                                        }
                                      }else{
                                        println("Error: Entered I2C address is invalid: "+i2cAddStr);
                                        displayMessageBoxError("ERROR: Invalid I2C Address","Enter a valid I2C address (00 to 7F)");
                                      }
                                  }catch(Exception e){
                                      println("Error: Exception:"+e);
                                      displayMessageBoxError("ERROR: Invalid I2C Address","Enter a valid I2C address (00 to 7F)");
                                  }
                              }else{ // If the input box is closed.
                                  println("Error: I2C address is not entered.");
                                  displayMessageBoxError("ERROR: Invalid I2C Address","Enter a valid I2C address (00 to 7F)");
                              }
                            }
                        }catch(Exception e){
                          println("Error: Exception:"+ e);
                        }
                }else{ //If the I2C address selection box is closed
                  displayMessageBoxError("Error: I2C address is not selected", "Select a valid I2C address"); // String title, String label
                }
          } //End of while loop for connecting to I2C
    }else{  //MCD is not powered ON or STBY=L (No_Response_From_USB-I2C_Converter)
      println("MCD not powered ON or STBY=L");
      displayMessageBoxError("ERROR: MCD Power Supply Problem or STBY=L","Check Power Supply of MCD and STBY signal. Then Restart GUI.");
      exit();
    }
  }
  return validAddress;  //return true:Success (valid address set), false:Connection Error
}


//*********************************** Draw Functions ******************************************
void drawI2CandComPortSection(){     //Draws Non-Tooltip items of I2C & COM port section
//Labels without tooltip in global area 
//I2C_Address
  //I2C Address Label
  fill(labelColor);
  textSize(18);
  text("I2C_Add.                      "+textI2CRegAddress,815,375); //I2C reg Address display //space is to accomodate Dropdown button
  //I2C Status Box
  stroke(colorDarkGrey);
  fill((statusI2C==0)?(colorGreen):(colorRed));
  rect(1164,361,22,22);
  //I2C Status Text
  fill(labelColor);
  text((statusI2C==0)?("I2C"):("I2C    "+statusI2C),1134,378);
//COM port
  fill(220);
  rect(891,320,64,28);   // Rectangle behind COM port
  fill(labelColor);
  text("COM_Port  "+textComPort,811,340); //I2C reg Address display
  //Converter Status Box
  stroke(colorDarkGrey);
  fill((statusConverter==0)?(colorGreen):(colorRed));
  rect(1164,324,22,22);
  //Converter Status Text
  fill(labelColor);
  text("Converter",1084,342);
}

void drawMcdStatusSection(){  //Draws Non-Tooltip items of MCD Status section
//MCD Status
  //Background
  fill(250); //backgound same as graph background
  noStroke();
  rect(812,400,380,160);
  //Status Boxes
  stroke(colorDarkGrey);
  int x,y;
  x=910;
  y=432;
  fill((statusI2C==0)?((ic.getVar(ic.cp_low)==0)?(colorGreen):(colorRed)):(colorYellow));  //CP_LOW
  rect(x,y,22,22);  //Status Box
  fill((statusI2C==0)?((ic.getVar(ic.tsd)==0)?(colorGreen):(colorRed)):(colorYellow));  //TSD
  rect(x+116,y,22,22);
  fill((statusI2C==0)?((ic.getVar(ic.isd)==0)?(colorGreen):(colorRed)):(colorYellow));  //ISD
  rect(x+232,y,22,22);
  fill((statusI2C==0)?((ic.getVar(ic.ov_spd)==0)?(colorGreen):(colorRed)):(colorYellow));  //OV_SPD
  rect(x,y+32,22,22);
  fill((statusI2C==0)?((ic.getVar(ic.ud_spd)==0)?(colorGreen):(colorRed)):(colorYellow));  //UD_SPD
  rect(x+116,y+32,22,22);
  fill((statusI2C==0)?((ic.getVar(ic.st_fail)==0)?(colorGreen):(colorRed)):(colorYellow));  //ST_FAIL
  rect(x+232,y+32,22,22);
  //Speed Title
  fill(labelColor);
  text("SPEED",820,515);
//Speed
  //HZ_COUNT & HZ Box
  fill(220);
  rect(910,517,65,28);   // Rectangle behind HZ_COUNT display
  rect(1035,517,65,28);   // Rectangle behind HZ display
  fill(labelColor);
  textSize(19);
  textAlign(RIGHT);
  int temp =ic.getVar(ic.hz_cnt); 
  text(temp,966,537);  //Hz_Count Display
  if(temp==0){
    text("-",1096,537); //Hz Display
  }else{
    text(250000/temp,1096,537); //Hz Display
  }
  textAlign(LEFT);
  text("Hz",1110,537);
  textSize(30);
  text(">>",992,542);
  textSize(18);
}

void pollingMCD(){  //Reads status and speed registers 0,29,30 of MCD every 1 second
  if(millis()-timer>1000){
    if(statusI2C==0 && statusConverter==0){ //Polling occurs if only both I2C & USB-I2Cconverter connections are OK
      int temp=0;
      displyI2CMessages=0; //Stop displaying I2C messages. (To prevent polling messages from filling up the console log.)
      temp=cmdReadByte(0); //reading status register
      delay(50);
      if(statusI2C==0 && statusConverter==0){
        ic.setReg(0,temp);
        temp=cmdReadByte(29); //reading hz_count register 29
        delay(50);
        if(statusI2C==0 && statusConverter==0){
          ic.setReg(29,temp);
          temp=cmdReadByte(30); //reading hz_count register 30
          delay(50);
          if(statusI2C==0 && statusConverter==0){
            ic.setReg(30,temp);
          }
        }
      }
      displyI2CMessages=1; //Start displaying I2C messages
      //Updating register Data in Register Tab
      textArea_registerList.setText(ic.printRegAllInString());
      textArea_variableList.setText(ic.printVarAllInString());
    }
    timer=millis();
  }
}



void updateSlopeIfRequired(){
  if(sendSlopeRegDataFlag==true){
    sendSlopeRegDataFlag=false;
    sendSlopeData();
  }
}

void displayValidControllers(){
  if(openLoopFlag==true){
    cp5.getController("slider_MinOutputOL").setVisible(true);
    cp5.getController("slider_MaxOutputOL").setVisible(true);
    if(threePointProfile==true){
      cp5.getController("slider_ChangeOutputOL").setVisible(true);
      cp5.getController("label_UA_changePoint2").setVisible(false);
      cp5.getController("slider_ChangeDuty").setVisible(true);
      cp5.getController("label_UA_changePoint1").setVisible(false);
    }else{
      cp5.getController("slider_ChangeOutputOL").setVisible(false);
      cp5.getController("label_UA_changePoint2").setVisible(true);
      cp5.getController("slider_ChangeDuty").setVisible(false);
      cp5.getController("label_UA_changePoint1").setVisible(true);
    }
    cp5.getController("slider_MinOutputCL").setVisible(false);
    cp5.getController("slider_MaxOutputCL").setVisible(false);
    cp5.getController("slider_ChangeOutputCL").setVisible(false);
  }else{
    cp5.getController("slider_MinOutputOL").setVisible(false);
    cp5.getController("slider_MaxOutputOL").setVisible(false);
    cp5.getController("slider_ChangeOutputOL").setVisible(false);
    cp5.getController("slider_MinOutputCL").setVisible(true);
    cp5.getController("slider_MaxOutputCL").setVisible(true);
    if(threePointProfile==true){
      cp5.getController("slider_ChangeOutputCL").setVisible(true);
      cp5.getController("label_UA_changePoint2").setVisible(false);
      cp5.getController("slider_ChangeDuty").setVisible(true);
      cp5.getController("label_UA_changePoint1").setVisible(false);
    }else{
      cp5.getController("slider_ChangeOutputCL").setVisible(false);
      cp5.getController("label_UA_changePoint2").setVisible(true);
      cp5.getController("slider_ChangeDuty").setVisible(false);
      cp5.getController("label_UA_changePoint1").setVisible(true);
    }
  }  
}


void updateSliderDisplayLabel(){
    cp5.getController("label_SourceCurrentDisplay").setValueLabel(sourceCurrentData[int(cp5.getController("slider_SourceCurrent").getValue())]);
    cp5.getController("label_SinkCurrentDisplay").setValueLabel(sinkCurrentData[int(cp5.getController("slider_SinkCurrent").getValue())]);
    cp5.getController("label_DutyChnageLimitDisplay").setValueLabel(dutyChangeLimitData[int(cp5.getController("slider_DutyChangeLimit").getValue())]);
    cp5.getController("label_DutyChangeTimeDisplay").setValueLabel(dutyChangeTimeData[int(cp5.getController("slider_DutyChangeTime").getValue())]);
    cp5.getController("label_DeadTimeDisplay").setValueLabel(deadTimeData[int(cp5.getController("slider_DeadTime").getValue())]);
    cp5.getController("label_RestartTimeDisplay").setValueLabel(restartTimeData[int(cp5.getController("slider_RestartTime").getValue())]);
    cp5.getController("label_FirstDcExcitationDisplay").setValueLabel(firstDcExcitationData[int(cp5.getController("slider_FirstDcExcitation").getValue())]);
    cp5.getController("label_SecondDcExcitationDisplay").setValueLabel(secondDcExcitationData[int(cp5.getController("slider_SecondDcExcitation").getValue())]);
    cp5.getController("label_ForcedFreqDisplay").setValueLabel(forcedFreqData[int(cp5.getController("slider_ForcedFreq").getValue())]);
    cp5.getController("label_CompHysDisplay").setValueLabel(compHysData[int(cp5.getController("slider_CompHys").getValue())]);
    cp5.getController("label_StartCurrentDisplay").setValueLabel(startCurrentData[int(cp5.getController("slider_StartCurrent").getValue())]);
    cp5.getController("label_TargetCurrentDisplay").setValueLabel(targetCurrentData[int(cp5.getController("slider_TargetCurrent").getValue())]);
    cp5.getController("label_CurrentIncrDisplay").setValueLabel(currentIncrData[int(cp5.getController("slider_CurrentIncr").getValue())]);
    cp5.getController("label_SoftStartDutyDisplay").setValueLabel(sStartDutyData[int(cp5.getController("slider_SoftStartDuty").getValue())]);
    cp5.getController("label_WaitTimeDisplay").setValueLabel(waitTimeData[int(cp5.getController("slider_WaitTime").getValue())]);
}

void invalidSettingErrorDisplay(){
  float start =cp5.getController("slider_StartDuty").getValue();
  float stop =cp5.getController("slider_StopDuty").getValue();
  
  //INVALID START STOP SETTING
  if(start<stop){
    if(startStopErrorDisplayed==false){
    cp5.getController("slider_StartDuty").setMousePressed(false);
      cp5.getController("slider_StartDuty").setMouseOver(false);
    cp5.getController("slider_StopDuty").setMousePressed(false);
      cp5.getController("slider_StopDuty").setMouseOver(false);
    displayMessageBoxError("Invalid Setting","Start Duty should be greater than Stop Duty.");
    startStopErrorDisplayed=true;
    }
  }else{
    startStopErrorDisplayed=false;
  }
  int maxSlope=16; //16 in Openloop    1280 in CloseLoop
  if(openLoopFlag==false){ //CloseLoop
    maxSlope=1280;
  }
  //INVALID SLOPE SETTING
  if(slopeData<0 || slopeData>maxSlope){ //Invalid Setting
    if(slopeErrorDisplayed==false){
      cp5.getController("slider_StartDuty").setMousePressed(false);
      cp5.getController("slider_StartDuty").setMouseOver(false);
      cp5.getController("slider_MinOutputOL").setMousePressed(false);
      cp5.getController("slider_MinOutputOL").setMouseOver(false);
      cp5.getController("slider_MinOutputCL").setMousePressed(false);
      cp5.getController("slider_MinOutputCL").setMouseOver(false);
      cp5.getController("slider_MaxDuty").setMousePressed(false);
      cp5.getController("slider_MaxDuty").setMouseOver(false);
      cp5.getController("slider_MaxOutputOL").setMousePressed(false);
      cp5.getController("slider_MaxOutputOL").setMouseOver(false);
      cp5.getController("slider_MaxOutputCL").setMousePressed(false);
      cp5.getController("slider_MaxOutputCL").setMouseOver(false);
      cp5.getController("slider_ChangeDuty").setMousePressed(false);
      cp5.getController("slider_ChangeDuty").setMouseOver(false);
      cp5.getController("slider_ChangeOutputOL").setMousePressed(false);
      cp5.getController("slider_ChangeOutputOL").setMouseOver(false);
      cp5.getController("slider_ChangeOutputCL").setMousePressed(false);
      cp5.getController("slider_ChangeOutputCL").setMouseOver(false);
      cp5.getController("toggle_ChangePoint").setMousePressed(false);
      cp5.getController("toggle_ChangePoint").setMouseOver(false);
      displayMessageBoxError("Invalid Setting","Slope should be between 0 & "+maxSlope);
      slopeErrorDisplayed=true;
    }
  }else{
    slopeErrorDisplayed=false;
  }
  
  //INVALID SLOPE2 SETTING
  if(slope2Data<0 || slope2Data>maxSlope){ //Invalid Setting
    if(slope2ErrorDisplayed==false){
      cp5.getController("slider_StartDuty").setMousePressed(false);
      cp5.getController("slider_StartDuty").setMouseOver(false);
      cp5.getController("slider_MinOutputOL").setMousePressed(false);
      cp5.getController("slider_MinOutputOL").setMouseOver(false);
      cp5.getController("slider_MinOutputCL").setMousePressed(false);
      cp5.getController("slider_MinOutputCL").setMouseOver(false);
      cp5.getController("slider_MaxDuty").setMousePressed(false);
      cp5.getController("slider_MaxDuty").setMouseOver(false);
      cp5.getController("slider_MaxOutputOL").setMousePressed(false);
      cp5.getController("slider_MaxOutputOL").setMouseOver(false);
      cp5.getController("slider_MaxOutputCL").setMousePressed(false);
      cp5.getController("slider_MaxOutputCL").setMouseOver(false);
      cp5.getController("slider_ChangeDuty").setMousePressed(false);
      cp5.getController("slider_ChangeDuty").setMouseOver(false);
      cp5.getController("slider_ChangeOutputOL").setMousePressed(false);
      cp5.getController("slider_ChangeOutputOL").setMouseOver(false);
      cp5.getController("slider_ChangeOutputCL").setMousePressed(false);
      cp5.getController("slider_ChangeOutputCL").setMouseOver(false);
      cp5.getController("toggle_ChangePoint").setMousePressed(false);
      cp5.getController("toggle_ChangePoint").setMouseOver(false);
      displayMessageBoxError("Invalid Setting","Slope2 should be between 0 & "+maxSlope);
      slope2ErrorDisplayed=true;
    }
  }else{
    slope2ErrorDisplayed=false;
  }
}

void controlEvent(ControlEvent theEvent) {
  println("_EventProcessed: "+theEvent.getName()+"\n");
  //Updating RegisterMap Tab data
  textArea_registerList.setText(ic.printRegAllInString());
  textArea_variableList.setText(ic.printVarAllInString());  
}




//*********************************** EXIT ***********************************************
void exit() {
   println("\nExiting program");
   serialPort.stop();
   //your Code above. 
   super.exit();
}
