/**
 ****************************************************************************
 * @file	usercon.c
 * @brief   User Control
 * @version V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2023
 *****************************************************************************
 */
#include <stdlib.h>
#include <stdio.h>
#include "ipdefine.h"
#include "HPFC_drv.h"
#include "mcuip_drv.h"
#include "D_Para.h"
#include "D_Driver.h"
#include "ipdrv_adc.h"
#include "ipdrv_cg.h"
#include "ipdrv_siwdt.h"
#if defined(__USE_microE_COMMUNICATION_FUNC)
#include "ipdrv_uart.h"
#endif /* __USE_microE_COMMUNICATION_FUNC */
#include "dac_drv.h"
#include "B_User.h"
#define DEFINE_APP
#include "usercon.h"
#undef  DEFINE_APP


/*===================================================================*
	  Macro Definition
 *===================================================================*/

/*===================================================================*
	  Local Constant Data Definition
 *===================================================================*/
/* Key settting */
#define P_SW1			 TSB_PG_DATA_PG3					/* SW1 (KEY1) */
#define P_SW2			 TSB_PE_DATA_PE7					/* SW2 (KEY2) */
#define P_SW3			 TSB_PA_DATA_PA0					/* SW3 (KEY3) */

#define cKEY_CHATA_CNT	  (20)								/* [cnt] chattering counter for KEY SW */

/* Soft ADC Setting */
#define cADUNIT_USR		TSB_ADA								/* User ad data ADCUnit */
#define cADCH_VR_FAN	ADC_AIN17							/* ADC Channel for VR */
#define cADREG_VR_FAN	ADC_REG4							/* Result register No for FAN VR */
#define cADCH_VR_COMP	ADC_AIN18							/* ADC Channel for VR */
#define cADREG_VR_COMP	ADC_REG5							/* Result register No for COMP VR */

#define cADAVECNT		   (10)								/* ADC average count */
#define cPUSHSW_CHATA_CNT   (5)								/* [cnt] Chattering counter for Push SW */

#if defined(__USE_microE_COMMUNICATION_FUNC)
#define c2000MS_TIMER			(2000)						/* [ms] (4kHz * 4) * 2000 */

/* User Setting */
#define	cCONTROL_SINGLE			(0)							/* UART control: single */
#define	cCONTROL_UART			(1)							/* UART control: uart */

/* UART Setting */
#define	UART_ch					UART0						/* UART Channel */
#define	INTERRUPT_TX			INTSC0TX_IRQn				/* UART Interrupt request */
#define	INTERRUPT_RX			INTSC0RX_IRQn				/* UART Interrupt request */

#define	cSEND_DATA_NUM			(7)							/* Send data size */
#define	cRECEIVE_DATA_NUM		(6)							/* Receive data size */

#define	cUART_RECEIVE_WAIT		(0x00)						/* UART mode : data receive wait */
#define	cUART_ERR				(0x01)						/* UART mode : error */

#define	cREQ_SYSTEM_START		(0x10)						/* System start request */
#define	cREQ_ROTATE_MOTOR		(0x11)						/* Target speed updata request */
#define	cREQ_CHANGE_MOTOR		(0x12)						/* Control Rpm request */
#define	cREQ_ON_OFF_PFC			(0x13)						/* Control PFC request */
#define	cREQ_ALL_STOP_MOTOR		(0x14)						/* Stop PFC request */
#define	cREQ_STATUS_CH			(0x15)						/* Change channel request */
#define	cREQ_STATUS_DAC			(0x16)						/* Chnage Dac mode request */
#define	cREQ_MOTOR_ENABLE		(0x80)						/* Operating status */
#define	cGET_STATE_EMG			(0x81)						/* Emergency status */
#define	cGET_STAGE				(0x82)						/* Main stage */
#define	cGET_CONTROL_CH			(0x83)						/* Control channel */
#define	cGET_CARRIER_FREQUENCY	(0x84)						/* Carrier frequency */
#define	cGET_MOTOR_SPEED_MIN	(0x85)						/* Minimum rotation speed */
#define	cGET_MOTOR_SPEED_MAX	(0x86)						/* Maximum rotation speed */
#define	cGET_DEAD_TIME			(0x87)						/* Dead time */
#define	cGET_GATE_ACTIVE		(0x88)						/* Gate active */
#define	cGET_POSITION_DITECT	(0x89)						/* Position detect */
#define	cGET_VDC_VOLTAGE		(0x8A)						/* VDC voltage */
#define	cGET_OUTROOM_TEMP		(0x8B)						/* outroom templeture */
#define	cGET_U_VOLTAGE			(0x8C)						/* U-phase voltage */
#define	cGET_V_VOLTAGE			(0x8D)						/* V-phase voltage */
#define	cGET_W_VOLTAGE			(0x8E)						/* W-phase voltage */
#define	cGET_DAC_DATA			(0x8F)						/* Dac data */
#define	cGET_INTERNAL_AMP		(0x90)						/* Internal amp */
#define	cGET_DIRECTION			(0x91)						/* Direction */
#define	cGET_MODULATION			(0x92)						/* Modulation */
#define	cGET_MOTOR_SPEED		(0x94)						/* motor rotation speed */
#define	cGET_OUTPIPE_TEMP		(0x95)						/* outpipe templeture */
#define	cGET_EXHAUST_TEMP		(0x96)						/* exhaust templeture */
#define	cGET_DIODE_TEMP			(0x97)						/* diode templeture */
#define	cGET_IGBT_TEMP			(0x98)						/* IGBT templeture */
#define	cGET_HVMOS_TEMP			(0x99)						/* HVMOS templeture */

#define	cEMG_STATE_EMG_H		(0x00)						/* emergency status : over detect(hard) */
#define	cEMG_STATE_EMG_S		(0x01)						/* emergency status : over detect(soft) */
#define	cEMG_STATE_EMG_I		(0x02)						/* emergency status : current detect error */
#define	cEMG_STATE_EMG_DC		(0x03)						/* emergency status : over vdc */

#define	cGATE_ACTIVE_H_H		(0)							/* Gate active : H/H */
#define	cGATE_ACTIVE_L_L		(1)							/* Gate active : L/L */
#define	cGATE_ACTIVE_H_L		(2)							/* Gate active : H/L */
#define	cGATE_ACTIVE_L_H		(3)							/* Gate active : L/H */

#define	cPOSITION_DETECT_3SHUNT	(0)							/* Position Ditect : 3shunt */
#define	cPOSITION_DETECT_1SHUNT	(1)							/* Position Ditect : 1shunt */

#define	cDAC_DATA_TMPREG0		(0x00)						/* Dac data : TMPREG0 */
#define	cDAC_DATA_TMPREG1		(0x01)						/* Dac data : TMPREG1 */
#define	cDAC_DATA_TMPREG2		(0x02)						/* Dac data : TMPREG2 */
#define	cDAC_DATA_THETAHALF		(0x03)						/* Dac data : theta.half[1] */
#define	cDAC_DATA_IDREF			(0x04)						/* Dac data : Id_ref */
#define	cDAC_DATA_ID			(0x05)						/* Dac data : Id */
#define	cDAC_DATA_IQREF			(0x06)						/* Dac data : Iq_ref */
#define	cDAC_DATA_IQ			(0x07)						/* Dac data : Iq */
#define	cDAC_DATA_OMEGACOMHALH	(0x08)						/* Dac data : omega_com.half[1] */
#define	cDAC_DATA_OMEGAHALF		(0x09)						/* Dac data : omega.half[1] */
#define	cDAC_DATA_OMEGADEV		(0x0A)						/* Dac data : omega_dev */

#define	cINTERNAL_AMP_NO		(0)							/* Internal amp : External */
#define	cINTERNAL_AMP_YES		(1)							/* Internal amp : Internal */

#define	cDIRECTION_CW			(0)							/* Direction : plus */
#define	cDIRECTION_CCW			(1)							/* Direction : minus */

#define	cMOTOR_CTRL_FAN			(0)							/* motor control : fan */
#define	cMOTOR_CTRL_COMP		(1)							/* motor control : compressor */
#define	cMOTOR_CTRL_PFC			(2)							/* motor control : pfc */
#endif	/* __USE_microE_COMMUNICATION_FUNC */

/*===================================================================*
	  Typedef Definition
 *===================================================================*/
typedef struct
{
	uint8_t		select;										/* Dac output select */
	uint8_t		motch;										/* Dac motor ch select */
	uint8_t		datsft0;									/* Dac data shift for data0 */
	uint8_t		datsft1;									/* Dac data shift for data1 */
	uint8_t		datsft2;									/* Dac data shift for data2 */
	uint8_t		datsft3;									/* Dac data shift for data3 */
} dac_t;


/*===================================================================*
	  Local Variable Definition
 *===================================================================*/
/* No, MotorCh, sft0, sft1, sft2, sft3 */
#if !defined(__CONTROL_MOTOR_COMP)
dac_t		dac = { 0,  1,	3,	0,	0,	0 };				/* Dac setting. */
#else
/* dac_t		dac = { 0,  0,	0,	0,	0,	0 }; */				/* Dac setting. */
dac_t		dac = { 3,  0,	2,	2,	3,	0 };				/* PFC on/offmF */
#endif

#if defined(__USE_microE_COMMUNICATION_FUNC)
static	uint8_t		receive_timer;							/* uart receive timer counter */

static	uint8_t		uart_mode = 0;							/* uart control mode */
static	uint8_t		sendbuffer[cSEND_DATA_NUM];				/* Send data buffer */
static	uint8_t		receivebuffer[cRECEIVE_DATA_NUM];		/* Receive data buffer */
static	uint8_t		receivedata[cRECEIVE_DATA_NUM];			/* Receive data */
static	uint8_t		txpnt;									/* Pointer of send data */
static	uint8_t		rxpnt;									/* Pointer of receive data */
static	uint8_t		request_motor = 0;						/* control motor */
#endif	/* __USE_microE_COMMUNICATION_FUNC */

/*===================================================================*
	  Local Proto Type Definition
 *===================================================================*/
static void init_soft_adc(void);

#if defined(__USE_microE_COMMUNICATION_FUNC)
static void send_data_set(void);
static void data_check(void);
#endif	/* __USE_microE_COMMUNICATION_FUNC */
/*===================================================================*
	Function
 *===================================================================*/
#if defined(__TYPE_HIGH_VOLT) && defined(__RELAY)
/*===================================================================*
	  Select VDC Check Channel
 *===================================================================*/
#if defined(__CONTROL_MOTOR_FAN)

#define	__VDC_CHECK_CH	(0)			/* VDC Ch0 */

#elif defined(__CONTROL_MOTOR_COMP)

#define	__VDC_CHECK_CH	(1)			/* VDC Ch1 */

#else
#error "__VDC_CHECK_CH define error."
#endif
/**
 *********************************************************************************************
 * @brief	   Initial setting of Relay Port.
 *
 * @return	  none
 *********************************************************************************************
 */
void init_Relay(void)
{
	CG_SetFsysSupplyPeriphA(PO_RELAY_CG_FSYS_PERIPH_PORT, ENABLE);
	PO_RELAY_INI;
	relay_state = OFF;
}
/**
 *********************************************************************************************
 * @brief	   Turn on Relay.
 *
 * @return	  none
 *********************************************************************************************
 */
void Relay_On(void)
{
	uint32_t len_cnt;
	uint32_t lim_cnt;

	/***** Clear WDT *****/
	SIWDT_WriteClearCode();

	PO_RELAY_OFF;
	relay_state = OFF;

	len_cnt = 0;
	lim_cnt = 0;
	while ((relay_state != ON) && !(lim_cnt > (cRELAY_LIM / cMAINLOOP_PRD)))
	{
		if (M_Main_Counter >= cMAINLOOP_CNT)
		{
			M_Main_Counter = 0;

			/***** Clear WDT *****/
			SIWDT_WriteClearCode();

			len_cnt++;
			lim_cnt++;

#if (__VDC_CHECK_CH == 0)
			if (Motor_fan.drv.Vdc > (q15_t)(FIX_15 * cV_RELAY_ON / cV_MAX_FAN))
#elif (__VDC_CHECK_CH == 1)
			if (Motor_comp.drv.Vdc > (q15_t)(FIX_15 * cV_RELAY_ON / cV_MAX_COMP))
#endif
			{
				if (len_cnt > (cRELAY_LEN / cMAINLOOP_PRD))
				{
					relay_state = ON;
				}
			}
			else
			{
				len_cnt = 0;
			}
		}
	}
	if (relay_state == ON)
	{
		PO_RELAY_ON;
		Hpfc.drv.state.flg.emg_comp = 0;
	}
	else
	{
		/* relay on error */
		Hpfc.drv.state.flg.emg_comp = 1;
	}
}

#endif /* defined(__TYPE_HIGH_VOLT) && defined(__RELAY) */

/**
 *********************************************************************************************
 * @brief	   init_user_control
 *
 * @return	  none
 *********************************************************************************************
 */
void	init_user_control(void)
{

	init_soft_adc();									/* Initial setting of soft adc */
}


/**
 *********************************************************************************************
 * @brief	   Initialize softconvert adc.
 *
 * @return	  none
 *********************************************************************************************
 */
static void init_soft_adc(void)
{
	CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_PORTL, ENABLE);

	/* VR(AINC0) PJ2 */
	ADC_SetSWTrg(cADUNIT_USR, cADREG_VR_FAN, TRG_ENABLE(cADCH_VR_FAN));
	ADC_SetSWTrg(cADUNIT_USR, cADREG_VR_COMP, TRG_ENABLE(cADCH_VR_COMP));

	ADC_Start(cADUNIT_USR, ADC_TRG_SINGLE);							 /* adc start */
}


/**
 *********************************************************************************************
 * @brief	   Initialize Test point
 *
 * @return	  none
 *********************************************************************************************
 */
void init_TP(void)
{
	uint8_t led1shift = cLED1_SHIFT;
	uint8_t led2shift = cLED2_SHIFT;
	uint8_t led3shift = cLED3_SHIFT;
	uint8_t led4shift = cLED4_SHIFT;

	/* port setting */
	CG_SetFsysSupplyPeriphA(LED_CH1, ENABLE);
	LED1_CH->CR |= (1 << led1shift);	/* PC2 */
	CG_SetFsysSupplyPeriphA(LED_CH2, ENABLE);
	LED2_CH->CR |= (1 <<  led2shift);	/* PC3 */
	CG_SetFsysSupplyPeriphA(LED_CH3, ENABLE);
	LED3_CH->CR |= (1 <<  led3shift);	/* PA2 */
	CG_SetFsysSupplyPeriphA(LED_CH4, ENABLE);
	LED4_CH->CR |= ((1 <<  led4shift));	/* PU5 */

#if defined(__HW_TMPM4KNA)
	CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_PORTC, ENABLE);

	TSB_PC->IE |= ((1 << 3U));			/* PC3--pin48 DEBug */
#endif

	LED1 = 1;
	LED2 = 1;
	LED3 = 1;
	LED4 = 1;
}



/**
 ********************************************************************************************
 * @brief	   Initialize NBD.
 *
 * @return	  none
 ********************************************************************************************
 */
#if defined(__NBD)
void init_NBD(void)
{
	/* Port */
	CG_SetFsysSupplyPeriphA(NBD_PORTA, ENABLE);
	CG_SetFsysSupplyPeriphA(NBD_PORTB, ENABLE);

	/* NBDDATA2,3 (PN0, PN1) */
	NBD_TSB_FR0 |= cNBD_FR0;	/* ((1 << 1U) | (1 << 0U)) */
	NBD_TSB_CR0 |= cNBD_CR0;	/* ((1 << 1U) | (1 << 0U)) */
	NBD_TSB_IE0 |= cNBD_IE0;	/* ((1 << 1U) | (1 << 0U)) */

	/* NBDDATA0,1, NBDCLK, NBDSYNC (PF4-7) */
	NBD_TSB_FR1 |= cNBD_FR1;	/* ((1 << 1U) | (1 << 0U)) */
	NBD_TSB_CR1 |= cNBD_CR1;	/* ((1 << 1U) | (1 << 0U)) */
	NBD_TSB_IE1 |= cNBD_IE1;	/* ((1 << 1U) | (1 << 0U)) */

	/* Initialize */
	TSB_NBD->CR1 = 0xf9;
	TSB_NBD->CR0 |= 0x01;
}
#endif

#if defined(__USE_microE_COMMUNICATION_FUNC)
/**
 *********************************************************************************************
 * @brief		Timer counter.
 *
 * @return		none
 *********************************************************************************************
 */
void timer_count(void)
{
	if(receive_timer > 0)
	{
		if(--receive_timer <= 0)
		{
			rxpnt = 0;
		}
	}

}

/**
 *********************************************************************************************
 * @brief		Initialize UART.
 *
 * @return		none
 *********************************************************************************************
 */
void init_uart(void)
{
	UART_InitTypeDef UART_DispInit;

	if(UART_GetStatusEnableFlag(UART_ch) == PROGRAMMABLE){

		CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_UART0, ENABLE);
		CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_PORTC, ENABLE);

		/*Setting UART port.*/
		TSB_PC->FR1 |= ((1 << 0) | (1 << 1U));
		TSB_PC->IE |= (1 << 1U);	/* PC1:RXD */
		TSB_PC->CR |= (1 << 0U);	/* PC0:TXD */

		UART_SWReset(UART_ch);

		/* initialize the UART struct */
		UART_DispInit.BaudRate = 9600U;
		UART_DispInit.DataBits = UART_DATA_BITS_8;
		UART_DispInit.DataInvert = UART_NO_DATA_INVERT;
		UART_DispInit.TransferDirection = UART_LSB;	
		UART_DispInit.StopBits = UART_STOP_BITS_1;
		UART_DispInit.Parity = UART_NO_PARITY;
		UART_DispInit.Mode = (UART_ENABLE_TX | UART_ENABLE_RX);
		UART_DispInit.FlowCtrl = UART_NONE_FLOW_CTRL;

		UART_TxINTCtrl(UART_ch,ENABLE);
		UART_RxINTCtrl(UART_ch,ENABLE);

		UART_Init(UART_ch, &UART_DispInit);

		NVIC_SetPriority(INTERRUPT_TX, INT_UART_LEVEL);			/* Interruption level set */
		NVIC_ClearPendingIRQ(INTERRUPT_TX);
		NVIC_SetPriority(INTERRUPT_RX, INT_UART_LEVEL);			/* Interruption level set */
		NVIC_ClearPendingIRQ(INTERRUPT_RX);

		NVIC_EnableIRQ(INTERRUPT_TX);							/* INTTX enable */
		NVIC_EnableIRQ(INTERRUPT_RX);							/* INTRX enable */
	}

}


/**
 *********************************************************************************************
 * @brief		uart control.
 *
 * @return		none
 *********************************************************************************************
 */
void uart_control(void)
{
	if(uart_mode == cUART_RECEIVE_WAIT)
	{
		/* ʐM[hFM҂ */
		if(flg.uart.data_receive == 1)
		{
			/* M */
			flg.uart.data_receive = 0;
			data_check();

			/* Mf[^쐬 */
			send_data_set();

			/* MJn */
			txpnt = 0;
			UART_SetTxData(UART_ch, sendbuffer[0]);
		}
		else
		{
			/* no process */
		}
	}

}



/**
 *********************************************************************************************
 * @brief		send data setting.
 *
 * @return		none
 *********************************************************************************************
 */
static void send_data_set(void)
{
	uint8_t i;
	float32_t VDC_value;
	uint32_t set_value = 0;
	dac_t* const pdac = &dac;
	pfc_t *const _pfc = &Hpfc;
	vector_t* _motor;
	TSB_VE_TypeDef* pVE;
	temperature_t *const _temp = &Temp;

	pVE = TSB_VE0;

	if(status_channel == cMOTOR_CTRL_FAN)
	{
		_motor = &Motor_fan;						/* fan control */
	}
	else
	{
		_motor = &Motor_comp;					   /* compressor control */
	}

	for(i = 0; i < 7; i++)
	{
		sendbuffer[i] = 0x00;
	}

	if((_motor->drv.state.flg.emg_H) || (_motor->drv.state.flg.emg_S) ||
	   (_motor->drv.state.flg.emg_I) || (_pfc->drv.state.flg.emg_DC))
	{
		flg.uart.emg_bit = 1;
	}
	else
	{
		flg.uart.emg_bit = 0;
	}

	sendbuffer[0] = receivedata[0];					/* command id */
	sendbuffer[1] = ((flg.uart.emg_bit << 2) |		/* EMG bit on */
					 (flg.uart.ack_bit));

	if(flg.uart.ack_bit == 0){
		switch(receivedata[0])
		{
			case cGET_STATE_EMG:
				if(_motor->drv.state.flg.emg_H)
				{
					sendbuffer[2] = cEMG_STATE_EMG_H;				/* data0: over detect(hard) */
				}
				else if(_motor->drv.state.flg.emg_S)
				{
					sendbuffer[2] = cEMG_STATE_EMG_S;				/* data0: over detect(soft) */
				}
				else if(_motor->drv.state.flg.emg_I)
				{
					sendbuffer[2] = cEMG_STATE_EMG_I;				/* data0: current detect */
				}
				else if(_pfc->drv.state.flg.emg_DC)
				{
					sendbuffer[2] = cEMG_STATE_EMG_DC;				/* data0: over vdc */
				}
				else
				{
					/* no process */
				}
				break;

			case cGET_STAGE:
				sendbuffer[2] = _motor->stage.main;					/* data0: main stage */
				break;

			case cGET_CONTROL_CH:
				sendbuffer[2] = status_channel;						/* data0: status channel */
				break;

			case cGET_CARRIER_FREQUENCY:
				switch(status_channel)
				{
					case cMOTOR_CTRL_FAN:
						set_value = (uint32_t)(1000000 / cPWMPRD_FAN);
						break;

					case cMOTOR_CTRL_COMP:
						set_value = (uint32_t)(1000000 / cPWMPRD_COMP);
						break;

#if defined(__CONTROL_PFC)
					case cMOTOR_CTRL_PFC:
						set_value = (uint32_t)(1000000 / cPWMPRD_PFC);
						break;
#endif

					default:
						break;
				}
				sendbuffer[2] = (uint8_t)(set_value & 0xff);
				sendbuffer[3] = (uint8_t)((set_value >> 8) & 0xff);
				sendbuffer[4] = (uint8_t)((set_value >> 16) & 0xff);
				sendbuffer[5] = (uint8_t)((set_value >> 24) & 0xff);
				break;

			case cGET_MOTOR_SPEED_MIN:
				switch(status_channel)
				{
					case cMOTOR_CTRL_FAN:
						set_value = cHZ_MIN_FAN;
						break;

					case cMOTOR_CTRL_COMP:
						set_value = cHZ_MIN_COMP;
						break;

					default:
						break;
				}
				sendbuffer[2] = (uint8_t)(set_value & 0xff);
				sendbuffer[3] = (uint8_t)((set_value >> 8) & 0xff);
				sendbuffer[4] = (uint8_t)((set_value >> 16) & 0xff);
				sendbuffer[5] = (uint8_t)((set_value >> 24) & 0xff);
				break;

			case cGET_MOTOR_SPEED_MAX:
				switch(status_channel)
				{
					case cMOTOR_CTRL_FAN:
						set_value = cHZ_MIN_FAN;
						break;

					case cMOTOR_CTRL_COMP:
						set_value = cHZ_MIN_COMP;
						break;

					default:
						break;
				}
				sendbuffer[2] = (uint8_t)(set_value & 0xff);
				sendbuffer[3] = (uint8_t)((set_value >> 8) & 0xff);
				sendbuffer[4] = (uint8_t)((set_value >> 16) & 0xff);
				sendbuffer[5] = (uint8_t)((set_value >> 24) & 0xff);
				break;

			case cGET_DEAD_TIME:
				switch(status_channel)
				{
					case cMOTOR_CTRL_FAN:
						set_value = (cDEADTIME_FAN * 100);
						break;

					case cMOTOR_CTRL_COMP:
						set_value = (cDEADTIME_COMP * 100);
						break;

					default:
						break;
				}
				sendbuffer[2] = (uint8_t)(set_value & 0xff);
				sendbuffer[3] = (uint8_t)((set_value >> 8) & 0xff);
				sendbuffer[4] = (uint8_t)((set_value >> 16) & 0xff);
				sendbuffer[5] = (uint8_t)((set_value >> 24) & 0xff);
				break;

			case cGET_GATE_ACTIVE:
				switch(status_channel)
				{
					case cMOTOR_CTRL_FAN:
						if((cPOLH_FAN == 1) && (cPOLL_FAN == 1))
						{
							sendbuffer[2] = cGATE_ACTIVE_H_H;
						}
						else if((cPOLH_FAN == 0) && (cPOLL_FAN == 0))
						{
							sendbuffer[2] = cGATE_ACTIVE_L_L;
						}
						else if((cPOLH_FAN == 1) && (cPOLL_FAN == 0))
						{
							sendbuffer[2] = cGATE_ACTIVE_H_L;
						}
						else {
							sendbuffer[2] = cGATE_ACTIVE_L_H;
						}
						break;

					case cMOTOR_CTRL_COMP:
						if((cPOLH_COMP == 1) && (cPOLL_COMP == 1))
						{
							sendbuffer[2] = cGATE_ACTIVE_H_H;
						}
						else if((cPOLH_COMP == 0) && (cPOLL_COMP == 0))
						{
							sendbuffer[2] = cGATE_ACTIVE_L_L;
						}
						else if((cPOLH_COMP == 1) && (cPOLL_COMP == 0))
						{
							sendbuffer[2] = cGATE_ACTIVE_H_L;
						}
						else {
							sendbuffer[2] = cGATE_ACTIVE_L_H;
						}
						break;

#if defined(__CONTROL_PFC)
					case cMOTOR_CTRL_PFC:
						if((cPOLH_PFC == 1) && (cPOLL_PFC == 1))
						{
							sendbuffer[2] = cGATE_ACTIVE_H_H;
						}
						else if((cPOLH_PFC == 0) && (cPOLL_PFC == 0))
						{
							sendbuffer[2] = cGATE_ACTIVE_L_L;
						}
						else if((cPOLH_PFC == 1) && (cPOLL_PFC == 0))
						{
							sendbuffer[2] = cGATE_ACTIVE_H_L;
						}
						else {
							sendbuffer[2] = cGATE_ACTIVE_L_H;
						}
						break;
#endif
					default:
						break;
				}

			case cGET_POSITION_DITECT:
				switch(status_channel)
				{
					case cMOTOR_CTRL_FAN:
						if(cSHUNT_TYPE_FAN == c3shunt)
						{
							sendbuffer[2] = cPOSITION_DETECT_3SHUNT;
						}
						else if(cSHUNT_TYPE_FAN == c1shunt)
						{
							sendbuffer[2] = cPOSITION_DETECT_1SHUNT;
						}
						else{
							/* no process */
						}
						break;

					case cMOTOR_CTRL_COMP:
						if(cSHUNT_TYPE_COMP == c3shunt)
						{
							sendbuffer[2] = cPOSITION_DETECT_3SHUNT;
						}
						else if(cSHUNT_TYPE_COMP == c1shunt)
						{
							sendbuffer[2] = cPOSITION_DETECT_1SHUNT;
						}
						else
						{
							/* no process */
						}
						break;

					case cMOTOR_CTRL_PFC:
						sendbuffer[2] = c1shunt;
						break;

					default:
						break;
				}
				break;

			case cGET_VDC_VOLTAGE:
				switch(status_channel)
				{
					case cMOTOR_CTRL_FAN:
						VDC_value = (float32_t)(_motor->drv.Vdc * cV_MAX_FAN / FIX_15);
						break;

					case cMOTOR_CTRL_COMP:
						VDC_value = (float32_t)(_motor->drv.Vdc * cV_MAX_COMP / FIX_15);
						break;

#if defined(__CONTROL_PFC)
					case cMOTOR_CTRL_PFC:
						VDC_value = (float32_t)(_pfc->drv.Vdc_ref * cV_MAX_PFC / FIX_15);
						break;
#endif
					default:
						break;
				}
				set_value = (uint32_t)(VDC_value * 100);
				sendbuffer[2] = (uint8_t)(set_value & 0xff);
				sendbuffer[3] = (uint8_t)((set_value >> 8) & 0xff);
				sendbuffer[4] = (uint8_t)((set_value >> 16) & 0xff);
				sendbuffer[5] = (uint8_t)((set_value >> 24) & 0xff);
				break;

			case cGET_U_VOLTAGE:
				VDC_value = (float32_t)(pVE->IBADC * 5) / 65535;
				set_value = (uint32_t)(VDC_value * 100);
				sendbuffer[2] = (uint8_t)(set_value & 0xff);
				sendbuffer[3] = (uint8_t)((set_value >> 8) & 0xff);
				sendbuffer[4] = (uint8_t)((set_value >> 16) & 0xff);
				sendbuffer[5] = (uint8_t)((set_value >> 24) & 0xff);
				break;

			case cGET_V_VOLTAGE:
				VDC_value = (float32_t)(pVE->IBADC * 5) / 65535;
				set_value = (uint32_t)(VDC_value * 100);
				sendbuffer[2] = (uint8_t)(set_value & 0xff);
				sendbuffer[3] = (uint8_t)((set_value >> 8) & 0xff);
				sendbuffer[4] = (uint8_t)((set_value >> 16) & 0xff);
				sendbuffer[5] = (uint8_t)((set_value >> 24) & 0xff);
				break;

			case cGET_W_VOLTAGE:
				VDC_value = (float32_t)(pVE->ICADC * 5) / 65535;
				set_value = (uint32_t)(VDC_value * 100);
				sendbuffer[2] = (uint8_t)(set_value & 0xff);
				sendbuffer[3] = (uint8_t)((set_value >> 8) & 0xff);
				sendbuffer[4] = (uint8_t)((set_value >> 16) & 0xff);
				sendbuffer[5] = (uint8_t)((set_value >> 24) & 0xff);
				break;

			case cGET_DAC_DATA:
#if defined(__USE_DAC)
				switch(pdac->select)
				{
					case 0:
						sendbuffer[2] = cDAC_DATA_TMPREG0;
						sendbuffer[3] = cDAC_DATA_TMPREG1;
						sendbuffer[4] = cDAC_DATA_TMPREG2;
						sendbuffer[5] = cDAC_DATA_THETAHALF;
						break;

					case 1:
						sendbuffer[2] = cDAC_DATA_IDREF;
						sendbuffer[3] = cDAC_DATA_ID;
						sendbuffer[4] = cDAC_DATA_IQREF;
						sendbuffer[5] = cDAC_DATA_IQ;
						break;

					case 2:
						sendbuffer[2] = cDAC_DATA_OMEGACOMHALH;
						sendbuffer[3] = cDAC_DATA_OMEGAHALF;
						sendbuffer[4] = cDAC_DATA_OMEGADEV;
						sendbuffer[5] = cDAC_DATA_IQREF;
						break;

					case 3:
						sendbuffer[2] = cDAC_DATA_TMPREG0;
						sendbuffer[3] = cDAC_DATA_IQREF;
						sendbuffer[4] = cDAC_DATA_IDREF;
						sendbuffer[5] = cDAC_DATA_OMEGAHALF;
						break;

					default :
						break;
				}
#endif
				break;

			case cGET_INTERNAL_AMP:
				sendbuffer[2] = cINTERNAL_AMP_NO;
				break;

			case cGET_DIRECTION:
				sendbuffer[2] = cDIRECTION_CW;
				break;

			case cGET_MODULATION:
				sendbuffer[2] = _motor->usr.com_user.modul;
				break;

			case cGET_MOTOR_SPEED:
				switch(status_channel)
				{
					case cMOTOR_CTRL_FAN:
						sendbuffer[2] = (int8_t)((_motor->drv.omega.word * cHZ_MAX_FAN) / FIX_31);
						break;

					case cMOTOR_CTRL_COMP:
						sendbuffer[2] = (int8_t)((_motor->drv.omega.word * cHZ_MAX_COMP) / FIX_31);
						break;

					default:
						break;
				}
				break;

			case cREQ_CHANGE_MOTOR:
			  request_motor = receivedata[1];
				switch(request_motor)
				{
					case cMOTOR_CTRL_FAN:
						break;

					case cMOTOR_CTRL_COMP:
						break;

					default:
						break;
				}
				break;
			case cREQ_ON_OFF_PFC:
				flg.motor.pfc_ctrl = receivedata[1];
				if(flg.motor.pfc_ctrl == 0)
				{
					_pfc->pfc_mode_req = cPFC_ON;
				}
				else
				{
					_pfc->pfc_mode_req = cPFC_OFF;
				}
				break;

			case cREQ_ALL_STOP_MOTOR:
				flg.motor.all_stop = receivedata[1];
				target_spd_fan = 0;
				target_spd_comp = 0;
				break;

			case cREQ_STATUS_CH:
				status_channel = receivedata[1];
				break;

			case cREQ_STATUS_DAC:
				dac_mode = receivedata[1];
				break;

			case cGET_OUTROOM_TEMP:
				flg.uart.na_bit = 1;
				sendbuffer[2] = flg.uart.na_bit;
				break;

			case cGET_OUTPIPE_TEMP:
				flg.uart.na_bit = 1;
				sendbuffer[2] = flg.uart.na_bit;
				break;

			case cGET_EXHAUST_TEMP:
				flg.uart.na_bit = 1;
				sendbuffer[2] = flg.uart.na_bit;
				break;

			case cGET_DIODE_TEMP:
				if(1)
				{
					sendbuffer[2] = _temp->t_diode;
				}
				else
				{
					flg.uart.na_bit = 1;
					sendbuffer[2] = flg.uart.na_bit;
				}
				break;

			case cGET_IGBT_TEMP:
				if(1)
				{
					sendbuffer[2] = _temp->t_igbt;
				}
				else
				{
					flg.uart.na_bit = 1;
					sendbuffer[2] = flg.uart.na_bit;
				}
				break;

			case cGET_HVMOS_TEMP:
				if(1)
				{
					sendbuffer[2] = _temp->t_hvmos;
				}
				else
				{
					flg.uart.na_bit = 1;
					sendbuffer[2] = flg.uart.na_bit;
				}
				break;

			case cREQ_SYSTEM_START:
			case cREQ_ROTATE_MOTOR:
			default:
				break;
		}
	}

	for(i = 0; i < 6; i++)
	{
		sendbuffer[6] += sendbuffer[i];
	}

}


/**
 *********************************************************************************************
 * @brief		data check.
 *
 * @return		none
 *********************************************************************************************
 */
static void data_check(void)
{
	uint8_t i, checksum = 0;
	int32_t spd;

	flg.uart.na_bit = 0;
	flg.uart.ack_bit = 0;

	switch(receivedata[0])
	{
		case cGET_STATE_EMG:
		case cGET_STAGE:
		case cGET_CONTROL_CH:
		case cGET_CARRIER_FREQUENCY:
		case cGET_MOTOR_SPEED_MIN:
		case cGET_MOTOR_SPEED_MAX:
		case cGET_DEAD_TIME:
		case cGET_GATE_ACTIVE:
		case cGET_POSITION_DITECT:
		case cGET_VDC_VOLTAGE:
		case cGET_U_VOLTAGE:
		case cGET_V_VOLTAGE:
		case cGET_W_VOLTAGE:
		case cGET_DAC_DATA:
		case cGET_INTERNAL_AMP:
		case cGET_DIRECTION:
		case cGET_MODULATION:
		case cGET_MOTOR_SPEED:
		case cREQ_CHANGE_MOTOR:
		case cREQ_ON_OFF_PFC:
		case cREQ_ALL_STOP_MOTOR:
		case cREQ_STATUS_CH:
		case cREQ_STATUS_DAC:
		case cGET_OUTROOM_TEMP:
		case cGET_OUTPIPE_TEMP:
		case cGET_EXHAUST_TEMP:
		case cGET_DIODE_TEMP:
		case cGET_IGBT_TEMP:
		case cGET_HVMOS_TEMP:
		case cREQ_SYSTEM_START:
		case cREQ_ROTATE_MOTOR:
			/* no process */
			break;

		default:
			flg.uart.ack_bit = 1;
			break;
	}

	for(i = 0; i < 5; i++)
	{
		checksum = (uint8_t)(checksum + receivedata[i]);
	}

	if(checksum != receivedata[5])
	{
		flg.uart.ack_bit = 1;
	}

	if((receivedata[0] == cREQ_ROTATE_MOTOR) && (flg.uart.ack_bit == 0))
	{
		spd = receivedata[1];
		
		switch(request_motor)
		{
			case cMOTOR_CTRL_FAN:
				target_spd_fan = spd;
				break;

			case cMOTOR_CTRL_COMP:
				target_spd_comp = spd;
				break;

			default:
				break;
		}
	}

}


/**
 *********************************************************************************************
 * @brief		Transmission completion interrupt.
 *
 * @return		none
 *********************************************************************************************
 */
void INT_uart0tx(void)
{
	if(UART_GetErrState(UART_ch) == UART_NO_ERR)
	{
		if(++txpnt >= cSEND_DATA_NUM)
		{
			txpnt = 0;
		}
		else
		{
			UART_SetTxData(UART_ch, sendbuffer[txpnt]);
		}
	}
}


/**
 *********************************************************************************************
 * @brief		Receive completion interrupt.
 *
 * @return		none
 *********************************************************************************************
 */
void INT_uart0rx(void)
{
	if(UART_GetErrState(UART_ch) == UART_NO_ERR)
	{
		receivebuffer[rxpnt] = UART_GetRxData(UART_ch);

		/* receive timer init */
		receive_timer = 10;

		if(++rxpnt >= cRECEIVE_DATA_NUM)
		{
			receivedata[0] = receivebuffer[0];
			receivedata[1] = receivebuffer[1];
			receivedata[2] = receivebuffer[2];
			receivedata[3] = receivebuffer[3];
			receivedata[4] = receivebuffer[4];
			receivedata[5] = receivebuffer[5];

			flg.uart.data_receive = 1;
			receive_timer = 0;
			rxpnt = 0;
		}
	}
}

#endif /* __USE_microE_COMMUNICATION_FUNC */

/**
 ********************************************************************************************
 * @brief	   UiOutDataStart by VE.
 *
 * @param[in]   uint8_t _intch  .
 *
 * @return	  none
 ********************************************************************************************
 */
void UiOutDataStartByVE(uint8_t _intch)
{
#if defined(__USE_VE_FAN) || defined(__USE_VE_COMP)
	dac_t* const pdac = &dac;

	if (_intch == pdac->motch)	  /* Dac output ch = call intve ch? */
	{
		uint16_t* const pdacout = DacData;
		vector_t* pmotor = 0;
		TSB_VE_TypeDef* pVE = 0;
		pfc_t* const _pfc = &Hpfc;
		if (pdac->motch == 0)
		{
#if defined(__CONTROL_MOTOR_FAN)
			pmotor = &Motor_fan;
#if defined(__USE_VE_FAN)
			pVE = TSB_VE0;
#endif
#endif
		}
		else
		{
#if defined(__CONTROL_MOTOR_COMP)
			pmotor = &Motor_comp;
#if defined(__USE_VE_COMP)
			pVE = TSB_VE1;
#endif
#endif
		}

		switch (pdac->select)
		{
			case 0:
				*(pdacout + 0) = (pVE->IA						>> (16 - pdac->datsft0)) ^ 0x8000;
				*(pdacout + 1) = (pVE->IB						>> (16 - pdac->datsft0)) ^ 0x8000;
				*(pdacout + 2) = (pVE->IC						>> (16 - pdac->datsft0)) ^ 0x8000;
				*(pdacout + 3) = (pmotor->drv.theta.half[1]							   ) ^ 0x8000;
				break;
			case 1:
				*(pdacout + 0) = (pmotor->drv.Id_ref			<< pdac->datsft0) ^ 0x8000;
				*(pdacout + 1) = (pmotor->drv.Id				<< pdac->datsft0) ^ 0x8000;
				*(pdacout + 2) = (pmotor->drv.Iq_ref			<< pdac->datsft0) ^ 0x8000;
				*(pdacout + 3) = (pmotor->drv.Iq				<< pdac->datsft0) ^ 0x8000;
				break;
			case 2:
				*(pdacout + 0) = _pfc->drv.Iac_amp_ref * 100;
				*(pdacout + 1) = _pfc->drv.Iac_ref * 100;
				*(pdacout + 2) = _pfc->drv.Iac * 10;
				*(pdacout + 3) = _pfc->drv.m_Vac;
				break;
			case 3:
				*(pdacout + 0) = (_pfc->drv.Vdc_lpf0				<< pdac->datsft0);
				*(pdacout + 1) = (_pfc->drv.Vac					<< pdac->datsft1) ^ 0x8000;
				*(pdacout + 2) = (_pfc->drv.Iac					<< pdac->datsft2) ^ 0x8000;
				*(pdacout + 3) = (_pfc->drv.Iac_raw				<< pdac->datsft2) ^ 0x8000;
				break;
			case 4:
				*(pdacout + 0) = (pVE->IA					   >> (16 - pdac->datsft0)) ^ 0x8000;
				*(pdacout + 1) = (pmotor->drv.omega.half[1]	 << pdac->datsft2) ^ 0x8000;
				*(pdacout + 2) = (pmotor->drv.Iq_ref			<< pdac->datsft0) ^ 0x8000;
				*(pdacout + 3) = pmotor->drv.theta.half[1];
				break;
			case 5:
				*(pdacout + 0) = pmotor->drv.Vdc_ave.half[1]	<< pdac->datsft1;
				*(pdacout + 1) = pmotor->drv.Vdq_ave.half[1]	<< pdac->datsft1;
				*(pdacout + 2) = (pmotor->drv.Id_ref			<< pdac->datsft0) ^ 0x8000;
				*(pdacout + 3) = (pmotor->drv.Iq_ref			<< pdac->datsft0) ^ 0x8000;
				break;
			case 6:
				*(pdacout + 0) = (pmotor->drv.omega.half[1]	 << pdac->datsft2) ^ 0x8000;
				*(pdacout + 1) = (pmotor->drv.Iq				<< pdac->datsft0) ^ 0x8000;
				*(pdacout + 2) = (pmotor->drv.Ed				<< pdac->datsft1) ^ 0x8000;
				*(pdacout + 3) = (pmotor->drv.theta.half[1]);
				break;
			case 7:
				*(pdacout + 0) = (pVE->VD					   >> (16 - pdac->datsft1)) ^ 0x8000;
				*(pdacout + 1) = (pVE->IA					   >> (16 - pdac->datsft0)) ^ 0x8000;
				*(pdacout + 2) = (pmotor->drv.omega.half[1]	 << pdac->datsft2)		^ 0x8000;
				*(pdacout + 3) = pmotor->drv.theta.half[1];
				break;
			case 8:
				*(pdacout + 0) = (pVE->IAADC);
				*(pdacout + 1) = (pVE->IBADC);
				*(pdacout + 2) = (pVE->ICADC);
				*(pdacout + 3) = (pVE->VDC);
				break;
			case 9:
				*(pdacout + 0) = (pmotor->drv.Ia				<< pdac->datsft0) ^ 0x8000;
				*(pdacout + 1) = (pmotor->drv.Ib				<< pdac->datsft0) ^ 0x8000;
				*(pdacout + 2) = (pmotor->drv.Ic				<< pdac->datsft0) ^ 0x8000;
				*(pdacout + 3) = (pmotor->para.err_ovc		  << pdac->datsft0) ^ 0x8000;
				break;
			case 10:
#if defined(__USE_ENCODER_FAN) || (__USE_ENCODER_COMP)
				*(pdacout + 0) = (pmotor->drv.theta.half[1]);
				*(pdacout + 1) = (pmotor->drv.theta_enc >> 16);
				*(pdacout + 2) = (pmotor->drv.theta_com);
				*(pdacout + 3) = (pVE->IA >>					(16 - pdac->datsft0)) ^ 0x8000;
#endif
				break;
			case 11:
#if defined(__USE_ENCODER_FAN) || (__USE_ENCODER_COMP)
				*(pdacout + 0) = (pmotor->drv.theta.half[1]);
				*(pdacout + 1) = (pmotor->drv.theta_enc >> 16);
				*(pdacout + 2) = (pmotor->drv.theta_com);
				*(pdacout + 3) = (pmotor->drv.EnCnt);
#endif
				break;
			case 12:
				*(pdacout + 0) = (pmotor->drv.Id				<< pdac->datsft0) ^ 0x8000;
				*(pdacout + 1) = (pmotor->drv.Iq				<< pdac->datsft0) ^ 0x8000;
				*(pdacout + 2) = (pVE->IA					   >> (16 - pdac->datsft0)) ^ 0x8000;
				*(pdacout + 3) = pVE->SECTOR					<< 11;
				break;
			case 13:
				*(pdacout + 0) = _pfc->judgevac.Vac_balance_temp * 10;
				*(pdacout + 1) = _pfc->judgevac.Vac_high * 10; 
				*(pdacout + 2) = _pfc->drv.m_Vac * 10; 
				*(pdacout + 3) = _pfc->drv.Vac * 10;
				break;
			case 16:
				*(pdacout + 0) = (pmotor->drv.Ic + 0x8000);
				*(pdacout + 1) = (pmotor->drv.Ib + 0x8000);
				*(pdacout + 2) = (pmotor->drv.Ic + 0x8000);
				*(pdacout + 3) = (pmotor->drv.Ic + 0x8000);
				break;

			default:
				*(pdacout + 0) += 0x0100;
				*(pdacout + 1) += 0x0200;
				*(pdacout + 2) += 0x0300;
				*(pdacout + 3) += 0x0400;
				break;
		}

		DacWriteStart(DAC_SIO_CH);

	}
#endif /* defined(__USE_VE_FAN) || defined(__USE_VE_COMP) */
}

/**
 *********************************************************************************************
 * @brief		UiOutDataStart for Software Vector Control.
 *
 * @param		uint8_t	_intch	call intve ch
 *
 * @return		none
 *********************************************************************************************
 */
void UiOutDataStart(uint8_t _intch)
{
#if !defined(__USE_VE_FAN) || !defined(__USE_VE_COMP)
	dac_t* const pdac = &dac;

	pfc_t* const _pfc = &Hpfc;
	pdac->motch = 1;

	if (_intch == pdac->motch)		/* Dac output ch = call intve ch? */
	{
		uint16_t* const pdacout = DacData;
		vector_t* pmotor = 0;

		if (pdac->motch == 0)
		{
#if defined(__CONTROL_MOTOR_FAN)
			pmotor = &Motor_fan;
#endif
		}
		else
		{
#if defined(__CONTROL_MOTOR_COMP)
			pmotor = &Motor_comp;
#endif
		}

		switch (pdac->select)
		{
			case 0:
				*(pdacout + 0) = (pmotor->drv.Ia				<< pdac->datsft0) ^ 0x8000;
				*(pdacout + 1) = (pmotor->drv.Ib				<< pdac->datsft0) ^ 0x8000;
				*(pdacout + 2) = (pmotor->drv.Ic				<< pdac->datsft0) ^ 0x8000;
				*(pdacout + 3) = (pmotor->drv.theta.half[1]						) ^ 0x8000;
				break;
			case 1:
				*(pdacout + 0) = _pfc->drv.Vdc_lpf0 * 10; 
				*(pdacout + 1) = pmotor->drv.Ib + 0x8000;
				*(pdacout + 2) = _pfc->drv.state.flg.emg_DC * 0x8000; 
				*(pdacout + 3) = pmotor->drv.Iq + 0x8000;
				break;
			case 2:
				*(pdacout + 0) = _pfc->drv.Vdc_lpf0 * 10;
				*(pdacout + 1) = pmotor->drv.Ib + 0x8000;
				*(pdacout + 2) = _pfc->drv.state.flg.emg_DC * 0x8000; 
				*(pdacout + 3) = pmotor->drv.Iq + 0x8000;
				break;
			case 3:
				*(pdacout + 0) = (_pfc->drv.Vdc_lpf0				<< pdac->datsft0);
				*(pdacout + 1) = (_pfc->drv.Vac					<< pdac->datsft1) ^ 0x8000;
				*(pdacout + 2) = (_pfc->drv.Iac					<< pdac->datsft2) ^ 0x8000;
				*(pdacout + 3) = (_pfc->drv.Iac_raw				<< pdac->datsft2) ^ 0x8000;
				break;
			case 5:
				*(pdacout + 0) = pmotor->drv.Ic + 0x8000; 
				*(pdacout + 1) = pmotor->drv.Iq_ref + 0x8000; 
				*(pdacout + 2) = pmotor->drv.Iq + 0x8000; 
				*(pdacout + 3) = pmotor->drv.Ic_raw + 0x8000;
				break;
			case 6:
				*(pdacout + 0) = pmotor->drv.Ia_raw + 0x8000; 
				*(pdacout + 1) = pmotor->drv.Ia + 0x8000; 
				*(pdacout + 2) = pmotor->drv.Ic + 0x8000; 
				*(pdacout + 3) = pmotor->drv.Ic_raw + 0x8000;
				break;
			case 7:
				*(pdacout + 0) = pmotor->drv.Ia + 0x8000; 
				*(pdacout + 1) = pmotor->drv.Ib + 0x8000;
				*(pdacout + 2) = pmotor->drv.Ic + 0x8000; 
				*(pdacout + 3) = pmotor->drv.Ic + 0x8000;
				break;
			case 8:
				*(pdacout + 0) = pmotor->drv.Ia_raw + 0x8000;
				*(pdacout + 1) = pmotor->drv.Ib_raw + 0x8000;
				*(pdacout + 2) = pmotor->drv.Iq_ref + 0x8000;
				*(pdacout + 3) = pmotor->drv.Ic_raw + 0x8000;
				break;
			case 10:
				*(pdacout + 0) = pmotor->drv.Ia + 0x8000; 
				*(pdacout + 1) = pmotor->drv.theta.half[1]; 
				*(pdacout + 2) = pmotor->drv.Iq + 0x8000; 
				*(pdacout + 3) = pmotor->drv.Ia_raw + 0x8000;
#if defined(__USE_ENCODER_FAN) || (__USE_ENCODER_COMP)
				*(pdacout + 0) = (pmotor->drv.theta.half[1]);
				*(pdacout + 1) = (pmotor->drv.theta_enc >> 16);
				*(pdacout + 2) = (pmotor->drv.theta_com);
				*(pdacout + 3) = (pmotor->drv.Ia				<< pdac->datsft0) ^ 0x8000;
#endif
				break;
			case 11:
#if defined(__USE_ENCODER_FAN) || (__USE_ENCODER_COMP)
				*(pdacout + 0) = (pmotor->drv.theta.half[1]);
				*(pdacout + 1) = (pmotor->drv.theta_enc >> 16);
				*(pdacout + 2) = (pmotor->drv.theta_com);
				*(pdacout + 3) = pmotor->drv.EnCnt;
#endif
				break;
			case 12:
				*(pdacout + 0) = pmotor->drv.DutyU;
				*(pdacout + 1) = pmotor->drv.DutyV;
				*(pdacout + 2) = pmotor->drv.DutyW;
				*(pdacout + 3) = pmotor->drv.Sector0	<< 12;
				break;
			case 13:
				*(pdacout + 0) = _pfc->drv.Iac * 10 + 0x8000;
				*(pdacout + 1) = _pfc->dcave.Vdc_ave * 5; 
				*(pdacout + 2) = _pfc->drv.Iac_amp_ref * 20 + 0x8000;
				*(pdacout + 3) = _pfc->drv.Iac_ref * 10 + 0x8000;
				break;
			case 14:
				*(pdacout + 0) = (pmotor->drv.Ia_raw + 0x8000);
				*(pdacout + 1) = (pmotor->drv.idetect_error * 0x8000);
				*(pdacout + 2) = (pmotor->drv.Ib_raw + 0x8000);
				*(pdacout + 3) = (pmotor->drv.Ic_raw + 0x8000);
				break;
			case 15:
				*(pdacout + 0) = (pmotor->drv.Iq + 0x8000);
				*(pdacout + 1) = (pmotor->drv.Iq_ref + 0x8000);
				*(pdacout + 2) = (pmotor->drv.Id + 0x8000);
				*(pdacout + 3) = (pmotor->drv.Id_ref + 0x8000);
				break;
			case 19:
				*(pdacout + 0) = (pmotor->drv.Ia + 0x8000);
				*(pdacout + 1) = (pmotor->drv.idetect_error * 0x8000);
				*(pdacout + 2) = (pmotor->drv.Ia_raw + 0x8000);
				*(pdacout + 3) = (pmotor->drv.Ic + 0x8000);
				break;
			case 20:
				*(pdacout + 0) = (pmotor->drv.Iq + 0x8000);
				*(pdacout + 1) = (pmotor->drv.Ic_raw + 0x8000);
				*(pdacout + 2) = (pmotor->drv.Iq + 0x8000);
				*(pdacout + 3) = (pmotor->drv.idetect_error * 0x8000);
				break;
			case 21:
				*(pdacout + 0) = (pmotor->drv.Ib + 0x8000);
				*(pdacout + 1) = (pmotor->drv.Ia + 0x8000);
				*(pdacout + 2) = (pmotor->drv.Ic + 0x8000);
				*(pdacout + 3) = (pmotor->drv.Ib_raw + 0x8000);
				break;
			case 22:
				*(pdacout + 0) = pmotor->drv.theta.half[1] + 0x8000;
				*(pdacout + 1) = (pmotor->drv.Iq + 0x8000);
				*(pdacout + 2) = (pmotor->drv.Iq_ref + 0x8000);
				*(pdacout + 3) = (pmotor->drv.Ia + 0x8000);
				break;
			case 23:
				*(pdacout + 0) = (pmotor->drv.Ic_raw + 0x8000);
				*(pdacout + 1) = (pmotor->drv.Ib + 0x8000);
				*(pdacout + 2) = (pmotor->drv.Ic + 0x8000);
				*(pdacout + 3) = (pmotor->drv.idetect_error * 0x8000);
				break;
			case 24:
				*(pdacout + 0) = (pmotor->drv.Ib + 0x8000);
				*(pdacout + 1) = (pmotor->drv.Ib_raw + 0x8000);
				*(pdacout + 2) = (pmotor->drv.Ic + 0x8000);
				*(pdacout + 3) = (pmotor->drv.Ic_raw + 0x8000);
				break;
			case 25:
				*(pdacout + 0) = _pfc->drv.Vac_adc * 10; 
				*(pdacout + 1) = _pfc->drv.Iac_adc0 * 10; 
				*(pdacout + 2) = _pfc->drv.Iac_adc1 * 10;
				*(pdacout + 3) = _pfc->drv.Vac * 10;
				break;
			case 26:
				*(pdacout + 0) = pmotor->power.Iac_eff * 10;
				*(pdacout + 1) = pmotor->power.Iac_eff * 10;
				*(pdacout + 2) = pmotor->power.apparent_power * 10;
				*(pdacout + 3) = 0;
				break;
			default:
				*(pdacout + 0) += 0x0100;
				*(pdacout + 1) += 0x0200;
				*(pdacout + 2) += 0x0300;
				*(pdacout + 3) += 0x0400;
				break;
		}

		DacWriteStart(DAC_SIO_CH);

	}
#endif /* !defined(__USE_VE_FAN) || !defined(__USE_VE_COMP) */
}

/******************************** END OF FILE *********************************/
