/**
 ****************************************************************************
 * @file	B_User.h
 * @brief	Motor control for user Header File
 * @version V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2023
 *****************************************************************************
 */
#ifndef _B_USER_H
#define _B_USER_H

#undef EXTERN
#ifdef DEFINE_APP
#define EXTERN
#else
#define EXTERN  extern
#endif
#include "ipdrv_adc.h"
#include "Ipdefine.h"

/*===================================================================*
	  Constant Data Definition
 *===================================================================*/

#define cFAN_S1					(4000)	/*[ms]Compressor start delay count */
#define cFAN_S2					(20000)	/*[ms]20000 Fan stop delay count */
#define c500MS 					(500)	/*[ms] LED Blink on */
#define c2000MS 				(2500)	/*[ms] interval time between blink */
#define cERROR_CNT  			(60000)	/*[ms] 60s clear Error mode */

/*===================================================================*
  Macro Definition
 *===================================================================*/
typedef enum {
	cTEMP_NORMAL = 0,
	cTEMP_SLOW_INCREASE = 1,
	cTEMP_NO_INCREASE = 2,
	cTEMP_SLOW_DECREASE = 3,
	cTEMP_EMG = 4
} temp_protect_st_e;

typedef union {
	struct {
	uint8_t t_normal:1;			/*outdoor temperature sensor open flag */
	uint8_t t_comp_suc:1;		/*outdoor pipe sensor open flag */
	uint8_t t_igbt:1;			/*outdoor igbt sensor open flag */
	uint8_t t_diode:1; 			/*outdoor diode sensor open flag */
	uint8_t t_antifreeze:1;		
	uint8_t t_indoor_pipe:1;	/*indoor temperature sensor short flag */
	uint8_t t_outdoor_pipe:1;	/*outdoor pipe sensor short flag */
	uint8_t t_hvmos:1;			/*outdoor hvmos sensor short flag */
	uint8_t t_exh:1;			/*outdoor exhaust sensor short flag */
	} flg;
	uint8_t all;
}temp_protect_index_u;

typedef struct {
	uint8_t sLED;
	uint8_t time;
	uint8_t LED;
} led_t;

typedef struct {
	uint8_t	error_type;					/* PFC error type */
	uint8_t times[4];					/* LED blink times */
	uint16_t blink_cnt[4];				/* LED blink count */
	uint16_t interval_cnt[4];			/* LED interval count */
	uint8_t cnt;
	uint8_t num;
} error_led_t;

typedef union {
	struct {
		uint16_t f_open_outdoor:1;		/* outdoor temperature sensor open flag */
		uint16_t f_open_pipe:1;			/* outdoor pipe sensor open flag */
		uint16_t f_open_exhaust:1;		/* outdoor exhaust sensor open flag */
		uint16_t f_open_hvmos:1;		/* outdoor HVMOS sensor open flag */
		uint16_t f_open_igbt:1;
		uint16_t f_open_diode:1;
		uint16_t f_short_outdoor:1;		/* outdoor temperature sensor short flag */
		uint16_t f_short_pipe:1;		/* outdoor pipe sensor short flag */
		uint16_t f_short_exhaust:1;		/* outdoor exhaust sensor short flag */
		uint16_t f_short_hvmos:1;		/* outdoor HVMOS sensor short flag */
		uint16_t f_short_igbt:1;
		uint16_t f_short_diode:1;
		uint16_t f_over_temp:1;			/* outdoor HVMOS sensor short flag */
	} flg;
	uint16_t all;
} Tc_state_u;

typedef struct {
	int16_t tc_sl_up;					/* Compressor Slow up T point */
	int16_t tc_no_up;					/* Compressor no up T point */
	int16_t tc_sl_down;					/* Compressor Slow down T point */
	int16_t tc_over_emg;				/* Compressor EMG T point */
	uint32_t t_c0;						/* Compressor speed slow up filter count */
	uint32_t t_c1;						/* Compressor speed no up filter count */
	uint32_t t_c2;						/* Compressor speed slow down filter count */
	uint32_t t_c3;						/* Compressor speed EMG filter count */
	uint32_t t_c;						/* Compressor speed EMG filter count */
	uint8_t user_st;					/* Compressor temperature protect status 0: normal status; 1: slow up, 2: no increase, 3: force slow down, 4: EMG */
	uint8_t t_over_emg;					/* Over temperature flag */

} tc_Lim_t;

typedef struct {
	ADC_Result R_HVMOS;					/* Read temperature from AD */
	ADC_Result R_EXHAUST;				/* Read temperature from AD */
	ADC_Result R_OUTPIPE;				/* Read temperature from AD */
	ADC_Result R_OUTDOORT;				/* Read temperature from AD */
	ADC_Result R_IGBT;					/* Read temperature from AD */
	ADC_Result R_DIODE;					/* Read temperature from AD */
	uint8_t cnt0;						/* Outdoor T sensor open filter count */
	uint8_t cnt1;						/* Outpipe T sensor open filter count */
	uint8_t cnt2;						/* exhaust T sensor open filter count */
	uint8_t cnt3;						/* HVMOS T sensor open filter count */
	uint8_t cnt4;						/* outdoor T sensor short filter count */
	uint8_t cnt5;						/* outpipe T sensor short filter count */
	uint8_t cnt6;						/* Exhaust T sensor short filter count */
	uint8_t cnt7;						/* HVMOS T sensor short filter count */
	uint8_t cnt8;						/* IGBT T sensor open filter count */
	uint8_t cnt9;						/* IGBT T sensor short filter count */
	uint8_t cnt10;						/* Diode T sensor open filter count */
	uint8_t cnt11;						/* Diode T sensor short filter count */
	uint16_t r_exhaust_f;				/* Exhaust T fiter FIXPOINT value */
	uint16_t r_outdoort_f;				/* Outdoor T fiter FIXPOINT value */
	uint16_t r_outpipe_f;				/* Outpipe T fiter FIXPOINT value */
	uint16_t r_hvmos_f;					/* HVMOS T fiter FIXPOINT value */
	uint16_t r_igbt_f;					/* IGBT T fiter FIXPOINT value */
	uint16_t r_diode_f;					/* DIODE T fiter FIXPOINT value */
	int16_t t_outpipe;					/* [T(C)]*10 get T(C) from table */
	int16_t t_outdoor;					/* [T(C)]*10 get T(C) from table */
	int16_t t_exh;						/* [T(C)]*10 get T(C) from table */
	int16_t t_hvmos;					/* [T(C)]*10 get T(C) from table */
	int16_t t_igbt;						/* [T(C)]*10 get T(C) from table */
	int16_t t_diode;					/* [T(C)]*10 get T(C) from table */

	temp_protect_st_e t_st;				/* T protest Status */
	temp_protect_st_e t_st_pre;			/* T protest Pre Status */
	temp_protect_index_u t_index;		/* Temperature priority index*/
	Tc_state_u state;					/* Temperature check open/short states */
	q31_u omega_fl2;					/* Compressor Speed min speed limit */
	tc_Lim_t t_exh_protect;				/* Exhaust protect Range Setting */
	tc_Lim_t t_hvmos_protect;			/* HVMOS protect Range Setting */
	tc_Lim_t t_igbt_protect;			/* IGBT protect Range Setting */
	tc_Lim_t t_diode_protect;			/* DIODE protect Range Setting */
	tc_Lim_t T_outpipe_protect;			/* Outpipe protect Range Setting */
} temperature_t;

/*****************************************************************
 * Definition of variable
 *****************************************************************/
EXTERN led_t	LED_Disp[4];
EXTERN error_led_t  Error_led;
EXTERN temperature_t	Temp;
EXTERN error_t  Error;					/* Error structure for error */
/*===================================================================*
	  Proto Type Definition
 *===================================================================*/
EXTERN void B_Motor_Init(void);
EXTERN void B_User_MotorControl(void);
EXTERN void B_Protect_GetTemperature(temperature_t * const _temp);
EXTERN void B_Protect_CheckTemperatureError(temperature_t * const _temp);
EXTERN void B_Protect_Temperature(temperature_t * const _temp, vector_t* const _motor);
EXTERN void B_User_FanMotor_Rstart_Control(void);
EXTERN int16_t GetTempValue(const uint16_t* _temp_tbl, int16_t _index_min, int16_t _index_max, uint16_t _ad_value);
EXTERN void B_Protect_TemperatureStatusSel(tc_Lim_t * _tc, int16_t _t,uint16_t times);
EXTERN void B_Protect_Antifreeze_StatusSel(tc_Lim_t * _tc, int16_t _t,uint32_t times);
EXTERN void B_Protect_TemperatureMonitorInit(void);
EXTERN void B_Protect_TemperatureControlRef(temperature_t * const _temp, vector_t* const _motor);
EXTERN void B_Error_Display(uint8_t _time, uint8_t _sLED, uint8_t i);
EXTERN void B_Error_Collection(void);
EXTERN void B_Error_DisplayCtrl(void);
EXTERN int16_t Binary_search(const int16_t * _temp_tbl, int16_t _index_min, int16_t _index_max, int16_t _value);
#undef EXTERN
#endif /* _B_USER_H */

/*********************************** END OF FILE ******************************/
