/**
 ****************************************************************************
 * @file:	 dac_drv.c
 * @brief:	 DAC IC Control for AD5624 on TSPI.
 * @version  V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2023
 *****************************************************************************
 */
#include "ipdefine.h"

#include "ipdrv_tspi.h"
#include "ipdrv_cg.h"

#define DEFINE_APP
#include "dac_drv.h"
#undef	DEFINE_APP

/**
 * @note	DAC IC Control for AD5624.
 *
 */

/*===================================================================*
  Device local functions prototype.
 *===================================================================*/
static void DacLocalWriteInit(TSB_TSPI_TypeDef* const TSPIx);

/*===================================================================*
	  Local Variable Definition
 *===================================================================*/
uint32_t DacNo = DAC_MAX_CH;


/*===================================================================*
	  Const Definition
 *===================================================================*/
/* Command Definition */
#define	DAC_CMD_W_INPUT					0U		/* Write to input register n */
#define	DAC_CMD_UP_DAC					1U		/* Update DAC register n */
#define	DAC_CMD_W_INPUT_UP_ALL			2U		/* Write to input register n, update all(software LDAC) */
#define	DAC_CMD_W_UP_DAC				3U		/* Write to and update DAC channel n */
#define	DAC_CMD_POWER_DOWN				4U		/* Power down DAC (power-up) */
#define	DAC_CMD_RESET					5U		/* Reset */
#define	DAC_CMD_LOAD_LDAC				6U		/* Load LDAC register */

#define	DAC_RESET_MODE					1U		/* Reset mode :  0:DAC and Input shift register */
												/*            :  1:DAC and Input shift and LDAC and Power-down register */


#define	cDAC_POWERRESET					((DAC_CMD_RESET << 19U) | (DAC_RESET_MODE << 4U))
#define	cDAC_LDAC_NORMAL				((DAC_CMD_LOAD_LDAC << 19U) | 0)

/**
 *********************************************************************************************
 * @brief		D/A initialize.
 *
 * @param		TSB_TSPI_TypeDef * const	TSPIx		Select the tspi channel.
 *
 * @return		none
 *********************************************************************************************
 */
void init_Dac(TSB_TSPI_TypeDef* const TSPIx)
{
	/* Single transfer */
	uint8_t transfer_type = 0U;			/* 0: Single transfer   1: Burst transfer 1Byte  */

	if (TSPIx == TSB_TSPI1)
	{
		CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_TSPI1, ENABLE);
	}
	else if (TSPIx == TSB_TSPI0)
	{
		CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_TSPI0, ENABLE);
	}
	else{
	}

	/* maximum clock: clk = phiT0/2, divider=1 */
	TSPI_BaudClock clk = TSPI_BR_CLOCK_1;
	TSPI_Divider divider = TSPI_BR_DIVIDER_1;	/* CLK 20MHz@80MHz */

	TSPI_InitTypeDef init = { 0U };

	TSPI_Enable(TSPIx);
	TSPI_SWReset(TSPIx);

	TSPI_SetTransferNum(TSPIx, transfer_type);
	TSPI_SetTxRxCtrl(TSPIx, ENABLE);
	TSPI_SelectMode(TSPIx, TSPI_MODE_SIO);
	TSPI_SelectTransferMode(TSPIx, TSPI_TRMODE_TX);

	TSPI_SetFIFOLevelINT(TSPIx, TSPI_TX, 0U);
	TSPI_SetFIFOLevelINT(TSPIx, TSPI_RX, 1U);

	/* enable TSPI interrupt */
	TSPI_SetINT(TSPIx, TSPI_INT_TXEND, ENABLE);

	TSPI_SetBaudRate(TSPIx, clk, divider);

	init.DataDirection = TSPI_MSB_FIRST;
	init.FrameLength = 24U;
	init.CycleBetweenFrames = 0U;
	init.ClockPolarity = TSPI_CLKPOL_HIGH;
	init.ClockPhase = TSPI_CLOCKPHASE_SECONDEDGE;
	init.ShortestIdleCycle = 1U;
	init.CS0Polarity = TSPI_CS_POLARITY_NEGATIVE;
	init.CS1Polarity = TSPI_CS_POLARITY_NEGATIVE;
	init.CS2Polarity = TSPI_CS_POLARITY_NEGATIVE;
	init.CS3Polarity = TSPI_CS_POLARITY_NEGATIVE;
	init.DelayCycleNum_CS_SCLK = 1U;
	init.DelayCycleNum_Negate_CS = 1U;
	init.ParityCheck = DISABLE;
	init.Parity = TSPI_PARITY_ODD;
	init.LastBit_Hold_Time = 0U;

	TSPI_Init(TSPIx, &init);

	if (TSPIx == TSB_TSPI1)
	{
		NVIC_ClearPendingIRQ(INTSC1RX_IRQn);
		NVIC_ClearPendingIRQ(INTSC1TX_IRQn);
		NVIC_ClearPendingIRQ(INTSC1ERR_IRQn);

		/* TSPI1TXD:PG5, TSPI1CLK:PG6 */
		CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_PORTG, ENABLE);
		TSB_PG->FR1 |= ((1 << 6u) | (1 << 5u));
		TSB_PG->CR |= ((1 << 6u) | (1 << 5u));
	}
	else
	{
		/* Do nothing */
	}

	/* Setting ~SYNC port */
	PORT_DAC_SYNC_DATA_INI;
	PORT_DAC_SYNC_CR_INI;

	DacLocalWriteInit(TSPIx);

	NVIC_SetPriority(DAC_IRQ_INTTX, DAC_IRQ_LEVEL); 		/* Interruption level set */
	NVIC_ClearPendingIRQ(DAC_IRQ_INTTX);
	NVIC_EnableIRQ(DAC_IRQ_INTTX);							/* INTTX enable */

}


/**
 *********************************************************************************************
 * @brief		D/A initial output.
 *
 * @param		TSB_TSPI_TypeDef * const	TSPIx		Select the tspi channel.
 *
 * @return		none
 *********************************************************************************************
 */
static void DacLocalWriteInit(TSB_TSPI_TypeDef* const TSPIx)
{

	PORT_DAC_SYNC = 0;										/* begin tx */

	NVIC_ClearPendingIRQ(DAC_IRQ_INTTX);
	TSPIx->DR = cDAC_POWERRESET;
	while (!NVIC_GetPendingIRQ(DAC_IRQ_INTTX));

	PORT_DAC_SYNC = 1;										/* end tx */

	PORT_DAC_SYNC = 0;										/* begin tx */

	NVIC_ClearPendingIRQ(DAC_IRQ_INTTX);
	TSPIx->DR = cDAC_LDAC_NORMAL;
	while (!NVIC_GetPendingIRQ(DAC_IRQ_INTTX));

	PORT_DAC_SYNC = 1;										/* end tx */
}


/**
 *********************************************************************************************
 * @brief		D/A data output start.
 *
 * @param		TSB_TSPI_TypeDef * const	TSPIx		Select the tspi channel.
 *
 * @return		none
 *********************************************************************************************
 */
void DacWriteStart(TSB_TSPI_TypeDef* const TSPIx)
{
	if (DacNo >= DAC_MAX_CH)
	{
		DacNo = 0;
		DacWriteContinue(TSPIx);
	}
}


/**
 *********************************************************************************************
 * @brief		D/A data output continue.
 *
 * @param		TSB_TSPI_TypeDef * const	TSPIx		Select the tspi channel.
 *
 * @return		none
 *********************************************************************************************
 */
void DacWriteContinue(TSB_TSPI_TypeDef* const TSPIx)
{
	PORT_DAC_SYNC = 1;									/* end tx */

	if (DacNo < DAC_MAX_CH)
	{
		uint32_t txdata;
		
		PORT_DAC_SYNC = 0;								/* begin tx */
		if (DacNo != (DAC_MAX_CH - 1))
		{
			txdata = (DAC_CMD_W_INPUT << 19U) | (DacNo << 16U) | DacData[DacNo];
		}
		else
		{
			/* Software LDAC */
			txdata = (DAC_CMD_W_INPUT_UP_ALL << 19U) | (DacNo << 16U) | DacData[DacNo];
		}
		TSPIx->DR = txdata;
		DacNo++;
	}
}

/*********************************** END OF FILE ******************************/
