/**
 ****************************************************************************
 * @file    ipdrv_cg.c
 * @brief   This file provides API functions for M4Kx CG driver
 * @version V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2023
 *****************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "ipdrv_cg.h"
#include "ipdrv_adc.h"

/** @addtogroup TX04_Periph_Driver
  * @{
  */

/** @defgroup CG
  * @brief CG driver modules
  * @{
  */

/** @defgroup CG_Private_Defines
  * @{
  */

#define CG_FSYSMENB_WRITE1_MASK              ((uint32_t)0x70000000)
#define CG_FCEN_WRITE0_MASK                  ((uint32_t)0x1C800000)
#define ADC_ST_ADBF_MASK                     ((uint32_t)0x00000080)

/** @} */
/* End of group CG_Private_Defines */

/** @defgroup CG_Private_FunctionPrototypes
  * @{
  */

/** @} */
/* End of group CG_Private_FunctionPrototypes */

/** @defgroup CG_Private_Functions
  * @{
  */

/** @} */
/* End of group CG_Private_Functions */

/** @defgroup CG_Exported_Functions
  * @{
  */


 /**
   * @brief  Enable or disable supplying clock fsys to peripheries
   * @param  Periph: The target peripheral of CG supplies clock
   *   This parameter can be one of the following values or their combination:
   * CG_FSYS_PERIPH_PORTA, CG_FSYS_PERIPH_PORTB, CG_FSYS_PERIPH_PORTC
   * , CG_FSYS_PERIPH_PORTD, CG_FSYS_PERIPH_PORTE, CG_FSYS_PERIPH_PORTF
   * , CG_FSYS_PERIPH_PORTG, CG_FSYS_PERIPH_PORTH, CG_FSYS_PERIPH_PORTJ
   * , CG_FSYS_PERIPH_PORTK, CG_FSYS_PERIPH_PORTL, CG_FSYS_PERIPH_PORTM
   * , CG_FSYS_PERIPH_PORTN, CG_FSYS_PERIPH_PORTP, CG_FSYS_PERIPH_PORTR
   * , CG_FSYS_PERIPH_PORTT, CG_FSYS_PERIPH_PORTU, CG_FSYS_PERIPH_PORTV
   * , CG_FSYS_PERIPH_PORTW, CG_FSYS_PERIPH_TSPI0, CG_FSYS_PERIPH_TSPI1
   * , CG_FSYS_PERIPH_UART0, CG_FSYS_PERIPH_UART1, CG_FSYS_PERIPH_UART2
   * , CG_FSYS_PERIPH_UART3, CG_FSYS_PERIPH_I2C0, CG_FSYS_PERIPH_I2C1
   * , CG_FSYS_PERIPH_CAN, CG_FSYS_PERIPH_T32A0, CG_FSYS_PERIPH_T32A1
   * , CG_FSYS_PERIPH_T32A2, CG_FSYS_PERIPH_T32A3
   * @param  NewState: New state of clock supply setting.
   *   This parameter can be one of the following values:
   *   DISABLE or ENABLE
   * @retval None
   */
void CG_SetFsysSupplyPeriphA(uint32_t Periph, FunctionalState NewState)
{
    assert_param(IS_FUNCTIONAL_STATE(NewState));
    assert_param(IS_CG_FSYS_PERIPHA(Periph));

    if (NewState == ENABLE) {   /* write '1' to enable */
        TSB_CG->FSYSMENA |= Periph;
    } else {                    /* clear to '0' to disable */
        TSB_CG->FSYSMENA &= ~Periph;
    }
}

 /**
   * @brief  Enable or disable supplying clock fsys to peripheries
   * @param  Periph: The target peripheral of CG supplies clock
   *   This parameter can be one of the following values or their combination:   
   * CG_FSYS_PERIPH_T32A4, CG_FSYS_PERIPH_T32A5, CG_FSYS_PERIPH_ADC_A
   * , CG_FSYS_PERIPH_ADC_B, CG_FSYS_PERIPH_ADC_C, CG_FSYS_PERIPH_AMP
   * , CG_FSYS_PERIPH_A_ENC0, CG_FSYS_PERIPH_A_ENC1, CG_FSYS_PERIPH_A_ENC2
   * , CG_FSYS_PERIPH_PMD0, CG_FSYS_PERIPH_PMD1, CG_FSYS_PERIPH_PMD2
   * , CG_FSYS_PERIPH_A_VE_P, CG_FSYS_PERIPH_RAM_PARI1, CG_FSYS_PERIPH_OFD
   * , CG_FSYS_PERIPH_TRIM, CG_FSYS_PERIPH_TRGSEL, CG_FSYS_PERIPH_DMAC
   * , CG_FSYS_PERIPH_BIT29, CG_FSYS_PERIPH_NBD, CG_FSYS_PERIPH_SIWDT
   * @param  NewState: New state of clock supply setting.
   *   This parameter can be one of the following values:
   *   DISABLE or ENABLE
   * @retval None
   */
void CG_SetFsysSupplyPeriphB(uint32_t Periph, FunctionalState NewState)
{
    uint32_t regval = TSB_CG->FSYSMENB;

    assert_param(IS_FUNCTIONAL_STATE(NewState));
    assert_param(IS_CG_FSYS_PERIPHB(Periph));

    if (NewState == ENABLE) {   /* write '1' to enable */
        regval |= Periph;
    } else {                    /* clear to '0' to disable */
        regval &= ~Periph;
    }

    regval |= CG_FSYSMENB_WRITE1_MASK;  /* CGFSYSMENB bit 30-28 write "1" */
    TSB_CG->FSYSMENB = regval;
}


 /**
   * @brief  Enable or disable supplying clock fsys to peripheries
   * @param  Periph: The target peripheral of CG supplies clock
   *   This parameter can be one of the following values or their combination:
   * CG_FSYS_PERIPH_RAM_PARI0, CG_FSYS_PERIPH_CRC
   * @param  NewState: New state of clock supply setting.
   *   This parameter can be one of the following values:
   *   DISABLE or ENABLE
   * @retval None
   */
void CG_SetFsysSupplyPeriph(uint32_t Periph, FunctionalState NewState)
{
    assert_param(IS_FUNCTIONAL_STATE(NewState));
    assert_param(IS_CG_FSYS_PERIPH(Periph));

    if (NewState == ENABLE) {   /* write '1' to enable */
        TSB_CG->FSYSENA |= Periph;
    } else {                    /* clear to '0' to disable */
        TSB_CG->FSYSENA &= ~Periph;
    }
}


 /**
   * @brief  Enable or disable supplying clock fc to peripheries
   * @param  Periph: The target peripheral of CG supplies clock
   *   This parameter can be one of the following values or their combination:   
   *   CG_FC_PERIPH_DNF_A, CG_FC_PERIPH_DNF_B, CG_FC_PERIPH_DNF_C, CG_FC_PERIPH_ALL.
   * @param  NewState: New state of clock supply setting.
   *   This parameter can be one of the following values:
   *   DISABLE or ENABLE
   * @retval None
   */
void CG_SetFcSupplyPeriph(uint32_t Periph, FunctionalState NewState)
{
    uint32_t regval = TSB_CG->FCEN;

    assert_param(IS_FUNCTIONAL_STATE(NewState));
    assert_param(IS_CG_FC_PERIPH(Periph));

    if (NewState == ENABLE) {   /* write '1' to enable */
        regval |= Periph;
    } else {                    /* clear to '0' to disable */
        regval &= ~Periph;
    }

    regval &= CG_FCEN_WRITE0_MASK;  /* CGFCEN bit 31-29,25-24,22-0 write "0" */
    TSB_CG->FCEN = regval;
}


/**
  * @brief  Enable or disable supplying clock (fc or fplladc) for ADC.
  * @param  ADx: Select ADC unit
  *   This parameter can be one of the following values:
  *   TSB_ADA, TSB_ADB or TSB_ADC.
  * @param  NewState: New state of clock (fc or fplladc) supply setting for ADC.
  *   This parameter can be one of the following values:
  *   ENABLE or DISABLE.
  * @retval None.
  */
void CG_SetAdcClkSupply(TSB_AD_TypeDef * ADx, FunctionalState NewState)
{
    /* Check the parameters */
    assert_param(IS_ADC_UNIT(ADx));
    assert_param(IS_FUNCTIONAL_STATE(NewState));

    /* Check ADC is not during conversion */
    while ((ADx->ST & ADC_ST_ADBF_MASK) != 0U) {
        /* Do nothing */
    }

    /* Set CGSPCLKEN<ADCKEN>  */
    if (NewState == ENABLE) {
		if (ADx == TSB_ADA) {
	        TSB_CG_SPCLKEN_ADCKEN0 = 1U;
		} else if (ADx == TSB_ADB) {
	        TSB_CG_SPCLKEN_ADCKEN1 = 1U;
		} else {
	        TSB_CG_SPCLKEN_ADCKEN2 = 1U;
		}
    } else {
		if (ADx == TSB_ADA) {
	        TSB_CG_SPCLKEN_ADCKEN0 = 0U;
		} else if (ADx == TSB_ADB) {
	        TSB_CG_SPCLKEN_ADCKEN1 = 0U;
		} else {
    	    TSB_CG_SPCLKEN_ADCKEN2 = 0U;
		}
    }
}


/**
  * @brief  Enable or disable supplying clock for Trace.
  * @param  NewState: New state of clock supply setting for Trace.
  *   This parameter can be one of the following values:
  *   ENABLE or DISABLE.
  * @retval None.
  */
void CG_SetTraceClkSupply(FunctionalState NewState)
{
    /* Check the parameters */
    assert_param(IS_FUNCTIONAL_STATE(NewState));

    /* Set CGSPCLKEN<ADCKEN>  */
    if (NewState == ENABLE) {
        TSB_CG_SPCLKEN_TRCKEN = 1U;
    } else {
        TSB_CG_SPCLKEN_TRCKEN = 0U;
    }
}



/** @} */
/* End of group CG_Exported_Functions */

/** @} */
/* End of group CG */

/** @} */
/* End of group TX04_Periph_Driver */

