/**
 *******************************************************************************
 * @file    ipdrv_tspi.h
 * @brief   This file provides all the functions prototypes for M4Ky TSPI driver.
 * @version V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 *
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __IPDRV_TSPI_H
#define __IPDRV_TSPI_H

#ifdef __cplusplus
extern "C" {
#endif

/* Includes ------------------------------------------------------------------*/
#include "TMPM4KyA.h"
#include "ipdrv_common.h"

/** @addtogroup TX04_Periph_Driver
  * @{
  */

/** @addtogroup TSPI
  * @{
  */

/** @defgroup TSPI_Exported_Types
  * @{
  */

#define IS_TSPI_UNIT(param)            (((param) == TSB_TSPI0) || \
                                        ((param) == TSB_TSPI1))

#define TSPI_TX                         (uint8_t)0x00
#define TSPI_RX                         (uint8_t)0x01
#define IS_TSPI_TXRX(param)             ((param) <= TSPI_RX)

#define IS_TSPI_FIFOLEVELINT(param)     ((param) <= 8U)

/* for CR2 bit2, 4-7 : INT control */
#define TSPI_INT_ERR       (uint32_t)0x04       /* Parity error interrupt control */
#define TSPI_INT_RXEND     (uint32_t)0x10       /* Receive completation interrupt control */
#define TSPI_INT_RXFIFO    (uint32_t)0x20       /* Receive FIFO interrupt control */
#define TSPI_INT_TXEND     (uint32_t)0x40       /* Transmit completation interrupt control */
#define TSPI_INT_TXFIFO    (uint32_t)0x80       /* Transmit FIFO interrupt control */
#define TSPI_INT_ALL       (uint32_t)0xF4       /* All above interrupt control */
#define IS_TSPI_INTSRC(param)          (((param) & TSPI_INT_ALL) && \
                                       (!((param) & (uint32_t) (~TSPI_INT_ALL))))

    typedef enum {
        TSPI_MODE_SPI = 0U,
        TSPI_MODE_SIO = 1U
    } TSPI_Mode;
#define IS_TSPI_MODE(param)             ((param) <= TSPI_MODE_SIO)

    typedef enum {
        TSPI_TRMODE_TX = 1U,    /* Half duplex (Transmit) mode */
        TSPI_TRMODE_RX = 2U,    /* Half duplex (Receive) mode  */
        TSPI_TRMODE_TXRX = 3U   /* Full duplex mode */
    } TSPI_TransferMode;
#define IS_TSPI_TRANSFERMODE(param)     (((param)>= TSPI_TRMODE_TX) && \
                                         ((param) <= TSPI_TRMODE_TXRX))
    typedef enum {
        TSPI_BR_CLOCK_0   = 0U,     /* phiT0 */
        TSPI_BR_CLOCK_1   = 1U,     /* phiT0/2 */
        TSPI_BR_CLOCK_2   = 2U,     /* phiT0/4 */
        TSPI_BR_CLOCK_4   = 3U,     /* phiT0/8 */
        TSPI_BR_CLOCK_8   = 4U,     /* phiT0/16 */
        TSPI_BR_CLOCK_16  = 5U,     /* phiT0/32 */
        TSPI_BR_CLOCK_32  = 6U,     /* phiT0/64 */
        TSPI_BR_CLOCK_64  = 7U,     /* phiT0/128 */
        TSPI_BR_CLOCK_128 = 8U,     /* phiT0/256 */
        TSPI_BR_CLOCK_256 = 9U      /* phiT0/512 */
    } TSPI_BaudClock;
#define IS_TSPI_BAUDCLOCK(param)              ((param) <= TSPI_BR_CLOCK_256)

    typedef enum {
        TSPI_BR_DIVIDER_16 = 0U,       /* 1/16 */
        TSPI_BR_DIVIDER_1  = 1U,       /* 1/1 */
        TSPI_BR_DIVIDER_2  = 2U,       /* 1/2 */
        TSPI_BR_DIVIDER_3  = 3U,       /* 1/3 */
        TSPI_BR_DIVIDER_4  = 4U,       /* 1/4 */
        TSPI_BR_DIVIDER_5  = 5U,       /* 1/5 */
        TSPI_BR_DIVIDER_6  = 6U,       /* 1/6 */
        TSPI_BR_DIVIDER_7  = 7U,       /* 1/7 */
        TSPI_BR_DIVIDER_8  = 8U,       /* 1/8 */
        TSPI_BR_DIVIDER_9  = 9U,       /* 1/9 */
        TSPI_BR_DIVIDER_10 = 10U,      /* 1/10 */
        TSPI_BR_DIVIDER_11 = 11U,      /* 1/11 */
        TSPI_BR_DIVIDER_12 = 12U,      /* 1/12 */
        TSPI_BR_DIVIDER_13 = 13U,      /* 1/13 */
        TSPI_BR_DIVIDER_14 = 14U,      /* 1/14 */
        TSPI_BR_DIVIDER_15 = 15U       /* 1/15 */
    } TSPI_Divider;
#define IS_TSPI_DIVIDER(param)                ((param) <= TSPI_BR_DIVIDER_15)

#define TSPI_LSB_FIRST               (uint8_t)0x00
#define TSPI_MSB_FIRST               (uint8_t)0x01

#define TSPI_CS_POLARITY_NEGATIVE    (uint8_t)0x00
#define TSPI_CS_POLARITY_POSITIVE    (uint8_t)0x01

#define TSPI_CLOCKPHASE_FIRSTEDGE    (uint8_t)0x00
#define TSPI_CLOCKPHASE_SECONDEDGE   (uint8_t)0x01

#define TSPI_CLKPOL_LOW              (uint8_t)0x00
#define TSPI_CLKPOL_HIGH             (uint8_t)0x01

#define TSPI_PARITY_EVEN             (uint8_t)0x00
#define TSPI_PARITY_ODD              (uint8_t)0x01

#define TSPI_HOLDTIME_FC_DIVIDE_2    (uint8_t)0x00
#define TSPI_HOLDTIME_FC_DIVIDE_4    (uint8_t)0x01
#define TSPI_HOLDTIME_FC_DIVIDE_8    (uint8_t)0x02
#define TSPI_HOLDTIME_FC_DIVIDE_16   (uint8_t)0x03
#define TSPI_HOLDTIME_FC_DIVIDE_32   (uint8_t)0x04
#define TSPI_HOLDTIME_FC_DIVIDE_64   (uint8_t)0x05
#define TSPI_HOLDTIME_FC_DIVIDE_128  (uint8_t)0x06

    typedef struct {
        uint8_t DataDirection;  /* LSB or MSB first */
        uint8_t FrameLength;    /* 8 to 32 bits */
        uint8_t CycleBetweenFrames;     /* 0 to 15 cycles (Interval cycle between frames in burst mode) */
        uint8_t CS3Polarity;    /* Negative logic or Positive logic for pin CS3 */
        uint8_t CS2Polarity;    /* Negative logic or Positive logic for pin CS2 */
        uint8_t CS1Polarity;    /* Negative logic or Positive logic for pin CS1 */
        uint8_t CS0Polarity;    /* Negative logic or Positive logic for pin CS0 */
        uint8_t ClockPhase;     /* The data sampling phase */
        uint8_t ClockPolarity;  /* Level of serial clock during idle time */
        uint8_t ShortestIdleCycle;      /* 1 to 15 cycles */
        uint8_t DelayCycleNum_CS_SCLK;  /* 1 to 16 (Delay cycle from asserting TSPIxCS to output serial clock) */
        uint8_t DelayCycleNum_Negate_CS;  /* 1 to 16 (Delay cycle of negating TSPIxCS) */
        uint8_t LastBit_Hold_Time;      /* Last bit holding time of TSPIxTXD pin in SIO slave mode */
        FunctionalState ParityCheck;    /* ENABLE or DISABLE */
        uint8_t Parity;         /* TSPI_PARITY_ODD or TSPI_PARITY_EVEN */
    } TSPI_InitTypeDef;

#define IS_TSPI_DATADIRECTION(param)        ((param) <= TSPI_MSB_FIRST)
#define IS_TSPI_FRAMELENGTH(param)          (((param)>= 8U) && ((param) <= 32U))
#define IS_TSPI_CYCLEBETWEENFRAMES(param)   ((param) <= 15U)
#define IS_TSPI_CSPOLARITY(param)           ((param) <= TSPI_CS_POLARITY_POSITIVE)
#define IS_TSPI_CLOCKPHASE(param)           ((param) <= TSPI_CLOCKPHASE_SECONDEDGE)
#define IS_TSPI_CLOCKPOLARITY(param)        ((param) <= TSPI_CLKPOL_HIGH)
#define IS_TSPI_SHORTESTIDLECYCLE(param)    (((param)>= 1U) && ((param) <= 15U))
#define IS_TSPI_DELAYCYCLENUM(param)        (((param)>= 1U) && ((param) <= 16U))
#define IS_TSPI_PARITYTYPE(param)           ((param) <= TSPI_PARITY_ODD)
#define IS_TSPI_HOLDTIME(param)             ((param) <= TSPI_HOLDTIME_FC_DIVIDE_128)

/**
  * @brief Union to store TSPI status and store TSPI result
  */
    typedef union {
        uint32_t All;
        struct {
            uint32_t RxFIFOFillLevel:4; /* bit3-0   : Receive FIFO fill level */
            uint32_t RxFIFOFull:1;      /* bit4: '1': Receive FIFO is full */
            uint32_t RxFIFO_INT:1;      /* bit5: '1': Receive FIFO interrupt generated */
            uint32_t RxCompleted:1;     /* bit6: '1': Reception completed */
            uint32_t RxRUN:1;           /* bit7: '1': Receive shift is operating */
            uint32_t Reserved1:8;
            uint32_t TxFIFOFillLevel:4; /* bit19-16  : Transmit FIFO fill level */
            uint32_t TxFIFOEmpty:1;     /* bit20: '1': Transmit FIFO is empty  */
            uint32_t TxFIFO_INT:1;      /* bit21: '1': Transmit FIFO interrupt generated */
            uint32_t TxCompleted:1;     /* bit22: '1': Transmission completed */
            uint32_t TxRUN:1;           /* bit23: '1': Transmit shift is operating */
            uint32_t Reserved2:7;
            uint32_t TSPI_RegUsing:1;   /* bit31: '1': TSPI registers must not be modified */
        } Bit;
    } TSPI_StatusFlag;

/** @} */
/* End of group TSPI_Exported_Types */

/** @defgroup TSPI_Exported_FunctionPrototypes
  * @{
  */
    void TSPI_SWReset(TSB_TSPI_TypeDef * TSPIx);
    void TSPI_Enable(TSB_TSPI_TypeDef * TSPIx);
    void TSPI_SetTxRxCtrl(TSB_TSPI_TypeDef * TSPIx, FunctionalState NewState);
    void TSPI_SelectMode(TSB_TSPI_TypeDef * TSPIx, TSPI_Mode Mode);
    void TSPI_SelectTransferMode(TSB_TSPI_TypeDef * TSPIx, TSPI_TransferMode TrMode);
    void TSPI_SetTransferNum(TSB_TSPI_TypeDef * TSPIx, uint8_t Num);
    void TSPI_SetFIFOLevelINT(TSB_TSPI_TypeDef * TSPIx, uint8_t TxRx, uint8_t Level);
    void TSPI_SetINT(TSB_TSPI_TypeDef * TSPIx, uint32_t IntSrc, FunctionalState NewState);
    void TSPI_SetBaudRate(TSB_TSPI_TypeDef * TSPIx, TSPI_BaudClock Clk, TSPI_Divider Divider);
    void TSPI_Init(TSB_TSPI_TypeDef * TSPIx, TSPI_InitTypeDef * Init);
    TSPI_StatusFlag TSPI_GetStatus(TSB_TSPI_TypeDef * TSPIx);

/** @} */
/* End of group TSPI_Exported_FunctionPrototypes */

/** @} */
/* End of group TSPI */

/** @} */
/* End of group TX04_Periph_Driver */

#ifdef __cplusplus
}
#endif
#endif                          /* __IPDRV_TSPI_H */
