/**
 ****************************************************************************
 * @file    Ipdrv_flash.c
 * @brief   FLASH_DATA Sample Application.
 * @version V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2023
 *****************************************************************************
 */

#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <string.h>
#include "TMPM4KLA.h"  /*!< TMPM4Ky Group Header file. */
#include "ipdrv_flash.h"
#include "ipdrv_siwdt.h"
/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup FLASH_DATA FLASH_DATA Sample Appli
 *  @{
 */

/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup FLASH_DATA_Private_define FLASH_DATA Private Define
 *  @{
 */
#define FC_KCR_KEYCODE      (0xA74A9D23UL)              /*!< The specific code that writes the FCKCR register. */
#define FC_BRANK_VALUE      (uint32_t)(0xFFFFFFFFUL)    /*!< Brank value */
#define FC_MAX_PAGES        (uint8_t)(0x21)             /*!< Maxmum pages */
#define FC_MAX_BLOCKS       (uint8_t)(0x16)             /*!< Maxmum blocks */
#define FC_MAX_AREAS        (uint8_t)(0x1)              /*!< Maxmum areas */
#define FC_MAX_DATA_PAGES   (uint8_t)(0x21)             /*!< Maxmum pages */
#define FC_MAX_DATA_BLOCKS  (uint8_t)(0x8)              /*!< Maxmum blocks */
#define FC_CMD_ADDRESS_MASK (uint32_t)(0xFFFF0000UL)    /*!< Upper address mask of the upper address */
#define FC_CMD_BC1_ADDR     (0x00000550UL)  /*!< The lower address of the first bus cycle when uses commans */
#define FC_CMD_BC2_ADDR     (0x00000AA0UL)  /*!< The lower address of the second bus cycle when uses commans */

/* FCSR0 register */
#define     FC_AREASEL_EXPECT_AREA0 (uint32_t)(0x00000000UL)    /*!< RW, Selects expect area0       */
#define     FC_AREASEL_EXPECT_AREA4 (uint32_t)(0x00000000UL)    /*!< RW, Selects expect area0       */
#define     FC_AREASEL_AREA0        (uint32_t)(0x00000007UL)    /*!< RW, Selects area0              */
#define     FC_AREASEL_AREA4        (uint32_t)(0x00070000UL)    /*!< RW, Selects area4              */
#define     FC_AREASEL_MASK_AREA0   (uint32_t)(0xFF8F0FF8UL)    /*!< RW, Selects area0              */
#define     FC_AREASEL_MASK_AREA4   (uint32_t)(0xFF880FFFUL)    /*!< RW, Selects area0              */
#define     FC_AREASEL_WRITE_MODE   (uint32_t)(0x04000000UL)    /*!< R,  Write Mode                 */
#define     FC_AREASEL4_WRITE_MODE  (uint32_t)(0x40000000UL)    /*!< R,  Write Mode                 */

static uint32_t fc_const_data_flash_address[FC_MAX_DATA_PAGES] = {
    (0x30000000UL),   /*!< DATA FLASH Page0 */
    (0x30000100UL),   /*!< DATA FLASH Page1 */
    (0x30000200UL),   /*!< DATA FLASH Page2 */
    (0x30000300UL),   /*!< DATA FLASH Page3 */
    (0x30000400UL),   /*!< DATA FLASH Page4 */
    (0x30000500UL),   /*!< DATA FLASH Page5 */
    (0x30000600UL),   /*!< DATA FLASH Page6 */
    (0x30000700UL),   /*!< DATA FLASH Page7 */
    (0x30000800UL),   /*!< DATA FLASH Page8 */
    (0x30000900UL),   /*!< DATA FLASH Page9 */
    (0x30000A00UL),   /*!< DATA FLASH Page10 */
    (0x30000B00UL),   /*!< DATA FLASH Page11 */
    (0x30000C00UL),   /*!< DATA FLASH Page12 */
    (0x30000D00UL),   /*!< DATA FLASH Page13 */
    (0x30000E00UL),   /*!< DATA FLASH Page14 */
    (0x30000F00UL),   /*!< DATA FLASH Page15 */
    (0x30001000UL),   /*!< DATA FLASH Page16 */
    (0x30001100UL),   /*!< DATA FLASH Page17 */
    (0x30001200UL),   /*!< DATA FLASH Page18 */
    (0x30001300UL),   /*!< DATA FLASH Page19 */
    (0x30001400UL),   /*!< DATA FLASH Page20 */
    (0x30001500UL),   /*!< DATA FLASH Page21 */
    (0x30001600UL),   /*!< DATA FLASH Page22 */
    (0x30001700UL),   /*!< DATA FLASH Page23 */
    (0x30001800UL),   /*!< DATA FLASH Page24 */
    (0x30001900UL),   /*!< DATA FLASH Page25 */
    (0x30001A00UL),   /*!< DATA FLASH Page26 */
    (0x30001B00UL),   /*!< DATA FLASH Page27 */
    (0x30001C00UL),   /*!< DATA FLASH Page28 */
    (0x30001D00UL),   /*!< DATA FLASH Page29 */
    (0x30001E00UL),   /*!< DATA FLASH Page30 */
    (0x30001F00UL),   /*!< DATA FLASH Page31 */
    (0x30002000UL)    /*!< DATA FLASH Page32 */
};

static uint32_t fc_const_data_flash_block_address[FC_MAX_DATA_BLOCKS] = {
    (0x30000000UL),   /*!< DATA FLASH Block0 */
    (0x30001000UL),   /*!< DATA FLASH Block1 */
    (0x30002000UL),   /*!< DATA FLASH Block2 */
    (0x30003000UL),   /*!< DATA FLASH Block3 */
    (0x30004000UL),   /*!< DATA FLASH Block4 */
    (0x30005000UL),   /*!< DATA FLASH Block5 */
    (0x30006000UL),   /*!< DATA FLASH Block6 */
    (0x30007000UL)    /*!< DATA FLASH Block7 */
};

static uint32_t fc_const_data_flash_area_address[FC_MAX_AREAS] = {
    (0x30000000UL)   /*!< DATA FLASH AREA0 */
};

 /** 
 *  @defgroup  TXZ_RESULT TXZ_RESULT
 *  @brief     Result
 *  @{
 */
#define TXZ_SUCCESS                     ((uint32_t)0x00000000)    /*!< Success                 */
#define TXZ_ERROR                       ((uint32_t)0x00000001)    /*!< Error                   */
/** 
 *  @}
 */ /* End of group TXZ_RESULT */
 /** 
 *  @defgroup  TXZ_STATE TXZ_STATE
 *  @brief     Status
 *  @{
 */
#define TXZ_DONE                        ((uint32_t)0x00000000)    /*!< Done                    */
#define TXZ_BUSY                        ((uint32_t)0x00000001)    /*!< Busy                    */
/** 
 *  @}
 */ /* End of group TXZ_STATE */
/** 
 *  @}
 */ /* End of group FLASH_DATA_Private_define */

/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup FLASH_DATA_Private_fuctions FLASH_DATA Private Fuctions
 *  @{
 */
__STATIC_INLINE uint32_t fc_enable_areasel4(void);
__STATIC_INLINE uint32_t fc_disable_areasel4(void);
static void fc_write_command(uint32_t* src_address, uint32_t* dst_address, uint32_t size);
static void fc_erase_command(uint32_t* flash_top_address, uint32_t* erase_top_address, fc_erase_kind_t kind);

/*--------------------------------------------------*/
/** 
  * @brief  Enables the AREA4.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  *         This function write the FCAREASEL regiset.
  */
/*--------------------------------------------------*/
__STATIC_INLINE uint32_t fc_enable_areasel4(void)
{
    uint32_t retval = TXZ_ERROR;
    uint32_t reg = TSB_FC->AREASEL & FC_AREASEL_MASK_AREA4;

    reg |= FC_AREASEL_AREA4;
    /* Writes the FCKER register the KEYCODE. */
	__disable_irq();
	TSB_FC->KCR = FC_KCR_KEYCODE;

	/* Selects the area4 */
	TSB_FC->AREASEL = reg;
	__enable_irq();
    /* Confirms the FCAREASEL register the SSF4 was set. */
    while(1){
        uint32_t i = TSB_FC->AREASEL;
        if((i & FC_AREASEL4_WRITE_MODE) == FC_AREASEL4_WRITE_MODE){
            retval = TXZ_SUCCESS;
            break;
        }
    }

    return(retval);
}

/*--------------------------------------------------*/
/** 
  * @brief  Disables the AREA4.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  *         This function write the FCAREASEL regiset.
  */
/*--------------------------------------------------*/
__STATIC_INLINE uint32_t fc_disable_areasel4(void)
{
    uint32_t retval = TXZ_ERROR;
    uint32_t reg = TSB_FC->AREASEL & FC_AREASEL_MASK_AREA4;

    /* Writes the FCKER register the KEYCODE. */
	__disable_irq();
	TSB_FC->KCR = FC_KCR_KEYCODE;

	/* Selects the area4 */
	TSB_FC->AREASEL = FC_AREASEL_EXPECT_AREA4;
	__enable_irq();

    /* Confirms the SSF0 of the FCAREASEL  register is not set. */
    while(1){
        uint32_t i = TSB_FC->AREASEL;
        if((i & FC_AREASEL4_WRITE_MODE) != FC_AREASEL4_WRITE_MODE){
            retval = TXZ_SUCCESS;
            break;
        }
    }

    return(retval);
}
/*--------------------------------------------------*/
/** 
  * @brief  Writes data of the DATA Flash.
  * @param  src_address  :source address
  * @param  dst_address  :destination address
  * @param  size         :data size
  * @return -
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
static void fc_write_command(uint32_t* src_address, uint32_t* dst_address, uint32_t size)
{
    uint32_t retval;
    volatile uint32_t* addr1;
    volatile uint32_t* addr2;
    volatile uint32_t* addr3;
    uint32_t* source = (uint32_t *) src_address;

    addr1 = (uint32_t *) ((uint32_t)FC_CODE_DATA_ADDRESS_TOP + FC_CMD_BC1_ADDR);
    addr2 = (uint32_t *) ((uint32_t)FC_CODE_DATA_ADDRESS_TOP + FC_CMD_BC2_ADDR);
    addr3 = (uint32_t *) ((uint32_t)dst_address);
    /* Enables the AREA4. Write Mode. */
    retval = fc_enable_areasel4();

    if(retval == TXZ_SUCCESS){
        uint32_t i;

        *addr1 = (0x000000AAUL); /* bus cycle 1 */
        *addr2 = (0x00000055UL); /* bus cycle 2 */
        if((uint32_t)dst_address >= 0x30000000)
        {
          *addr1 = (0x000000C0UL); /* bus cycle 3 */
        }else{
          *addr1 = (0x000000A0UL); /* bus cycle 3 */
        }
        for(i=(0UL); i<size; i+=(0x4UL)){
            *addr3 = *source;
            source++;
        }

        /* Confirmation of the works start of ROM. */
        while(fc_get_status(FC_SR0_RDYBSY) == TXZ_DONE){
			SIWDT_WriteClearCode();
        };

        /* Waits for a finish of the works in the code DATA Flash. */
        while(fc_get_status(FC_SR0_RDYBSY) == TXZ_BUSY){
			SIWDT_WriteClearCode();
        };
    }

    /* Disables the AREA4. Read Mode. */
    retval = fc_disable_areasel4();
}

/*--------------------------------------------------*/
/** 
  * @brief  Auto page erase command of the flash ROM.
  * @param  flash_top_address : flash top address
  * @param  erase_top_address : erase top address
  * @param  kind              : Chip, Area, Block, Page, etc.
  * @return -
  * @note   This function erases specified place of the flash ROM.
  */
/*--------------------------------------------------*/
static void fc_erase_command(uint32_t* flash_top_address, uint32_t* erase_top_address, fc_erase_kind_t kind)
{
    uint32_t retval;
    volatile uint32_t *addr1 = (uint32_t *) ((uint32_t)flash_top_address + FC_CMD_BC1_ADDR);
    volatile uint32_t *addr2 = (uint32_t *) ((uint32_t)flash_top_address + FC_CMD_BC2_ADDR);
    volatile uint32_t *addr3 = (uint32_t *) erase_top_address;

    /* Enables the AREA4. Write Mode. */
    retval = fc_enable_areasel4();

    if(retval == TXZ_SUCCESS){
        *addr1 = (0x000000AAUL);
        *addr2 = (0x00000055UL);
        *addr1 = (0x00000080UL);
        *addr1 = (0x000000AAUL);
        *addr2 = (0x00000055UL);
        *addr3 = kind;

        /* Confirmation of the works start of ROM. */
        while(fc_get_status(FC_SR0_RDYBSY) == TXZ_DONE){
			SIWDT_WriteClearCode();
        };

        /* Waits for a finish of the works in the code DATA Flash. */
        while(fc_get_status(FC_SR0_RDYBSY) == TXZ_BUSY){
			SIWDT_WriteClearCode();
        };
    }

    /* Disables the AREA4. Read Mode. */
    retval = fc_disable_areasel4();
}

/*--------------------------------------------------*/
/** 
  * @brief  Checks a blank of the DATA Flash every 4bytes.
  * @param  addrress
  * @param  size
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
static uint32_t fc_blank_check(uint32_t* address, uint32_t size)
{
    uint32_t retval = TXZ_SUCCESS;

    uint32_t i;

    for(i=0; i<(size/sizeof(uint32_t)); i++){
        uint32_t* addr = &address[i];
        if(*addr != FC_BRANK_VALUE){
            retval = TXZ_ERROR;
            break;
        }
    }

    return (retval);
}

/**
 *  @}
 */ /* End of group FLASH_DATA_Private_fuctions */


/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup FLASH_DATA_Exported_functions FLASH_DATA Exported Functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  Get the status of the flash auto operation.
  * @param  status
  * @return Result.
  * @retval TXZ_BUSY : Busy.
  * @retval TXZ_DONE : Done.
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
uint32_t fc_get_status(fc_sr0_t status)
{
   uint32_t retval= TXZ_BUSY;
    uint32_t work32;
    /* Reads the FCSR0. Masks the other specfic status */
    work32 = TSB_FC->SR0 & (uint32_t)status;

    /* Confirms the specific status of the flash ROM */
    if(work32 == (uint32_t)status){
        retval = TXZ_DONE;
    }

    return (retval);
}

/*--------------------------------------------------*/
/** 
  * @brief  Auto write command of the data flash.
  * @param  src_address  :source address
  * @param  dst_address  :destination address
  * @param  size         :data size
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  *         This function writes 16bytes data to the DATA Flash.
  */
/*--------------------------------------------------*/
uint32_t fc_write_data_flash(uint32_t* src_address, uint32_t* dst_address, uint32_t size)
{
    uint32_t retval = TXZ_SUCCESS;

    /* Checks the code DATA Flash status */
    if(fc_get_status(FC_SR0_RDYBSY) == TXZ_DONE){

        uint32_t i;
        /* Checks the DATA Flash status */
        for(i=0;i<(size*4);i+=(uint32_t)(0x4UL)){
            /* Writes 4bytes data. */
            fc_write_command((uint32_t*)((uint32_t)src_address+i), (uint32_t*)((uint32_t)dst_address+i), (uint32_t)(0x4UL));
        }
    }

    return (retval);
}

/*--------------------------------------------------*/
/** 
  * @brief  Auto page erase command of the data flash.
  * @param  first_page      : The first page to erase
  * @param  num_of_pages    : The number of pages to erase.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  *         This function erases specified page of the data Flash and checks a blank.
  */
/*--------------------------------------------------*/
uint32_t fc_erase_page_data_flash(fc_data_flash_page_number_t first_page, uint8_t num_of_pages)
{
    uint32_t retval = TXZ_SUCCESS;

    /* Checks the data Flash status */
    if(fc_get_status(FC_SR0_RDYBSY) == TXZ_DONE){
        /* Checks the number of maximum pages. */
        if((first_page + num_of_pages) <= FC_MAX_PAGES){
            uint8_t i;
            for(i=0; i<num_of_pages ; i++){
                /* Erases the specific page. */
                fc_erase_command((uint32_t*)FC_DATA_FLASH_ADDRESS_TOP,
                                (uint32_t*)fc_const_data_flash_address[first_page+i],
                                FC_ERASE_KIND_PAGE);
            }
            /* Checks a blank of the specific page. */
            if(fc_blank_check((uint32_t*)fc_const_data_flash_address[first_page], FC_DATA_PAGE_SIZE*(uint32_t)num_of_pages) == TXZ_ERROR){
                retval = TXZ_ERROR;
            }
        }
        else{
            retval = TXZ_ERROR;
        }
    }
    else {
        retval = TXZ_ERROR;
    }

    return (retval);
}

/*--------------------------------------------------*/
/** 
  * @brief  Checks a blank of the data Flash of specified pages.
  * @param  first_page      : The first page which checks a blank.
  * @param  last_page       : The last page which checks a blank.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
uint32_t fc_blank_check_page_data_flash(fc_data_flash_page_number_t first_page, fc_data_flash_page_number_t last_page)
{
    uint32_t retval;

    uint32_t* address = (uint32_t*)fc_const_data_flash_address[first_page];
    uint32_t size = ((uint32_t)(last_page - first_page + 1) * (uint32_t)FC_DATA_PAGE_SIZE);
    
    retval = fc_blank_check(address, size);

    return (retval);
}

/*--------------------------------------------------*/
/** 
  * @brief  Auto block erase command of the data flash.
  * @param  first_block      : The first block to erase
  * @param  num_of_block     : The number of blocks to erase.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  *         This function erases specified block of the data Flash and checks a blank.
  */
/*--------------------------------------------------*/
uint32_t fc_erase_block_data_flash(fc_data_flash_block_number_t first_block, uint8_t num_of_block)
{
    uint32_t retval = TXZ_SUCCESS;

    /* Checks the data Flash status */
    if(fc_get_status(FC_SR0_RDYBSY) == TXZ_DONE){
        /* Checks the number of maximum blocks. */
        if((first_block + num_of_block) <= FC_MAX_BLOCKS){
            uint8_t i;
            for(i=0; i<num_of_block ; i++){
                /* Erases the specific block. */
                fc_erase_command((uint32_t*)FC_DATA_FLASH_ADDRESS_TOP,
                                (uint32_t*)fc_const_data_flash_block_address[first_block+i],
                                FC_ERASE_KIND_BLOCK);
            }
            /* Checks a blank of the specific block. */
            if(fc_blank_check((uint32_t*)fc_const_data_flash_block_address[first_block], FC_DATA_BLOCK_SIZE*(uint32_t)num_of_block) == TXZ_ERROR){
                retval = TXZ_ERROR;
            }
        }
        else{
            retval = TXZ_ERROR;
        }
    }
    else {
        retval = TXZ_ERROR;
    }

    return (retval);
}

/*--------------------------------------------------*/
/** 
  * @brief  Checks a blank of the data Flash of specified blocks.
  * @param  first_block      : The first block which checks a blank.
  * @param  last_block       : The last block which checks a blank..
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
uint32_t fc_blank_check_block_data_flash(fc_data_flash_block_number_t first_block, fc_data_flash_block_number_t last_block)
{
    uint32_t retval;

    uint32_t* address = (uint32_t*)fc_const_data_flash_block_address[first_block];
    uint32_t size = ((uint32_t)(last_block - first_block + 1) * (uint32_t)FC_DATA_BLOCK_SIZE);
    
    retval = fc_blank_check(address, size);

    return (retval);
}

/*--------------------------------------------------*/
/** 
  * @brief  Auto area erase command of the data flash.
  * @param  area      : The area block to erase
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  *         This function erases specified block of the data Flash and checks a blank.
  */
/*--------------------------------------------------*/
uint32_t fc_erase_area_data_flash(fc_data_flash_area_number_t area)
{
    uint32_t retval = TXZ_SUCCESS;

    /* Checks the data Flash status */
    if(fc_get_status(FC_SR0_RDYBSY) == TXZ_DONE){
        /* Checks the number of maximum blocks. */
        if(area == 0){
            /* Erases the specific block. */
            fc_erase_command((uint32_t*)FC_DATA_FLASH_ADDRESS_TOP,
                            (uint32_t*)fc_const_data_flash_area_address[area],
                            FC_ERASE_KIND_AREA);
            /* Checks a blank of the specific block. */
            if(fc_blank_check((uint32_t*)fc_const_data_flash_area_address[area], FC_DATA_AREA_SIZE) == TXZ_ERROR){
                retval = TXZ_ERROR;
            }
        }
        else{
            retval = TXZ_ERROR;
        }
    }
    else {
        retval = TXZ_ERROR;
    }

    return (retval);
}

/*--------------------------------------------------*/
/** 
  * @brief  Checks a blank of the data Flash of specified area.
  * @param  area      : The area which checks a blank.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
uint32_t fc_blank_check_area_data_flash(fc_data_flash_area_number_t area)
{
    uint32_t retval;

    uint32_t* address = (uint32_t*)fc_const_data_flash_area_address[area];
    uint32_t size = ((uint32_t)(1) * (uint32_t)FC_DATA_AREA_SIZE);
    
    retval = fc_blank_check(address, size);

    return (retval);
}
/**
 *  @}
 */ /* End of group FLASH_DATA_Exported_functions */

/**
 *  @}
 */ /* End of group FLASH_DATA */

/**
 *  @} 
 */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
