/**
 ****************************************************************************
 * @file:	 mcuip_drv.h
 * @brief:	 PMD and VE driver
 * @version  V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2023
 *****************************************************************************
 */
#ifndef _MCUIP_DRV_H
#define _MCUIP_DRV_H

/*===================================================================*
	  Register bit position Definition
 *===================================================================*/
/* VExFMODE */
#define	B_SPWMEN				1U
#define	B_C2PEN					0U

/* VExMCTLF */
#define	B_LAVF					0U
#define	B_LAVFM					1U
#define	B_PLSLF					4U
#define	B_SFT2ST				14U
#define	B_SFT2STM				15U

/* PMDxEMGSTA */
#define	B_EMGST					0U
#define	B_EMGI					1U

/* PMDxEMGCR */
#define	B_EMGEN					0U
#define	B_EMGRS					1U

/* PMDxOVVCR */
#define	B_OVVEN					0U

/* PMDxMDPOT */
#define	B_POLL					2U
#define	B_POLH					3U

/* ADxST */
#define	B_PMDF					(0U)					/* PMDF bit */

/*===================================================================*
	  Constant Data Definition
 *===================================================================*/
/* PMD Trigger Interrupt Select Registers */
#define	cPINTS_A				0U
#define	cPINTS_B				1U
#define	cPINTS_C				2U
#define	cPINTS_D				3U

/* PMD CH */
#define	cPMD0					0U
#define	cPMD1					1U
#define cPMD2					2U

/** For PMD,VE ******************************************************/

/* Define the constant data for MODESEL */
#define WBUF_BUS				0								/* W-BUF data from Bus (by software) */
#define WBUF_VE 				0x0f							/* W-BUF data from VE (automatic)	 */


/****************************************************************************************/
/* Define the constant data for MDCR */
/****************************************************************************************/
/* PWM interrupt period */
#define	INTPRD_HALF				0U								/* Interrupt request at every 0.5 PWM period (<PWMMD>= "1" only) */
#define	INTPRD_1				1U								/* Interrupt request at every PWM period */
#define	INTPRD_2				2U								/* Interrupt request at every 2 PWM periods */
#define	INTPRD_4				3U								/* Interrupt request at every 4 PWM periods */

/* PWM interrupt timing */
#define	PINT_CENTER				0U								/* PWM carrier center */
#define	PINT_END				1U								/* PWM carrier end */

/* Duty mode */
#define	DTYMD_COMMON			0U								/* 0: 3-phase common mode */
#define	DTYMD_INDEPENDENT		1U								/* 1: 3-phase independent mode */

/* Port output mode */
#define	SYNTMD_0				0U
#define	SYNTMD_1				1U

#define	DCMEN_DISABLE			0U								/* enable */
#define	DCMEN_ENABLE			1U								/* disable */

/* Deadtime correction */
#define	DTCREN_DISABLE			0U
#define	DTCREN_ENABLE			1U

/* Double buffer update timing for the duty compare register and PWM period register. */
#define	DSYNCS_INTCYC			0U								/* Depends on interrupt cycle setting */
#define	DSYNCS_CENTER			1U								/* Updates at PWM carrier center */
#define	DSYNCS_END				2U								/* Updates at PWM carrier end */
#define	DSYNCS_CENTER_END		3U								/* Updates at both PWM carrier center and end */

/* PWM carrier waveform */
#define	PWMMD_SAWTOOTH			0U								/* edge-aligned PWM, sawtooth wave */
#define	PWMMD_TRIANGULAR		1U								/* center-aligned PWM, triangular wave */
#define	PWMMD_SAWTOOTH_REV		2U
#define	PWMMD_TRIANGULAR_REV	3U


/*--------------------------------------------------------------------------------------*/
#define	cWPWMMD					PWMMD_TRIANGULAR
#define	cVPWMMD					PWMMD_TRIANGULAR
#define	cUPWMMD					PWMMD_TRIANGULAR
#define	cDSYNCS					DSYNCS_END						/* Double buffer update timing */
#define	cDTCREN					DTCREN_DISABLE					/* Deadtime correction */
#define	cDCMEN					DCMEN_DISABLE					/*   */
#define	cPWMSYNT				SYNTMD_0						/* Special PWM mode */
#define	cPWMSPCFY				DTYMD_INDEPENDENT				/* Duty mode */
#define	cPWMINT					PINT_CENTER						/* INTPWM timing */
#define	cPWMINTPRD				INTPRD_1						/* INTPWM period */

#define	cWPWMMD_PFC				PWMMD_SAWTOOTH_REV
#define	cVPWMMD_PFC				PWMMD_SAWTOOTH_REV
#define	cUPWMMD_PFC				PWMMD_SAWTOOTH_REV
#define	cPWMINT2				PINT_END
/*--------------------------------------------------------------------------------------*/
#define cMDCR_INI				( (cWPWMMD	 << 14) | (cVPWMMD	  << 12) | (cUPWMMD   << 10) \
								| (cDSYNCS	 << 8 ) | (cDTCREN	  << 7 ) \
								| (cDCMEN << 6 ) | (cPWMSYNT   << 5 ) | (cPWMSPCFY << 4) \
								| (cPWMINT	 << 3 ) | (INTPRD_1 << 1 ) )
#define cMDCR_INI_PFC			( (cWPWMMD_PFC << 14) | (cVPWMMD_PFC	  << 12) | (cUPWMMD_PFC   << 10) \
								| (cDSYNCS	 << 8 ) | (cDTCREN	  << 7 ) \
								| (cDCMEN << 6 ) | (cPWMSYNT   << 5 ) | (cPWMSPCFY << 4) \
								| (cPWMINT2	 << 3 ) | (cPWMINTPRD << 1 ) )
/****************************************************************************************/


/****************************************************************************************/
/* Define the constant data for MDOUT */
/****************************************************************************************/
#define cMDOUT_OFF				BIT16(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0)	/* OUTPUT all L */
#define cMDOUT_ON				BIT16(0,0,0,0,0,1,1,1,0,0,1,1,1,1,1,1)	/* OUTPUT all PWM/~PWM */
#if defined(__U_PFCOUT)
#define cMDOUT_PFC_ON			BIT16(0,0,0,0,0,0,0,1,0,0,0,0,0,0,1,0)
#elif defined(__V_PFCOUT)
#define cMDOUT_PFC_ON			BIT16(0,0,0,0,0,0,1,0,0,0,0,0,1,0,0,0)
#elif defined(__W_PFCOUT)
#define cMDOUT_PFC_ON			BIT16(0,0,0,0,0,1,0,0,0,0,1,0,0,0,0,0)
#endif
#define cMDOUT_BOOTSTRAP		BIT16(0,0,0,0,0,0,0,0,0,0,0,1,0,1,0,1)	/* OUTPUT short circuit brake */
								/* 0  | | | | | | | | | | | | | | | +--	UOC X out 0:OFF 1:ON	*/
								/* 1  | | | | | | | | | | | | | | +----		U out 0:OFF 1:ON	*/
								/* 2  | | | | | | | | | | | | | +------	VOC Y out 0:OFF 1:ON	*/
								/* 3  | | | | | | | | | | | +--------		V out 0:OFF 1:ON	*/
								/* 4  | | | | | | | | | | | +----------	WOC Z out 0:OFF 1:ON	*/
								/* 5  | | | | | | | | | | +------------		W out 0:OFF 1:ON	*/
								/* 6  | | | | | | | | | +--------------	0						*/
								/* 7  | | | | | | | | +----------------	0						*/
								/* 8  | | | | | | | +------------------	UPWM U out 0:H/L 1:PWM	*/
								/* 9  | | | | | | +--------------------	UPWM U out 0:H/L 1:PWM	*/
								/*10  | | | | | +----------------------	UPWM U out 0:H/L 1:PWM	*/
								/*11  | | | | +------------------------	0						*/
								/*12  | | | +--------------------------	0						*/
								/*13  | | +----------------------------	0						*/
								/*14  | +------------------------------	0						*/
								/*15  +--------------------------------	0						*/

/****************************************************************************************/

/****************************************************************************************/
/* Define the constant data for MDPOT */
/****************************************************************************************/
/* Selects PMDxMDOUT(VExOUTCR) transfer timing (trigger synchronous setting). */
#define	SYNCS_ASYNC				0U								/* asynchronous */
#define	SYNCS_INTENC			1U								/* when INTENCx (ENCx interrupt request) occurs */
#define	SYNCS_NTTB				2U								/* when INTT32Ax0 (TMRBx interrupt request) occurs */
#define	SYNCS_CTRGO				3U								/* when CTRGO(ENCx MCMP completed) occurs */

/* Selects PMDxMDOUT(VExOUTCR) and PMDxMDCR(nPWMMD) transfer timing (PWM synchronous setting). */
#define	PSYNCS_WRITE			0U								/* Reflect when write */
#define	PSYNCS_CENTER			1U								/* Reflect when PWM carrier center. */
#define	PSYNCS_END				2U								/* Reflect when PWM carrier end. */
#define	PSYNCS_CENTER_END		3U								/* Reflect when PWM carrier center and end. */

/* Port polarity */
#define	POLH_LACT				0U
#define	POLH_HACT				1U
#define	POLL_LACT				0U
#define	POLL_HACT				1U

/*--------------------------------------------------------------------------------------*/
#define	cSYNCS					(SYNCS_ASYNC)
#define	cPSYNCS					(PSYNCS_END)
/*--------------------------------------------------------------------------------------*/
#define cMDPOT_INI				((cSYNCS << 8) | (cPSYNCS << 0))
/****************************************************************************************/


/****************************************************************************************/
/* Define the constant data for PORTMD */
/****************************************************************************************/
/* Port control settings at tool break. */
#define	PORTMD_ALLHIZ			0U								/* Upper phases = High-z / lower phases = High-z */
#define	PORTMD_UHIZ_LON			1U								/* Upper phases = High-z / ower phases = PMD output */
#define	PORTMD_UON_LHIZ			2U								/* Upper phases = PMD output / lower phases = High-z */
#define	PORTMD_ALLON			3U								/* Upper phases = PMD output / lower phases = PMD output */

/*--------------------------------------------------------------------------------------*/
#define	cPORTMD					PORTMD_ALLHIZ
/*--------------------------------------------------------------------------------------*/
#define	cPORTMD_INI				((cPORTMD) << 0)
/****************************************************************************************/

/****************************************************************************************/
/* Define the constant data for TRGCR */
/****************************************************************************************/
#define	TRGMD_DIS				0U								/* Trigger output disabled */
#define	TRGMD_TRI_FIRST_HALF	1U								/* Trigger output at triangle wave first half match */
#define	TRGMD_TRI_SECND_HALF	2U								/* Trigger output at triangle wave second half match */
#define	TRGMD_TRI_F_S_CMP		3U								/* Trigger output at first-/second-half match */
#define	TRGMD_END				4U								/* Trigger output at PWM carrier end */
#define	TRGMD_CENTER			5U								/* Trigger output at PWM carrier center */
#define	TRGMD_ENDCENTER			6U								/* Trigger output at PWM carrier end/center */
#define	TRGMD_DIS7				7U								/* Trigger output disabled */

#define	TRGBE_SYNC				0U								/* sync */
#define	TRGBE_ASYNC				1U								/* The value written to PMDTRGx is immediately reflected.) */

#define	CARSEL_BASE				0U								/* Compared by the base carrier. */
#define	CARSEL_PHASE			1U								/* Compared by TEMPREG0/1=U,REG2=V,REG3=W-phase carrier. */


/*--------------------------------------------------------------------------------------*/
#define	cTRGMD_3SHT				TRGMD_TRI_F_S_CMP				/* Trigger timing setting for 3shunt */
#define	cTRGMD_1SHT				TRGMD_TRI_F_S_CMP				/* Trigger timing setting for 1shunt */
#define cTRGMD_PFC				TRGMD_TRI_F_S_CMP
#define	cBUFSYNC				TRGBE_SYNC						/* Triger Buffer update timing */
#define	cCARSEL					CARSEL_BASE						/* Selection of a comparison career */
/*--------------------------------------------------------------------------------------*/
/* triger setting for 3shunt */
#define cTRG0_3SHUNTSETTING		((cCARSEL << 16) |(cBUFSYNC << 3) | (cTRGMD_3SHT))
#define cTRG1_3SHUNTSETTING		((cCARSEL << 16) |(cBUFSYNC << 3) | (TRGMD_DIS))
#define cTRG2_3SHUNTSETTING		((cCARSEL << 16) |(cBUFSYNC << 3) | (TRGMD_DIS))
#define cTRG3_3SHUNTSETTING		((cCARSEL << 16) |(cBUFSYNC << 3) | (TRGMD_DIS))

/* triger setting for 1shunt */
#define cTRG0_1SHUNTSETTING		((cCARSEL << 16) |(cBUFSYNC << 3) | (cTRGMD_1SHT))
#define cTRG1_1SHUNTSETTING		((cCARSEL << 16) |(cBUFSYNC << 3) | (cTRGMD_1SHT))
#define cTRG2_1SHUNTSETTING		((cCARSEL << 16) |(cBUFSYNC << 3) | (TRGMD_END))
#define cTRG3_1SHUNTSETTING		((cCARSEL << 16) |(cBUFSYNC << 3) | (TRGMD_DIS))

#define cTRG0_1SHUNTSETTING_FAN_NORMAL		((cCARSEL << 16) | (cBUFSYNC << 3) | (TRGMD_TRI_SECND_HALF))
#define cTRG1_1SHUNTSETTING_FAN_NORMAL		((cCARSEL << 16) | (cBUFSYNC << 3) | (TRGMD_TRI_SECND_HALF))
#define cTRG2_1SHUNTSETTING_FAN_NORMAL		((cCARSEL << 16) | (cBUFSYNC << 3) | (TRGMD_END))
#define cTRG3_1SHUNTSETTING_FAN_NORMAL		((cCARSEL << 16) | (cBUFSYNC << 3) | (TRGMD_TRI_SECND_HALF))

#define cTRG0_SETTING_PFC					((cCARSEL << 16) | (cBUFSYNC << 3) | (cTRGMD_PFC))
#define cTRG1_SETTING_PFC					((cCARSEL << 16) | (cBUFSYNC << 3) | (TRGMD_END))
#define cTRG2_SETTING_PFC					((cCARSEL << 16) | (cBUFSYNC << 3) | (TRGMD_DIS))
#define cTRG3_SETTING_PFC					((cCARSEL << 16) | (cBUFSYNC << 3) | (TRGMD_DIS))

/*--------------------------------------------------------------------------------------*/
#define cTRG_3SHUNT 	 		((cTRG3_3SHUNTSETTING << 12) | (cTRG2_3SHUNTSETTING << 8) | (cTRG1_3SHUNTSETTING << 4) | (cTRG0_3SHUNTSETTING << 0))
#define cTRG_1SHUNT 			((cTRG3_1SHUNTSETTING << 12) | (cTRG2_1SHUNTSETTING << 8) | (cTRG1_1SHUNTSETTING << 4) | (cTRG0_1SHUNTSETTING << 0))
#define cTRG_1SHUNT_FAN_NORAML	((cTRG3_1SHUNTSETTING_FAN_NORMAL << 12) | (cTRG2_1SHUNTSETTING_FAN_NORMAL << 8) | (cTRG1_1SHUNTSETTING_FAN_NORMAL << 4) | (cTRG0_1SHUNTSETTING_FAN_NORMAL << 0))
#define cTRG_PFC				((cTRG3_SETTING_PFC << 12) | (cTRG2_SETTING_PFC << 8) | (cTRG1_SETTING_PFC << 4) | (cTRG0_SETTING_PFC << 0))		
/****************************************************************************************/


/****************************************************************************************/
/* Define the constant data for TRGMD */
/****************************************************************************************/
/* Trigger output mode */
#define	TRGOUT_FIX				0U								/* Fixed trigger output */
#define	TRGOUT_SEL				1U								/* Variable trigger output */

/* Output enable in EMG protection state */
#define	EMGTGE_DISABLE			0U								/* Disable trigger output in the protection state */
#define	EMGTGE_ENABLE			1U								/* Enable trigger output in the protection state */

/*--------------------------------------------------------------------------------------*/
#define	cEMGTGE					EMGTGE_ENABLE
/*--------------------------------------------------------------------------------------*/
#define	cTRGMD_3SHUNT			( (TRGOUT_SEL << 1) | (cEMGTGE << 0) )
#define	cTRGMD_1SHUNT			( (TRGOUT_FIX << 1) | (cEMGTGE << 0) )
#define	cTRGMD_SHUNT_PFC		( (TRGOUT_FIX << 1) | (cEMGTGE << 0) )
/****************************************************************************************/

/****************************************************************************************/
/* Define the constant data for TRGSYNCR */
/****************************************************************************************/
/* Trigger Update Timing Setting */
#define	TSYNCS_TRGMD			0U								/* PWM carrier end or PWM carrier center is set for each trigger by setting PMDxTRGCR<TRGxMD>. */
#define	TSYNCS_CENTER			1U								/* PWM carrier center */
#define	TSYNCS_END				2U								/* PWM carrier end */
#define	TSYNCS_CENTER_END		3U								/* PWM carrier end and PWM carrier center */

/*--------------------------------------------------------------------------------------*/
#define	cTRGSYNCR_INI			(TSYNCS_END)
/****************************************************************************************/


/****************************************************************************************/
/* Define the constant data for EMGCR */
/****************************************************************************************/
/* Tool break enable/disable */
#define	INHEN_DISABLE			0U
#define	INHEN_ENABLE			1U

/* EMG protection mode select */
#define	EMGMD_ALLHIZ			0U								/* PWM output control disabled / Port output = All phases High-Z */
#define	EMGMD_UON_LHIZ			1U								/* All upper phases ON, all lower phases OFF / Port output = Lower phases High-Z */
#define	EMGMD_UHIZ_LON			2U								/* All upper phases OFF, all lower phases ON / Port output = Upper phases High-Z */
#define	EMGMD_ALLHIZ2			3U								/* All phases OFF / Port output = All phases High-Z */

/* EMG protection release */
#define	EMGRS_NOP				0U
#define	EMGRS_RELEASE			1U

/* EMG protection circuit enable/disable */
#define	EMGEN_DISABLE			0U
#define	EMGEN_ENABLE			1U

/*--------------------------------------------------------------------------------------*/
#define	cEMGCNT					5U								/* Detection counter unit.16/fsys  ex.200ns@80MHz */
#define	cINHEN					INHEN_ENABLE					/* Tool Break */
#define	cEMGMD					EMGMD_ALLHIZ					/* Stop mode */
/*--------------------------------------------------------------------------------------*/
#define cEMG_ENA				((cEMGCNT << 8) | (cINHEN << 5) | (cEMGMD << 3) | (EMGEN_ENABLE << 0))
#define cEMG_DIS				((cEMGCNT << 8) | (cINHEN << 5) | (cEMGMD << 3) | (EMGEN_DISABLE << 0) )
#define	cEMG_Release			((cEMGCNT << 8) | (cINHEN << 5) | (cEMGMD << 3) | (EMGRS_RELEASE << 1) | (EMGEN_ENABLE << 0))
/****************************************************************************************/


/****************************************************************************************/
/* Define the constant data for OVVCR */
/****************************************************************************************/
/* ADC B monitor interrupt input enable */
#define	ADIN1EN_DISABLE			0U
#define	ADIN1EN_ENABLE			1U

/* ADC A monitor interrupt input enable */
#define	ADIN0EN_DISABLE			0U
#define	ADIN0EN_ENABLE			1U

/* OVV protection mode */
#define	OVVMD_NOCON				0U
#define	OVVMD_UON_LOFF			1U
#define	OVVMD_UOFF_LON			2U
#define	OVVMD_ALLOFF			3U


/* OVV input select */
#define	OVVISEL_PORT			0U
#define	OVVISEL_ADC				1U

/* OVV protection release */
#define	OVVRS_NORMAL			0U
#define	OVVRS_AUTO				1U

/* OVV protection circuit enable/disable */
#define	OVVEN_DISABLE			0U
#define	OVVEN_ENABLE			1U

/*--------------------------------------------------------------------------------------*/
#define	cOVVCNT					5U								/* Detection counter unit.16/fsys  ex.200ns@80MHz */
#define	cADIN1EN				ADIN1EN_DISABLE					/* AD1 watch input */
#define	cADIN0EN				ADIN0EN_DISABLE					/* AD0 watch input */
#define	cOVVMD					OVVMD_ALLOFF					/* Stop mode */
#define	cOVVISEL				OVVISEL_PORT					/* Sourse of OVV */
#define	cOVVRS					OVVRS_NORMAL					/* Auto release */
/*--------------------------------------------------------------------------------------*/
#define cOVV_ENA				( (cOVVCNT << 8) | (cADIN1EN << 6) | (cADIN0EN << 5) | \
								  (cOVVMD << 3) | (cOVVISEL << 2) | (cOVVRS << 1) | (OVVEN_ENABLE << 0) )

#define cOVV_DIS				( (cOVVCNT << 8) | (cADIN1EN << 6) | (cADIN0EN << 5) | \
								  (cOVVMD << 3) | (cOVVISEL << 2) | (cOVVRS << 1) | (OVVEN_DISABLE << 0) )
/****************************************************************************************/


/****************************************************************************************/
/* Define the constant data for DBGOUTCR */
/****************************************************************************************/
/*  */
#define	INIFF_0					0U
#define	INIFF_1					1U
#define	INIFF_2					3U
/*  */
#define	TRG5EN_DISABLE			0U
#define	TRG5EN_ENABLE			1U

/*  */
#define	TRG4EN_DISABLE			0U
#define	TRG4EN_ENABLE			1U

/*  */
#define	TRG3EN_DISABLE			0U
#define	TRG3EN_ENABLE			1U

/*  */
#define	TRG2EN_DISABLE			0U
#define	TRG2EN_ENABLE			1U

/*  */
#define	TRG1EN_DISABLE			0U
#define	TRG1EN_ENABLE			1U

/*  */
#define	TRG0EN_DISABLE			0U
#define	TRG0EN_ENABLE			1U

/*  */
#define	IENCEN_DISABLE			0U
#define	IENCEN_ENABLE			1U

/*  */
#define	IVEEN_DISABLE			0U
#define	IVEEN_ENABLE			1U

/*  */
#define	IOVVEN_DISABLE			0U
#define	IOVVEN_ENABLE			1U

/*  */
#define	IEMGEN_DISABLE			0U
#define	IEMGEN_ENABLE			1U

/*  */
#define	IPMDEN_DISABLE			0U
#define	IPMDEN_ENABLE			1U

/*  */
#define	IADEEN_DISABLE			0U
#define	IADEEN_ENABLE			1U

/*  */
#define	IADDEN_DISABLE			0U
#define	IADDEN_ENABLE			1U

/*  */
#define	IADCEN_DISABLE			0U
#define	IADCEN_ENABLE			1U

/*  */
#define	IADBEN_DISABLE			0U
#define	IADBEN_ENABLE			1U

/*  */
#define	IADAEN_DISABLE			0U
#define	IADAEN_ENABLE			1U

/*  */
#define	DBGMD_ADC				0U
#define	DBGMD_PMDTRG			1U
#define	DBGMD_INT				2U
#define	DBGMD_VETASK			3U

/*  */
#define	DBGEN_DISABLE			0U
#define	DBGEN_ENABLE			1U

/*--------------------------------------------------------------------------------------*/
#define	cINIFF					INIFF_0
/*--------------------------------------------------------------------------------------*/
#define	DBGOUT_DIS				( (cINIFF << 31) | (DBGEN_DISABLE << 0) )
#define	DBGOUT_ENA_ADC			( (cINIFF << 31) | (DBGMD_ADC << 1) | (DBGEN_ENABLE << 0) )
#define	DBGOUT_ENA_VE			( (cINIFF << 31) | (DBGMD_VETASK << 1) | (DBGEN_ENABLE << 0) )
/****************************************************************************************/




/** For VE ******************************************************/

#define	cTADC					(1.0f * 1e-6)					/* [s] AD convert time for 1shunt shift pwm */

/****************************************************************************************/
/* Define the constant data for FMODE */
/****************************************************************************************/
/* Selects a PWM shift mode. */
#define	SPWMMD_SPWM1			0U								/* Shift 1 */
#define	SPWMMD_SPWM2U			1U								/* Shift 2 (U-phase Normal PWM) */
#define	SPWMMD_SPWM2V			2U								/* Shift 2 (V-phase Normal PWM) */
#define	SPWMMD_SPWM2W			3U								/* Shift 2 (W-phase Normal PWM) */

/* Phase-change mode selection. */
#define	CCVMD_RELATIVE			0U								/* relative */
#define	CCVMD_ABSOLUTE			1U								/* absolute */

/* Disables/enables phase transformation. */
#define	PHCVDIS_ENABLE			0U								/* 2-3 phase transformation is enabled. (3-phase AC output) */
#define	PHCVDIS_DISABLE			1U								/* 2-3 phase transformation is disabled. (2-phase AC output) */

/* Controls voltage scalar limitation */
#define	VSLIMMD_DIS				0U								/* Scalar limitation is disabled. (The limitation of each axis is enabled.) */
#define	VSLIMMD_D				1U								/* Limits the voltage on the d-axis. */
#define	VSLIMMD_Q				2U								/* Limits the voltage on the q-axis. */
#define	VSLIMMD_DQ				3U								/* dq proportional limitation */

/* Keep the previous value of SIN/COS/SECTOR */
#define	MREGDIS_EFFECTIVE		0U
#define	MREGDIS_NOEFFECTIVE		1U

/* Trigger correction */
#define	CRCEN_DISABLE			0U								/* Disable trigger correction. */
#define	CRCEN_ENABLE			1U								/* Enable trigger correction. */

/* Direction of current detection  */
#define IPLMD_SHUNT				0U
#define IPLMD_SENSOR			1U

/* De-coupling mode selection */
#define	IDQSEL_FEEDBACK			0U								/* Feedback current(ID, IQ) */
#define	IDQSEL_INSTRUCTION		1U								/* Instruction current(IDREF, IQREF) */

/* Current detection mode */
#define IDMODE_3SHUNT			0U								/* 3-shunt */
#define IDMODE_2SENSOR			1U								/* 2-sensor */
#define IDMODE_1SHUNT_UP		2U								/* 1-shunt (for up count PMDTRG) */
#define IDMODE_1SHUNT_DOWN		3U								/* 1-shunt (for down count PMDTRG) */

/*--------------------------------------------------------------------------------------*/
#define	cSPWMMD					(SPWMMD_SPWM1)
#define	cCCVMD					(CCVMD_RELATIVE)
#define	cPHCVDIS				(PHCVDIS_ENABLE)
#define	cVSLIMMD				(VSLIMMD_DIS)
#define	cMREGDIS				(MREGDIS_EFFECTIVE)
#define	cCRCEN					(CRCEN_ENABLE)
#define	cIDQSEL					(IDQSEL_FEEDBACK)
#define cIAPLMD					(IPLMD_SHUNT)
#define cIBPLMD					(IPLMD_SHUNT)
#define cICPLMD					(IPLMD_SHUNT)

/*--------------------------------------------------------------------------------------*/
#define	cFMODE_INI_3SHUNT		(					(cCCVMD << 13) | (cPHCVDIS << 12) | (cVSLIMMD << 10) | (cMREGDIS << 9) | (cIDQSEL << 4) | (IDMODE_3SHUNT << 2) )
#define	cFMODE_INI_1SHUNT		( (cSPWMMD << 14) | (cCCVMD << 13) | (cPHCVDIS << 12) | (cVSLIMMD << 10) | (cMREGDIS << 9) | (cCRCEN << 8) | (cICPLMD << 7) | (cIBPLMD << 6) | (cIAPLMD << 5)  | (cIDQSEL << 4) | (IDMODE_1SHUNT_UP << 2) )

/****************************************************************************************/
/****************************************************************************************/
/* Define the constant data for CIDKG */
/****************************************************************************************/
/* PI control d-axis proportional coefficient range selection. */
#define	CIDKPG_RANGE			(0)		/* Constant of Range 0:1/1, 1:1/2^4, 2:1/2^8, 3:1/2^12, 4:1/2^16, 9:2, 10:2^2, 11:2^3, 12: 2^4 */
#define	CIDKIG_RANGE			(0)		/* Constant of Range 0:1/1, 1:1/2^4, 2:1/2^8, 3:1/2^12, 4:1/2^16 */
#define	CIQKPG_RANGE			(0)		/* Constant of Range 0:1/1, 1:1/2^4, 2:1/2^8, 3:1/2^12, 4:1/2^16, 9:2, 10:2^2, 11:2^3, 12: 2^4 */
#define	CIQKIG_RANGE			(0)		/* Constant of Range 0:1/1, 1:1/2^4, 2:1/2^8, 3:1/2^12, 4:1/2^16 */
/*--------------------------------------------------------------------------------------*/
#define	cCIDKG					((CIDKPG_RANGE << 8) | (CIDKIG_RANGE << 0))
#define	cCIQKG					((CIQKPG_RANGE << 8) | (CIQKIG_RANGE << 0))

/****************************************************************************************/
/****************************************************************************************/


/****************************************************************************************/
#define	TAPP_OUT_CON			0U			/* Output control 1 */
#define	TAPP_TRG_GEN			1U			/* Trigger generation */
#define	TAPP_IN_PRC				2U			/* Input process 1 */
#define	TAPP_IN_PHSCNV			3U			/* Input phase transformation */
#define	TAPP_IN_AXISCNV			4U			/* Input coordinate axis transformation */
#define	TAPP_CRT_CON			5U			/* Current control */
#define	TAPP_SINCOS				6U			/* SIN/COS calculation */
#define	TAPP_OUT_AXISCNV		7U			/* Output coordinate axis transformation */
#define	TAPP_OUT_PHSCNV			8U			/* Output phase transformation 1[SVM] */
#define	TAPP_OUT_CON2			9U			/* Output control 2 */
#define	TAPP_IN_PRC2			10U			/* Input process 2 */
#define	TAPP_OUT_PHSCNV2		11U			/* Output phase transformation 2[I-Clarke] */
#define	TAPP_ATAN2				12U			/* ATAN2 */
#define	TAPP_SQRT				13U			/* SQRT */

#define	SCH_1					1U
#define	SCH_2					2U
#define	SCH_3					3U
#define	SCH_4					4U
#define	SCH_5					5U
#define	SCH_6					6U
#define	SCH_7					7U
#define	SCH_8					8U
#define	SCH_9					9U
#define	SCH_10					10U
#define	SCH_11					11U
#define	SCH_12					12U
#define	SCH_13					13U
#define	SCH_14					14U
#define	SCH_15					15U
/****************************************************************************************/


/****************************************************************************************/
/* VETRGMODE */
/****************************************************************************************/
#define	TRGMODE_IGNORE			0U
#define	TRGMODE_UNITA			1U
/*--------------------------------------------------------------------------------------*/
#define	cTRGMODE1_INI			(TRGMODE_IGNORE)
#define	cTRGMODE0_INI			(TRGMODE_UNITA)

/****************************************************************************************/


/****************************************************************************************/
/* VEERRINTEN */
/****************************************************************************************/
/* Controls a task completion interrupt. */
#define	INTTEN_DISABLE			0U
#define	INTTEN_ENABLE			1U

/* Controls an interrupt at error detection. */
#define	VERREN_DISABLE			0U
#define	VERREN_ENABLE			1U

/*--------------------------------------------------------------------------------------*/
#define	cINTTEN					INTTEN_DISABLE
#define	cVERREN					VERREN_DISABLE
/*--------------------------------------------------------------------------------------*/
#define	cVERREN_INI				((cINTTEN << 2) | (cVERREN << 0))
/****************************************************************************************/

/****************************************************************************************/
/* VECOMPEND */
/****************************************************************************************/
#define	VCEND_TERMINATE			1U
/*--------------------------------------------------------------------------------------*/
#define	VE_FORCE_STOP			(VCEND_TERMINATE << 0U)
/****************************************************************************************/

/****************************************************************************************/
/* VEMODE */
/****************************************************************************************/
/* Current polarity determination */
#define	IPDEN_DISABLE			0U
#define	IPDEN_ENABLE			1U

/* Dead time correction contorol of the PMD */
#define	PMDDTCEN_DISABLE		0U
#define	PMDDTCEN_ENABLE			1U

/* Duty of 100% setting when the upper-limit */
#define	PWMFLEN_DISABLE			0U
#define	PWMFLEN_ENABLE			1U

/* Duty of 0% setting when the lower-limit */
#define	PWMBLEN_DISABLE			0U
#define	PWMBLEN_ENABLE			1U

/* Non-interference control */
#define	NICEN_DISABLE			0U
#define	NICEN_ENABLE			1U

/* Expansion control (Non-interference control and voltage scalar limitation) */
#define	T5ECEN_DISABLE			0U
#define	T5ECEN_ENABLE			1U

/* Specifies anti-windup (AWU) control */
#define	AWUMD_DISABLE			0U
#define	AWUMD_ENABLE4			1U
#define	AWUMD_ENABLE2			2U
#define	AWUMD_ENABLE1			3U

/* Phase clipping control */
#define	CLPEN_DISABLE			0U
#define	CLPEN_ENABLE			1U

/* Specifies ATAN calculation control */
#define	ATANMD_DISABLE			0U
#define	ATANMD_IDQ				2U
#define	ATANMD_EDQ				3U

/* Selects the supply voltage store register */
#define	VDCSEL_VDC				0U
#define	VDCSEL_VDCL				1U

/* Output control */
#define	OCRMD_OUT_OFF			0U
#define	OCRMD_OUT_ON			1U
#define	OCRMD_OUT_ON_LOWPH		2U
#define	OCRMD_OUT_OFF_EMGRET	3U

/* Zero current detection */
#define	ZIEN_DISABLE			0U
#define	ZIEN_ENABLE				1U

/* Phase interpolation */
#define	PVIEN_DISABLE			0U
#define	PVIEN_ENABLE			1U

/*--------------------------------------------------------------------------------------*/
#define	cIPDEN					(IPDEN_DISABLE)					/* Current polarity determination */
#define	cPMDDTCEN				(PMDDTCEN_DISABLE)				/* Dead time correction contorol of the PMD */
#define	cPWMFLEN				(PWMFLEN_DISABLE)				/* Duty of 100% setting when the upper-limit */
#define	cPWMBLEN				(PWMBLEN_DISABLE)				/* Duty of 0% setting when the lower-limit */
#define	cNICEN					(NICEN_DISABLE)					/* Non-interference control */
#define	cT5ECEN					(T5ECEN_DISABLE)				/* Expansion control */
#define	cAWUMD					(AWUMD_DISABLE)					/* Specifies anti-windup (AWU) control */
#define	cCLPEN					(CLPEN_DISABLE)					/* Phase clipping control */
#define	cATANMD					(ATANMD_DISABLE)				/* Specifies ATAN calculation control */
#define	cVDCSEL					(VDCSEL_VDC)					/* Selects the supply voltage store register */
#define	cZIEN					(ZIEN_DISABLE)					/* Zero current detection */
#define	cPVIEN					(PVIEN_DISABLE)					/* Phase interpolation */
/*--------------------------------------------------------------------------------------*/
#define	VEMODE_STOP				( (cIPDEN << 15) | (cPMDDTCEN << 14) | (cPWMFLEN << 13) | (cPWMBLEN << 12) \
								| (cNICEN << 11) | (cT5ECEN << 10) | (cAWUMD << 8) | (cCLPEN << 7) | (cATANMD << 5) | (cVDCSEL << 4) \
								| (OCRMD_OUT_OFF << 2) | (cZIEN << 1) | (cPVIEN << 0) )

#define	VEMODE_BOOT				( (cIPDEN << 15) | (cPMDDTCEN << 14) | (cPWMFLEN << 13) | (cPWMBLEN << 12) \
								| (cNICEN << 11) | (cT5ECEN << 10) | (cAWUMD << 8) | (cCLPEN << 7) | (cATANMD << 5) | (cVDCSEL << 4) \
								| (OCRMD_OUT_ON_LOWPH << 2)  | (cZIEN << 1) | (cPVIEN << 0) )

#define	VEMODE_INITPOS			( (cIPDEN << 15) | (cPMDDTCEN << 14) | (cPWMFLEN << 13) | (cPWMBLEN << 12) \
								| (cNICEN << 11) | (cT5ECEN << 10) | (cAWUMD << 8) | (cCLPEN << 7) | (cATANMD << 5) | (cVDCSEL << 4) \
								| (OCRMD_OUT_ON << 2)  | (cZIEN << 1) | (cPVIEN << 0) )

#define	VEMODE_FORCE			( (cIPDEN << 15) | (cPMDDTCEN << 14) | (cPWMFLEN << 13) | (cPWMBLEN << 12) \
								| (cNICEN << 11) | (cT5ECEN << 10) | (cAWUMD << 8) | (cCLPEN << 7) | (cATANMD << 5) | (cVDCSEL << 4) \
								| (OCRMD_OUT_ON << 2)  | (cZIEN << 1) | (cPVIEN << 0) )

#define	VEMODE_CHGUP			( (cIPDEN << 15) | (cPMDDTCEN << 14) | (cPWMFLEN << 13) | (cPWMBLEN << 12) \
								| (cNICEN << 11) | (cT5ECEN << 10) | (cAWUMD << 8) | (cCLPEN << 7) | (cATANMD << 5) | (cVDCSEL << 4) \
								| (OCRMD_OUT_ON << 2)  | (cZIEN << 1) | (cPVIEN << 0) )

#define	VEMODE_STEADY			( (cIPDEN << 15) | (cPMDDTCEN << 14) | (cPWMFLEN << 13) | (cPWMBLEN << 12) \
								| (cNICEN << 11) | (cT5ECEN << 10) | (cAWUMD << 8) | (cCLPEN << 7) | (cATANMD << 5) | (cVDCSEL << 4) \
								| (OCRMD_OUT_ON << 2)  | (cZIEN << 1) | (cPVIEN << 0) )

#define	VEMODE_EMG				( (cIPDEN << 15) | (cPMDDTCEN << 14) | (cPWMFLEN << 13) | (cPWMBLEN << 12) \
								| (cNICEN << 11) | (cT5ECEN << 10) | (cAWUMD << 8) | (cCLPEN << 7) | (cATANMD << 5) | (cVDCSEL << 4) \
								| (OCRMD_OUT_OFF << 2) | (cZIEN << 1) | (cPVIEN << 0) )
/****************************************************************************************/

/****************************************************************************************/
/* VECPURUNTRG */
/****************************************************************************************/
#define	VCPURT_START			1U
/*--------------------------------------------------------------------------------------*/
#define	VE_START				(VCPURT_START << 0U)
/****************************************************************************************/


#define	SET_TAPP(x)				(x << 0U)
#define	MASK_TAPP				((uint32_t)0xFFFFFFF0)

#define	SET_ACTSCH(x)			(x << 0U)
#define	MASK_ACTSCH				((uint32_t)0xFFFFFFF0)

#define	SET_REPTIME(x)			(x << 0U)
#define	SET_TRGMODE(x)			(x << 0U)
#define	SET_ERRINTEN(x)			(x << 0U)


/** For Encoder ******************************************************/
#define	cENCMULTIPLE			(4)


/** For AMP **********************************************************/
#define	AMPGAIN_FOLD_2_0		0U				/* *2.0 */
#define	AMPGAIN_FOLD_2_5		1U				/* *2.5 */
#define	AMPGAIN_FOLD_3_0		2U				/* *3.0 */
#define	AMPGAIN_FOLD_3_5		3U				/* *3.5 */
#define	AMPGAIN_FOLD_4_0		4U				/* *4.0 */
#define	AMPGAIN_FOLD_4_5		5U				/* *4.5 */
#define	AMPGAIN_FOLD_6_0		6U				/* *6.0 */
#define	AMPGAIN_FOLD_7_0		7U				/* *7.0 */
#define	AMPGAIN_FOLD_8_0		8U				/* *8.0 */
#define	AMPGAIN_FOLD_10_0		9U				/* *10.0 */
#define	AMPGAIN_FOLD_12_0		10U				/* *12.0 */
#define	AMPGAIN_FOLD_15_0		11U				/* *15.0 */

/*--------------------------------------------------------------------------------------*/
#define	AMP_ENABLE(fold)		((uint8_t)((fold << 1) | 0x01U))
#define	AMP_DISABLE(fold)		((uint8_t)(fold << 1))
/****************************************************************************************/


/** For CMP **********************************************************/
#define	CMPSEL_RAW				0U
#define	CMPSEL_OPAMP			1U
/*--------------------------------------------------------------------------------------*/
#define	CMP_ENABLE(sel)			((uint8_t)((sel << 1) | 0x01U))
#define	CMP_DISABLE(sel)		((uint8_t)(sel << 1))
/****************************************************************************************/

/** For ADC **********************************************************/
/* Select conversion result storage register  */
#define	PMD_REG0_3   			0U		/* ADxREG0, ADxREG1, ADxREG2, ADxREG3 	*/
#define	PMD_REG4_7   			1U		/* ADxREG4, ADxREG5, ADxREG6, ADxREG7 	*/
#define	PMD_REG8_11  			2U		/* ADxREG8, ADxREG9, ADxREG10,ADxREG11	*/
#define	PMD_REG12_15 			3U		/* ADxREG12,ADxREG13,ADxREG14,ADxREG15 	*/
#define	PMD_REG16_19 			4U		/* ADxREG16,ADxREG17,ADxREG18,ADxREG19 	*/
#define	PMD_REG20_23 			5U		/* ADxREG20,ADxREG21,ADxREG22,ADxREG23 	*/
#define	PMD_REG24_27 			6U		/* ADxREG24,ADxREG25,ADxREG26,ADxREG27	*/
#define	PMD_REG28_31 			7U		/* ADxREG28,ADxREG29,ADxREG30,ADxREG31 	*/

/*--------------------------------------------------------------------------------------*/
#define cPMD0_PREGS				(PMD_REG0_3)
#define cPMD1_PREGS				(PMD_REG8_11)
#define cPMD2_PREGS				(PMD_REG0_3)
/*--------------------------------------------------------------------------------------*/
#define cPMD0_PREGS_3SHUNT		((cPMD0_PREGS << 0U) | (cPMD0_PREGS << 4U) | (cPMD0_PREGS << 8U) | (cPMD0_PREGS << 12U) | (cPMD0_PREGS << 16U) | (cPMD0_PREGS << 20U))
#define cPMD0_PREGS_1SHUNT 		((cPMD0_PREGS << 0U) | (cPMD0_PREGS << 4U) | (cPMD0_PREGS << 8U))
#define cPMD0_ADREG_OFFSET		(cPMD0_PREGS * 4)

#define cPMD1_PREGS_3SHUNT		((cPMD1_PREGS << 0U) | (cPMD1_PREGS << 4U) | (cPMD1_PREGS << 8U) | (cPMD1_PREGS << 12U) | (cPMD1_PREGS << 16U) | (cPMD1_PREGS << 20U))
#define cPMD1_PREGS_1SHUNT 		((cPMD1_PREGS << 0U) | (cPMD1_PREGS << 4U) | (cPMD1_PREGS << 8U))
#define cPMD1_ADREG_OFFSET		(cPMD1_PREGS * 4)

#define cPMD2_PREGS_PFCSHUNT 	((cPMD2_PREGS << 0U) | (PMD_REG4_7  << 4U) | (PMD_REG12_15 << 8U)| (cPMD2_PREGS << 12U) )  /* Program2 save data to REG12-15 */

/*--------------------------------------------------------------------------------------*/


/****************************************************************************************/
/**** Define the constant data for CGPLL1SEL *********************************************/
/* Register bit position Definition	*/
#define B_PLL1ON				0U
#define B_PLL1SEL				1U

#define	PLL1SET_OSC_6M			0x1C5028UL
#define	PLL1SET_OSC_8M			0x26501EUL
#define	PLL1SET_OSC_10M			0x2E5018UL
#define	PLL1SET_OSC_12M			0x36D014UL
/*--------------------------------------------------------------------------------------*/
#define	cPLL1SET				PLL1SET_OSC_10M
/*--------------------------------------------------------------------------------------*/

/**** Define the constant data for CGWUPHCR *********************************************/
#define	WUCLK_INTERNAL			0U
#define	WUCLK_EXTERNAL			1U

#define	WUON_START				1U
/*--------------------------------------------------------------------------------------*/
#define	OSC_FRQ					(10.0)					/* [MHz] Oscillator freq */
#define	cWUCLK					WUCLK_EXTERNAL			/* Oscillator : External or internal */

/* Warm-up timer */
#define	cWUPT_100US				(((uint32_t)(100.0 * OSC_FRQ) - 16) >> 4UL)		/* 100us */
#define	cWUPT_400US				(((uint32_t)(400.0 * OSC_FRQ) - 16) >> 4UL)		/* 400us */

/*--------------------------------------------------------------------------------------*/
#define	cWUPHCR_INIT_TIME		((cWUPT_100US << 20U) | (cWUCLK << 8U))
#define	cWUPHCR_LOCKUP_TIME		((cWUPT_400US << 20U) | (cWUCLK << 8U))
/*--------------------------------------------------------------------------------------*/
/****************************************************************************************/


/*===================================================================*
	  Typedef Definition
 *===================================================================*/
typedef enum
{
	cNormal		 	 = 0,
	cEMGProtected	 = 1
} emg_status_e;

/**
  * @brief	PMD Init Structure definition
  */
typedef struct
{
	uint8_t		shunt;
	uint8_t		poll;
	uint8_t		polh;
	uint16_t	pwmrate;
	uint16_t	deadtime;
	uint8_t		busmode;
} PMD_InitTypeDef;

/**
  * @brief	VE Init Structure definition
  */
typedef struct
{
	uint8_t		shunt;

	uint16_t	reptime;
	uint16_t	trgmode;
	uint16_t	tpwm;

	uint16_t	idkp;
	uint16_t	idki;
	uint16_t	idkg;
	uint16_t	iqkp;
	uint16_t	iqki;
	uint16_t	iqkg;

	uint16_t	zero_offset;

} VE_InitTypeDef;


/**
  * @brief	AD Init Structure definition
  */
typedef struct
{
	uint8_t		shunt;
	uint8_t		iuch;
	uint8_t		ivch;
	uint8_t		iwch;
	uint8_t		idcch;
	uint8_t		vdcch;
	uint8_t		pmd_ch;
	uint8_t		pints;
	uint32_t**	p_adreg0;
	uint32_t**	p_adreg1;
	uint32_t**	p_adreg2;
	uint32_t**	p_adreg3;
} AD_InitTypeDef;

/*===================================================================*
	  Proto Type Definition
 *===================================================================*/
emg_status_e	PMD_GetEMG_Status(TSB_PMD_TypeDef* const PMDx);
void	PMD_ReleaseEMG_Protection(TSB_PMD_TypeDef* const PMDx);
void	PMD_RegDataSet(TSB_PMD_TypeDef* const PMDx, uint16_t duty_u, uint16_t duty_v, uint16_t duty_w, q15_t adtrg0, q15_t adrtg1);
void	PMD_SetOutputMode(TSB_PMD_TypeDef* const PMDx, uint32_t _mdout);
void	PMD_SetTriggerSelect(TSB_PMD_TypeDef* const PMDx, uint32_t _trgsel);
void	VE_Start(TSB_VE_TypeDef* const VEx);
void	VE_GetPhaseCurrent(TSB_VE_TypeDef* const VEx, q15_t* _ia, q15_t* _ib, q15_t* _ic);
void	VE_GetCurrentAdcData(TSB_VE_TypeDef* const VEx, uint32_t* _adc_ia, uint32_t* _adc_ib, uint32_t* _adc_ic);
void	VE_GetdataFromVEreg(TSB_VE_TypeDef* const VEx, vector_t* const _motor);
uint32_t	VE_GetPWM_DutyU(TSB_VE_TypeDef* const VEx);
uint32_t	VE_GetPWM_DutyV(TSB_VE_TypeDef* const VEx);
uint32_t	VE_GetPWM_DutyW(TSB_VE_TypeDef* const VEx);
uint32_t	VE_GetPWM_DutyMed(TSB_VE_TypeDef* const VEx);
uint32_t	VE_GetPWM_DutyMin(TSB_VE_TypeDef* const VEx);
int		VE_GetPWM_Modulation(TSB_VE_TypeDef* const VEx);
uint32_t	VE_GetSector(TSB_VE_TypeDef* const VEx);
int		VE_GetShiftPWMState(TSB_VE_TypeDef* const VEx);
int		VE_GetOutputMode(TSB_VE_TypeDef* const VEx);
void	VE_SetdataToVEreg_Stop(TSB_VE_TypeDef* const VEx, const vector_t* const _motor);
void	VE_SetdataToVEreg_Bootstrap(TSB_VE_TypeDef* const VEx, const vector_t* const _motor);
void	VE_SetdataToVEreg_Initposition_i(TSB_VE_TypeDef* const VEx, const vector_t* const _motor);
void	VE_SetdataToVEreg_Initposition_v(TSB_VE_TypeDef* const VEx, const vector_t* const _motor);
void	VE_SetdataToVEreg_Force_i(TSB_VE_TypeDef* const VEx, const vector_t* const _motor);
void	VE_SetdataToVEreg_Force_v(TSB_VE_TypeDef* const VEx, const vector_t* const _motor);
void	VE_SetdataToVEreg_Change_up(TSB_VE_TypeDef* const VEx, const vector_t* const _motor);
void	VE_SetdataToVEreg_Steady_A(TSB_PMD_TypeDef* const PMDx,TSB_VE_TypeDef* const VEx, const vector_t* const _motor);
void	VE_SetdataToVEreg_Emergency(TSB_VE_TypeDef* const VEx, const vector_t* const _motor);
void	VE_SetZeroCurrentData(TSB_VE_TypeDef* const VEx, uint32_t _z_ia, uint32_t _z_ib, uint32_t _z_ic);
void	VE_SetVDCreg(TSB_VE_TypeDef* const VEx, q15_t _dat);
void	VE_SetSpwmMode(TSB_VE_TypeDef* const VEx, uint8_t _dat);
void	VE_SetModulType(TSB_VE_TypeDef* const VEx, uint8_t _dat);
q15_t 	VE_GET_Cal_Vdq(q15_t vd, q15_t vq);
void	AD_GetIdcFirstHalf(vector_t* const _motor);
void	AD_GetIdcSecondHalf(vector_t* const _motor);
void	IP_PMD_init(TSB_PMD_TypeDef* const PMDx, PMD_InitTypeDef* const _initdata);
void	IP_VE_init(TSB_VE_TypeDef* const VEx, VE_InitTypeDef* const _initdata);
void		IP_PLL_ADCLK_init(void);
void	IP_ADC_init(TSB_AD_TypeDef* const ADx, AD_InitTypeDef* const _initdata);
void	IP_AMP_init(void);

#endif /* _MCUIP_DRV_H */

/*********************************** END OF FILE ******************************/


