/**
 *******************************************************************************
 * @file    ipdrv_siwdt.c
 * @brief   This file provides all the functions prototypes for M4Kx SIWDT driver.
 * @version V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 *
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "ipdrv_siwdt.h"

/** @addtogroup TX04_Periph_Driver
  * @{
  */

/** @defgroup SIWDT
  * @brief SIWDT driver modules
  * @{
  */
/** @defgroup SIWDT_Private_Defines
  * @{
  */

#define SIWDT_EN_WDTE_ENABLE     ((uint32_t)0x00000001)
#define SIWDT_EN_WDTE_DISABLE    ((uint32_t)0x00000000)

#define SIWDT_CR_DISABLE_CODE    ((uint32_t)0x000000B1)
#define SIWDT_CR_CLEAR_CODE      ((uint32_t)0x0000004E)

#define SIWDT_MOD_WDTP_MASK      ((uint32_t)0x00000700)
#define SIWDT_MOD_WDCLS_MASK     ((uint32_t)0x00003000)
#define SIWDT_MOD_WDCWD_MASK     ((uint32_t)0x00000030)
#define SIWDT_MOD_INTF_ENABLE    ((uint32_t)0x00000002)
#define SIWDT_MOD_INTF_DISABLE   ((uint32_t)0x00000000)

#define SIWDT_PRO_NO_PROTECTION  ((uint32_t)0x0000001E)
#define SIWDT_PRO_PROTECT_A_MODE ((uint32_t)0x000000A9)
#define SIWDT_PRO_PROTECT_B_MODE ((uint32_t)0x00000074)

/** @} */
/* End of group SIWDT_Private_Defines */

/** @defgroup SIWDT_Private_FunctionPrototypes
  * @{
  */

static SIWDT_State SIWDT_GetState(void);

/** @} */
/* End of group SIWDT_Private_FunctionPrototypes */

/** @defgroup SIWDT_Private_Functions
  * @{
  */

/**
  * @brief  Get state of SIWDT
  * @param  None
  * @retval State of SIWDT
  *   The value returned can be one of the following values:
  *   SIWDT_STATE_STOPPING or SIWDT_STATE_OPERATING
  * @note When the setting of the SIWDT is changed, disable SIWDT before change
  * @register The used register:
  *   SIWDxEN<WDTE>
  */
static SIWDT_State SIWDT_GetState(void)
{
    SIWDT_State retval = SIWDT_STATE_STOPPING;

    if (TSB_SIWD0_EN_WDTF == 0) {
        /* Do nothing */
    } else {
        retval = SIWDT_STATE_OPERATING;
    }

    return retval;
}

/** @} */
/* End of group group SIWDT_Private_Functions */

/** @defgroup SIWDT_Exported_Functions
  * @{
  */

/**
  * @brief  Enable the SIWDT.
  * @param  None
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result SIWDT_Enable(void)
{
    Result retval = ERROR;

    if (SIWDT_GetProtectMode() == SIWDT_NO_PROTECTION) {
        /* To start the SIWDT, write "1" to WDMOD<WDTE> */
        TSB_SIWD0_EN_WDTE = SIWDT_EN_WDTE_ENABLE;
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Set detection time
  * @param  DetectTime: Set the Detection time.
  *   This parameter can be one of the following values:
  *   SIWDT_DETECT_TIME_EXP_15, SIWDT_DETECT_TIME_EXP_17, SIWDT_DETECT_TIME_EXP_19,
  *   SIWDT_DETECT_TIME_EXP_21, SIWDT_DETECT_TIME_EXP_23 or SIWDT_DETECT_TIME_EXP_25,
  *   SIWDT_DETECT_TIME_EXP_27, SIWDT_DETECT_TIME_EXP_29.
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  * @register The used register:
  *   SIWDxMOD<WDTP[2:0]>
  */
Result SIWDT_SetDetectTime(SIWDT_DetectTime DetectTime)
{
    Result retval = ERROR;
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_SIWDT_DETECT_TIME(DetectTime));

    if ((SIWDT_GetState() == SIWDT_STATE_STOPPING) &&
        (SIWDT_GetProtectMode() == SIWDT_NO_PROTECTION)) {
        /* Set SIWDT Detection time */
        tmp = TSB_SIWD0->MOD;
        tmp &= ~SIWDT_MOD_WDTP_MASK;
        tmp |= DetectTime << 8U;
        TSB_SIWD0->MOD = tmp;
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Set clock source
  * @param  ClockSource: Clock source for SIWDT.
  *   This parameter can be one of the following values:
  *   SIWDT_CLOCK_SOURCE_FSYS_DIVIDE_FOUR,
  *   SIWDT_CLOCK_SOURCE_INTERNAL_OSCILLATOR_FOR_SYSTEM,
  *   SIWDT_CLOCK_SOURCE_INTERNAL_OSCILLATOR_FOR_OFD,
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  * @register The used register:
  *   SIWDxMOD<WDCLS[1:0]>
  */
Result SIWDT_SetClockSource(SIWDT_ClockSource ClockSource)
{
    Result retval = ERROR;
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_SIWDT_CLOCK_SOURCE(ClockSource));

    if ((SIWDT_GetState() == SIWDT_STATE_STOPPING) &&
        (SIWDT_GetProtectMode() == SIWDT_NO_PROTECTION)) {
        /* Set SIWDT clock source */
        tmp = TSB_SIWD0->MOD;
        tmp &= ~SIWDT_MOD_WDCLS_MASK;
        tmp |= ClockSource << 12U;
        TSB_SIWD0->MOD = tmp;
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Set clear window
  * @param  ClearWindow: Clear window for SIWDT.
  *   This parameter can be one of the following values:
  *   SIWDT_CLEAR_WINDOW_NOT_SPECIFIED,
  *   SIWDT_CLEAR_WINDOW_LATTER_A_HALF,
  *   SIWDT_CLEAR_WINDOW_LATTER_A_QUARTER,
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  * @register The used register:
  *   SIWDxMOD<WDCWD[1:0]>
  */
Result SIWDT_SetClearWindow(SIWDT_ClearWindow ClearWindow)
{
    Result retval = ERROR;
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_SIWDT_CLEAR_WINDOW(ClearWindow));

    if ((SIWDT_GetState() == SIWDT_STATE_STOPPING) &&
        (SIWDT_GetProtectMode() == SIWDT_NO_PROTECTION)) {
        /* Set SIWDT clear window */
        tmp = TSB_SIWD0->MOD;
        tmp &= ~SIWDT_MOD_WDCWD_MASK;
        tmp |= ClearWindow << 4U;
        TSB_SIWD0->MOD = tmp;
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Set operation after detect malfunction
  * @param  OperationAfterDetectMalFunction: Operation when clear code is written before the clear window time.
  *   This parameter can be one of the following values:
  *   SIWDT_OPERATION_INTERRUPT,
  *   SIWDT_OPERATION_RESET,
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  * @register The used register:
  *   SIWDxMOD<RESCR>
  */
Result SIWDT_SetOperationAfterDetectMalFunction(SIWDT_OperationAfterDetectMalFunction OperationAfterDetectMalFunction)
{
    Result retval = ERROR;

    /* Check the parameters */
    assert_param(IS_SIWDT_OPERATION_AFTER_DETECT_MALFUNCTION(OperationAfterDetectMalFunction));

    if ((SIWDT_GetState() == SIWDT_STATE_STOPPING) &&
        (SIWDT_GetProtectMode() == SIWDT_NO_PROTECTION)) {
        /* Set SIWDT operation after detect malfunction */
        TSB_SIWD0_MOD_RESCR = OperationAfterDetectMalFunction;
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Write the clear code.
  * @param  None
  * @retval None
  * @register The used register:
  *   SIWDxCR<WDCR[7:0]>
  */
void SIWDT_WriteClearCode(void)
{
    TSB_SIWD0->CR = SIWDT_CR_CLEAR_CODE;
}

/**
  * @brief  Get protect mode of SIWDT
  * @param  None
  * @retval The value returned can be one of the following values:
  *   SIWDT_NO_PROTECTION, SIWDT_PROTECT_A_MODE or SIWDT_PROTECT_B_MODE
  * @register The used register:
  *   SIWDxPRO<PROTECT[7:0]>
  */
SIWDT_ProtectMode SIWDT_GetProtectMode(void)
{
    SIWDT_ProtectMode retval = SIWDT_NO_PROTECTION;

    if (TSB_SIWD0->PRO == SIWDT_PRO_NO_PROTECTION) {
        /* Do nothing */
    } else if (TSB_SIWD0->PRO == SIWDT_PRO_PROTECT_A_MODE) {
        retval = SIWDT_PROTECT_A_MODE;
    } else if (TSB_SIWD0->PRO == SIWDT_PRO_PROTECT_B_MODE) {
        retval = SIWDT_PROTECT_B_MODE;
    }

    return retval;
}

/**
  * @brief  Set protect mode of SIWDT
  * @param  ProtectMode: This parameter can be one of the following values:
  *   SIWDT_NO_PROTECTION, SIWDT_PROTECT_A_MODE, SIWDT_PROTECT_B_MODE
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  * @register The used register:
  *   SIWDxPRO<PROTECT[7:0]>
  */
Result SIWDT_SetProtectMode(SIWDT_ProtectMode ProtectMode)
{
    Result retval = ERROR;

    /* Check the parameters */
    assert_param(IS_SIWDT_PROTECT_MODE(ProtectMode));

    /* The protect mode should be set while SIWDxEN<WDTE> is "1" */
    if (SIWDT_GetState() == SIWDT_STATE_OPERATING) {
        SIWDT_ProtectMode mode = SIWDT_GetProtectMode();

        switch (mode) {
        case SIWDT_PROTECT_A_MODE:
            /* Protect A mode cannot be canceled except initialization by reset */
            /* Do nothing */
            break;

        case SIWDT_PROTECT_B_MODE:
            if (ProtectMode == SIWDT_NO_PROTECTION) {
                TSB_SIWD0->PRO = SIWDT_PRO_NO_PROTECTION;
                retval = SUCCESS;
            } else {
                /* In Protect B mode, data writing to SIWDxPRO <PROTECT> other than "0x1E" (
                   cancelling Protect B mode) are ignored */
                /* Do nothing */
            }
            break;

        case SIWDT_NO_PROTECTION:
            if (ProtectMode == SIWDT_NO_PROTECTION) {
                /* Already in no protection mode, do nothing */
            } else if (ProtectMode == SIWDT_PROTECT_A_MODE) {
                TSB_SIWD0->PRO = SIWDT_PRO_PROTECT_A_MODE;
                retval = SUCCESS;
            } else if (ProtectMode == SIWDT_PROTECT_B_MODE) {
                TSB_SIWD0->PRO = SIWDT_PRO_PROTECT_B_MODE;
                retval = SUCCESS;
            }
            break;

        default:
            /* Do nothing */
            break;
        }
    }

    return retval;
}

/** @} */
/* End of group SIWDT_Exported_Functions */

/** @} */
/* End of group SIWDT */

/** @} */
/* End of group TX04_Periph_Driver */
