/**
 *******************************************************************************
 * @file    ipdrv_tspi.c
 * @brief   This file provides API functions for M4Ky TSPI driver.
 * @version V1.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "ipdrv_tspi.h"

/** @addtogroup TX04_Periph_Driver
  * @{
  */

/** @defgroup TSPI 
  * @brief TSPI driver modules
  * @{
  */

/** @defgroup TSPI_Private_Defines
  * @{
  */

#define TSPI_RESET_CLEAR     ((uint32_t)0xFFFFFF3F)

#define CR1_TRXE_MASK        ((uint32_t)0xFFFFBFFF)
#define CR1_TSPIMS_MASK      ((uint32_t)0xFFFFDFFF)
#define CR1_MSTR_MASK        ((uint32_t)0xFFFFEFFF)
#define CR1_TMMD_MASK        ((uint32_t)0xFFFFF3FF)
#define CR1_CSSEL_MASK       ((uint32_t)0xFFFFFCFF)
#define CR1_FC_MASK          ((uint32_t)0xFFFFFF00)

#define CR2_TIDLE_MASK       ((uint32_t)0xFF3FFFFF)
#define CR2_TXDEMP_MASK      ((uint32_t)0xFFDFFFFF)
#define CR2_TIL_MASK         ((uint32_t)0xFFFF0FFF)
#define CR2_RIL_MASK         ((uint32_t)0xFFFFF0FF)
#define CR2_INT_MASK         ((uint32_t)0xFFFFFF0B)
#define CR2_DMATE_MASK       ((uint32_t)0xFFFFFFFD)
#define CR2_DMARE_MASK       ((uint32_t)0xFFFFFFFE)

/** @} */
/* End of group TSPI_Private_Defines */

/** @defgroup TSPI_Private_FunctionPrototypes
  * @{
  */

/** @} */
/* End of group TSPI_Private_FunctionPrototypes */

/** @defgroup TSPI_Private_Functions
  * @{
  */

/** @} */
/* End of group TSPI_Private_Functions */

/** @defgroup TSPI_Exported_Functions
  * @{
  */


/**
  * @brief Software reset TSPI module.
  * @param TSPIx: Select TSPI unit.
  *   This parameter can be one of the following values:
  *   TSB_TSPI0, TSB_TSPI1.
  * @retval None.
  */
void TSPI_SWReset(TSB_TSPI_TypeDef * TSPIx)
{
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_TSPI_UNIT(TSPIx));

    /* Write '10' then '01' to bit7-6 */
    tmp = TSPIx->CR0;
    tmp &= TSPI_RESET_CLEAR;
    tmp |= (uint32_t) (0x02U << 6U);
    TSPIx->CR0 = tmp;
    tmp &= TSPI_RESET_CLEAR;
    tmp |= (uint32_t) (0x01U << 6U);
    TSPIx->CR0 = tmp;
}

/**
  * @brief Enable TSPI module, it must be called at first.
  * @param TSPIx: Select TSPI unit.
  *   This parameter can be one of the following values:
  *   TSB_TSPI0, TSB_TSPI1.
  * @retval None.
  */
void TSPI_Enable(TSB_TSPI_TypeDef * TSPIx)
{
    /* Check the parameters */
    assert_param(IS_TSPI_UNIT(TSPIx));

    /* Write '1' to bit0, no need to care about bit7-6 */
    TSPIx->CR0 = 0x01U;
}

/**
  * @brief Enable/Disable TSPI Transfer.
  * @param TSPIx: Select TSPI unit.
  *   This parameter can be one of the following values:
  *   TSB_TSPI0, TSB_TSPI1.
  * @param NewState: Specify the state for transfer.
  *   This parameter can be one of the following values:
  *   ENABLE or DISABLE.  
  * @retval None.
  */
void TSPI_SetTxRxCtrl(TSB_TSPI_TypeDef * TSPIx, FunctionalState NewState)
{
    /* CR1 used: bit15, 14, 13, 12, 11-10, 9-8, 7-0, clear bit14 here */
    uint32_t tmp = TSPIx->CR1 & CR1_TRXE_MASK;

    /* Check the parameters */
    assert_param(IS_TSPI_UNIT(TSPIx));
    assert_param(IS_FUNCTIONAL_STATE(NewState));

    /* Write '1' to bit 14 to enable, write '0' to disable */
    tmp |= (uint32_t) NewState << 14U;
    TSPIx->CR1 = tmp;
}

/**
  * @brief Select TSPI mode: as SIO or SPI.
  * @param TSPIx: Select TSPI unit.
  *   This parameter can be one of the following values:
  *   TSB_TSPI0, TSB_TSPI1.
  * @param Mode: Specify the mode for TSPI.
  *   This parameter can be one of the following values:
  *   TSPI_MODE_SPI or TSPI_MODE_SIO.  
  * @retval None.  
  */
void TSPI_SelectMode(TSB_TSPI_TypeDef * TSPIx, TSPI_Mode Mode)
{
    /* CR1 used: bit15, 14, 13, 12, 11-10, 9-8, 7-0, clear bit13 here */
    uint32_t tmp = TSPIx->CR1 & CR1_TSPIMS_MASK;

    /* Check the parameters */
    assert_param(IS_TSPI_UNIT(TSPIx));
    assert_param(IS_TSPI_MODE(Mode));

    /* Write '0' to bit 13 for SPI, write '1' for SIO */
    tmp |= (uint32_t) Mode << 13U;
    TSPIx->CR1 = tmp;
}

/**
  * @brief Select TSPI transfer mode
  * @param TSPIx: Select TSPI unit.
  *   This parameter can be one of the following values:
  *   TSB_TSPI0, TSB_TSPI1.
  * @param TrMode: Specify the transfer mode for TSPI.
  *   This parameter can be one of the following values:
  *   TSPI_TRMODE_TX, TSPI_TRMODE_RX or TSPI_TRMODE_TXRX.  
  * @retval None.  
  */
void TSPI_SelectTransferMode(TSB_TSPI_TypeDef * TSPIx, TSPI_TransferMode TrMode)
{
    /* CR1 used: bit15, 14, 13, 12, 11-10, 9-8, 7-0, clear bit11-10 here */
    uint32_t tmp = TSPIx->CR1 & CR1_TMMD_MASK;

    /* Check the parameters */
    assert_param(IS_TSPI_UNIT(TSPIx));
    assert_param(IS_TSPI_TRANSFERMODE(TrMode));

    /* Write 'TrMode' to bit11-10 */
    tmp |= (uint32_t) TrMode << 10U;
    TSPIx->CR1 = tmp;
}

/**
  * @brief Set the number of transfer frame. 
  * @param TSPIx: Select TSPI unit.
  *   This parameter can be one of the following values:
  *   TSB_TSPI0, TSB_TSPI1.
  * @param Num: Specify the number of transferring.
  *   This parameter can be one of the following values:
  *   0: One frame (Single transfer)  
  *   1-255: 1 to 255 frames (Burst transfer)  
  * @retval None.  
  */
void TSPI_SetTransferNum(TSB_TSPI_TypeDef * TSPIx, uint8_t Num)
{
    /* CR1 used: bit15, 14, 13, 12, 11-10, 9-8, 7-0, clear bit7-0 here */
    uint32_t tmp = TSPIx->CR1 & CR1_FC_MASK;

    /* Check the parameters */
    assert_param(IS_TSPI_UNIT(TSPIx));

    /* Write 'Num' to bit7-0 */
    tmp |= (uint32_t) Num;
    TSPIx->CR1 = tmp;
}

/**
  * @brief Set the FIFO fill level to generate transfer(Tx or Rx) interrupt. 
  * @param TSPIx: Select TSPI unit.
  *   This parameter can be one of the following values:
  *   TSB_TSPI0, TSB_TSPI1.
  * @param TxRx: Specify transmit or receive.
  *   This parameter can be one of the following values:
  *   TSPI_TX, TSPI_RX
  * @param Level: Specify the FIFO fill level to generate interrupt.
  *   This parameter can be one of the following values:
  *   0 to 8  (depended on the Data format)
  * @retval None.  
  */
void TSPI_SetFIFOLevelINT(TSB_TSPI_TypeDef * TSPIx, uint8_t TxRx, uint8_t Level)
{
    /* read CR2 */
    uint32_t tmp = TSPIx->CR2;

    /* Check the parameters */
    assert_param(IS_TSPI_UNIT(TSPIx));
    assert_param(IS_TSPI_TXRX(TxRx));
    assert_param(IS_TSPI_FIFOLEVELINT(Level));

    if (TxRx == TSPI_TX) {
        /* CR2 used: bit23-22, 21, 15-12, 11-8, 7,6,5,4, 2,1,0.  clear bit15-12 here */
        tmp &= CR2_TIL_MASK;
        tmp |= (uint32_t) Level << 12U;
    } else {                    /* is TSPI_RX */
        /* CR2 used: bit23-22, 21, 15-12, 11-8, 7,6,5,4, 2,1,0.  clear bit11-8 here */
        tmp &= CR2_RIL_MASK;
        tmp |= (uint32_t) Level << 8U;
    }

    TSPIx->CR2 = tmp;
}

/**
  * @brief Enable/Disable the interrupt source
  * @param TSPIx: Select TSPI unit.
  *   This parameter can be one of the following values:
  *   TSB_TSPI0, TSB_TSPI1.
  * @param IntSrc: Specify the interrupt source to be set
  *   This parameter can be one of the following values or their combination:
  *   TSPI_INT_ERR, 
  *   TSPI_INT_RXEND, TSPI_INT_RXFIFO,
  *   TSPI_INT_TXEND, TSPI_INT_TXFIFO, 
  *   TSPI_INT_ALL (all above interrupt source)    
  * @param NewState: Specify the state for interrupt control.
  *   This parameter can be one of the following values:
  *   ENABLE or DISABLE.   
  * @retval None.  
  */
void TSPI_SetINT(TSB_TSPI_TypeDef * TSPIx, uint32_t IntSrc, FunctionalState NewState)
{
    /* CR2 used: bit23-22, 21, 15-12, 11-8, 7,6,5,4, 2,1,0. */
    uint32_t tmp = TSPIx->CR2 & CR2_INT_MASK;

    /* Check the parameters */
    assert_param(IS_TSPI_UNIT(TSPIx));
    assert_param(IS_TSPI_INTSRC(IntSrc));
    assert_param(IS_FUNCTIONAL_STATE(NewState));
    if (NewState == ENABLE) {
        /* Write 'IntSrc' to bit7,6,5,4, 2 */
        tmp |= IntSrc;
    } else {
        tmp &= ~IntSrc;
    }
    TSPIx->CR2 = tmp;
}

/**
  * @brief Set the clock and divider for baud rate generator
  * @param TSPIx: Select TSPI unit.
  *   This parameter can be one of the following values:
  *   TSB_TSPI0, TSB_TSPI1.
  * @param Clk: Specify the input clock to the baud rate generator.
  *   This parameter can be one of the following values:
  *   TSPI_BR_CLOCK_0,     TSPI_BR_CLOCK_1
  *   TSPI_BR_CLOCK_2,     TSPI_BR_CLOCK_4  
  *   TSPI_BR_CLOCK_8,     TSPI_BR_CLOCK_16
  *   TSPI_BR_CLOCK_32,    TSPI_BR_CLOCK_64 
  *   TSPI_BR_CLOCK_128,   TSPI_BR_CLOCK_256
  * @param Divider: Specify the division ratio "N" 
  *   This parameter can be one of the following values:
  *   TSPI_BR_DIVIDER_16,  TSPI_BR_DIVIDER_1
  *   TSPI_BR_DIVIDER_2,   TSPI_BR_DIVIDER_3
  *   TSPI_BR_DIVIDER_4,   TSPI_BR_DIVIDER_5
  *   TSPI_BR_DIVIDER_6,   TSPI_BR_DIVIDER_7
  *   TSPI_BR_DIVIDER_8,   TSPI_BR_DIVIDER_9
  *   TSPI_BR_DIVIDER_10,  TSPI_BR_DIVIDER_11
  *   TSPI_BR_DIVIDER_12,  TSPI_BR_DIVIDER_13
  *   TSPI_BR_DIVIDER_14,  TSPI_BR_DIVIDER_15
  * @retval None.  
  */
void TSPI_SetBaudRate(TSB_TSPI_TypeDef * TSPIx, TSPI_BaudClock Clk, TSPI_Divider Divider)
{
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_TSPI_UNIT(TSPIx));
    assert_param(IS_TSPI_BAUDCLOCK(Clk));
    assert_param(IS_TSPI_DIVIDER(Divider));

    tmp = (uint32_t) Clk << 4U;
	tmp |= (uint32_t) Divider << 0U;
    TSPIx->BR = tmp;
}

/**
  * @brief Initialize the specified TSPI channel 
  * @param TSPIx: Select TSPI unit.
  *   This parameter can be one of the following values:
  *   TSB_TSPI0, TSB_TSPI1.
  * @param Init: The structure containing basic TSPI configuration.
  * @retval None.  
  * NOTE: call TSPI_SetTransferNum() at first.
  */
void TSPI_Init(TSB_TSPI_TypeDef * TSPIx, TSPI_InitTypeDef * Init)
{
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_TSPI_UNIT(TSPIx));
    assert_param(IS_POINTER_NOT_NULL(Init));
    assert_param(IS_TSPI_DATADIRECTION(Init->DataDirection));
    assert_param(IS_TSPI_FRAMELENGTH(Init->FrameLength));
    assert_param(IS_TSPI_CYCLEBETWEENFRAMES(Init->CycleBetweenFrames));
    assert_param(IS_TSPI_CSPOLARITY(Init->CS3Polarity));
    assert_param(IS_TSPI_CSPOLARITY(Init->CS2Polarity));
    assert_param(IS_TSPI_CSPOLARITY(Init->CS1Polarity));
    assert_param(IS_TSPI_CSPOLARITY(Init->CS0Polarity));
    assert_param(IS_TSPI_CLOCKPHASE(Init->ClockPhase));
    assert_param(IS_TSPI_CLOCKPOLARITY(Init->ClockPolarity));
    assert_param(IS_TSPI_SHORTESTIDLECYCLE(Init->ShortestIdleCycle));
    assert_param(IS_TSPI_DELAYCYCLENUM(Init->DelayCycleNum_CS_SCLK));
    assert_param(IS_TSPI_DELAYCYCLENUM(Init->DelayCycleNum_Negate_CS));
    assert_param(IS_TSPI_HOLDTIME(Init->LastBit_Hold_Time));
    assert_param(IS_FUNCTIONAL_STATE(Init->ParityCheck));
    assert_param(IS_TSPI_PARITYTYPE(Init->Parity));

    tmp = (uint32_t) Init->DataDirection << 31U;
    tmp |= (uint32_t) Init->FrameLength << 24U;
    tmp |= (uint32_t) Init->CycleBetweenFrames << 20U;
    tmp |= (uint32_t) Init->CS3Polarity << 19U;
    tmp |= (uint32_t) Init->CS2Polarity << 18U;
    tmp |= (uint32_t) Init->CS1Polarity << 17U;
    tmp |= (uint32_t) Init->CS0Polarity << 16U;
    tmp |= (uint32_t) Init->ClockPhase << 15U;
    tmp |= (uint32_t) Init->ClockPolarity << 14U;
    tmp |= (uint32_t) Init->ShortestIdleCycle << 10U;
    tmp |= ((uint32_t) Init->DelayCycleNum_CS_SCLK - 1U) << 4U;
    tmp |= ((uint32_t) Init->DelayCycleNum_Negate_CS - 1U) << 0U;
    TSPIx->FMTR0 = tmp;

    tmp = (uint32_t) Init->LastBit_Hold_Time << 4U;
    tmp |= (uint32_t) Init->ParityCheck << 1U;
    tmp |= (uint32_t) Init->Parity << 0U;
    TSPIx->FMTR1 = tmp;
}

/**
  * @brief Get the TSPI status.
  * @param TSPIx:  Select TSPI unit.
  *   This parameter can be one of the following values:
  *   TSB_TSPI0, TSB_TSPI1.
  * @retval The Tx/Rx FIFO, Run, Interrupt status of TSPI unit.
  */
TSPI_StatusFlag TSPI_GetStatus(TSB_TSPI_TypeDef * TSPIx)
{
    TSPI_StatusFlag tmp = { 0U };

    /* Check the parameters */
    assert_param(IS_TSPI_UNIT(TSPIx));

    /* the unused bit is read as '0', no need to mask them */
    tmp.All = TSPIx->SR;
    return tmp;
}

/** @} */
/* End of group TSPI_Exported_Functions */

/** @} */
/* End of group TSPI */

/** @} */
/* End of group TX04_Periph_Driver */

