/**
 ****************************************************************************
 * @file:	 ipdefine.h
 * @brief:	 Application Definitions
 * @version  V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2023
 *****************************************************************************
 */
#ifndef _IPDEFINE_H
#define _IPDEFINE_H

#include <arm_math.h>

/* Select MCU */
#if (defined(__MCU_M4KLA )||defined(__MCU_M4KMA) ||defined(__MCU_M4KNA))
#include "TMPM4KyA.h"
#elif (defined(__MCU_M4MLA) ||defined(__MCU_M4MMA) || defined(__MCU_M4MNA))
#include "TMPM4MxA.h"
#endif

#include "sys_macro.h"
#include "D_Para.h"
#include "ipdrv_adc.h"

/* #define	__MOTOR_DEBUG	*/					/* Motor debug */
/* #define	__MOTOR_DBGOUT_VECTOR_TIME_INT	*/	/* Debug output for vector control process time */
/* #define	__MOTOR_DBGOUT_IDET	*/				/* Debug output for current detection status */
/* #define	__MOTOR_PMDDBG	*/					/* Debug output for PMDxDBGOUT */
/* #define	__COM_DBGOUT_TIME_INT	*/			/* Debug output for communication process time */

#ifdef DEFINE_APP
#define EXTERN
#else
#define EXTERN	extern
#endif


/*===================================================================*
	  MCU Parameter Definition
 *===================================================================*/
/* Setting fc */
#if (defined(__TMPM4KNA_H__))
#define	cIMCLK_FRQ					(80)	/* [MHz] */
#elif (defined(__TMPM4KLA_H__))
#define	cIMCLK_FRQ					(80)	/* [MHz] */
#endif

/* Setting INTVC name */
#if (defined(__TMPM370_H__) || defined(__TMPM372_H__) || defined(__TMPM373_H__) || defined(__TMPM374_H__) || defined(__TMPM376_H__) || defined(__TMPM375_H__) || defined(__TMPM37A_H__))
#define	INTVCN0_IRQn			INTVCNA_IRQn
#define	INTVCN1_IRQn			INTVCNB_IRQn
#define	INTVCN0_IRQHandler		INTVCNA_IRQHandler
#define	INTVCN1_IRQHandler		INTVCNB_IRQHandler
#endif

/* Setting PMD type */
#if (defined(__TMPM370_H__) || defined(__TMPM372_H__) || defined(__TMPM373_H__) || defined(__TMPM374_H__) || defined(__TMPM376_H__) || defined(__TMPM375_H__) || defined(__TMPM37A_H__) || defined(__TMPM470_H__) || defined(__TMPM475_H__))
#define	__PMD
#elif (defined(__TMPM4K4A_H__) || defined(__TMPM4KN_H__))
#define	__A_PMD
#elif (defined(__TMPM4KNA_H__) || defined(__TMPM4KLA_H__))
#define	__A_PMD
#endif

/* Setting VE type */
#if (defined(__TMPM370_H__) || defined(__TMPM372_H__) || defined(__TMPM373_H__) || defined(__TMPM374_H__) || defined(__TMPM376_H__))
#define	__VE		/* VE */
#elif (defined(__TMPM375_H__) || defined(__TMPM37A_H__))
#define	__VE_P		/* VE Plus */
#elif (defined(__TMPM470_H__) || defined(__TMPM475_H__))
#define	__A_VE		/* Advanced VE */
#elif (defined(__TMPM4K4A_H__) || defined(__TMPM4KN_H__) || defined(__TMPM4KNA_H__) || defined(__TMPM4KLA_H__))
#define	__A_VE_P	/* Advanced VE Plus */
#endif

/* Setting ADC type */
#if (defined(__TMPM370_H__) || defined(__TMPM372_H__) || defined(__TMPM373_H__) || defined(__TMPM374_H__) || defined(__TMPM376_H__) || defined(__TMPM375_H__) || defined(__TMPM37A_H__))
#define	__ADC
#elif (defined(__TMPM470_H__) || defined(__TMPM475_H__))
#define	__ADC2
#elif (defined(__TMPM4K4A_H__) || defined(__TMPM4KN_H__) || defined(__TMPM4KNA_H__) || defined(__TMPM4KLA_H__))
#define	__ADCZ
#endif

/* Setting ENC type */
#if (defined(__TMPM370_H__) || defined(__TMPM372_H__) || defined(__TMPM373_H__) || defined(__TMPM374_H__) || defined(__TMPM376_H__) || defined(__TMPM375_H__))
#define	__ENC
#elif (defined(__TMPM470_H__) || defined(__TMPM475_H__) || defined(__TMPM4K4A_H__) || defined(__TMPM4KN_H__) || defined(__TMPM4KNA_H__))
#define	__A_ENC
#elif defined(__TMPM4KLA_H__)
#define   __NO_ENC
#endif

/* Setting AMP type */
#if (defined(__TMPM370_H__) || defined(__TMPM375_H__) || defined(__TMPM37A_H__) || defined(__TMPM4K4A_H__) || defined(__TMPM4KN_H__) || defined(__TMPM4KNA_H__) || defined(__TMPM4KLA_H__))
#define	__AMP
#endif

/* Setting AMP Unit */
#if defined(__TMPM370_H__)
#define	__AMP_UNIT_A
#define	__AMP_UNIT_B
#define	__AMP_UNIT_C
#define	__AMP_UNIT_D
#elif (defined(__TMPM375_H__) || defined(__TMPM37A_H__) || defined(__TMPM4K4A_H__) || defined(__TMPM4KN_H__) || defined(__TMPM4KNA_H__) || defined(__TMPM4KLA_H__))
#define	__AMP_UNIT_D
#endif

/* Setting CMP type */
#if defined(__TMPM370_H__)
#define	__CMP
#endif

/* Setting CMP Unit */
#if defined(__TMPM370_H__)
#define	__CMP_UNIT_A
#define	__CMP_UNIT_B
#define	__CMP_UNIT_C
#define	__CMP_UNIT_D
#endif



/* Setting SIO/UART type */
#if (defined(__TMPM370_H__) || defined(__TMPM372_H__) || defined(__TMPM373_H__) || defined(__TMPM374_H__) || defined(__TMPM376_H__) || defined(__TMPM4K4A_H__) || defined(__TMPM4KN_H__) || defined(__TMPM4KNA_H__) || defined(__TMPM4KLA_H__))
#define	__SIO
#define	__UART
#elif (defined(__TMPM375_H__) || defined(__TMPM37A_H__))
#define	__SIO2
#define	__UART2
#elif (defined(__TMPM470_H__) || defined(__TMPM475_H__))
#define	__SIO2
#define	__UART2
#endif

/* Setting PMD Unit */
#if (defined(__TMPM370_H__) || defined(__TMPM376_H__) || defined(__TMPM470_H__) || defined(__TMPM475_H__) || defined(__TMPM4K4A_H__) || defined(__TMPM4KN_H__) || defined(__TMPM4KNA_H__) || defined(__TMPM4KLA_H__))
#define	__PMD_FAN
#define	__PMD_COMP
#elif (defined(__TMPM372_H__) || defined(__TMPM373_H__) || defined(__TMPM374_H__) || defined(__TMPM375_H__) || defined(__TMPM37A_H__))
#define	__PMD_COMP
#endif

/* Setting VE Unit */
#if (defined(__TMPM370_H__) || defined(__TMPM376_H__) || defined(__TMPM470_H__) || defined(__TMPM475_H__))
#define	__VE_FAN
#define	__VE_COMP
#elif (defined(__TMPM372_H__) || defined(__TMPM373_H__) || defined(__TMPM374_H__) || defined(__TMPM375_H__) || defined(__TMPM37A_H__))
#define	__VE_COMP
#elif (defined(__TMPM4K4A_H__) || defined(__TMPM4KN_H__) || defined(__TMPM4KNA_H__) || defined(__TMPM4KLA_H__))
#define	__VE_FAN
#endif


/* Setting ADC Unit */
#if (defined(__TMPM4K4A_H__) || defined(__TMPM4KN_H__) || defined(__TMPM4KNA_H__) || defined(__TMPM4KLA_H__))
#define	__ADC_UNITA
#elif (defined(__TMPM370_H__) || defined(__TMPM376_H__) || defined(__TMPM470_H__) || defined(__TMPM475_H__))
#define	__ADC_UNITA
#define	__ADC_UNITB
#elif (defined(__TMPM372_H__) || defined(__TMPM373_H__) || defined(__TMPM374_H__) || defined(__TMPM375_H__) || defined(__TMPM37A_H__))
#define	__ADC_UNITB
#endif

/* Setting Enc Unit */
#if (defined(__TMPM370_H__) || defined(__TMPM376_H__) || defined(__TMPM470_H__) || defined(__TMPM475_H__))
#define	__ENC_CH0
#define	__ENC_CH1
#elif (defined(__TMPM372_H__) || defined(__TMPM373_H__) || defined(__TMPM374_H__))
#define	__ENC_CH1
#elif (defined(__TMPM375_H__) || defined(__TMPM4K4A_H__) || defined(__TMPM4KN_H__) || defined(__TMPM4KNA_H__) || defined(__TMPM4KLA_H__))
#define	__ENC_CH0
#endif

/* Setting SC Unit */
#if (defined(__TMPM370_H__) || defined(__TMPM376_H__) || defined(__TMPM372_H__) || defined(__TMPM470_H__) || defined(__TMPM475_H__))
#define	__SC_CH0
#define	__SC_CH1
#define	__SC_CH2
#define	__SC_CH3
#elif (defined(__TMPM373_H__) || defined(__TMPM374_H__))
#define	__SC_CH0
#define	__SC_CH1
#define	__SC_CH3
#elif defined(__TMPM375_H__)
#define	__SC_CH0
#define	__SC_CH1
#elif defined(__TMPM37A_H__)
#define	__SC_CH0
#endif

#if defined(__TMPM4KNA_H__)
#define __NBD
#elif defined(__TMPM4KLA_H__)
#endif

#if defined(__TMPM4KNA_H__)
#define __HW_TMPM4KNA
#elif defined(__TMPM4KLA_H__)
#define __HW_TMPM4KLA
#endif


/*------------------------------------------H/W Definition--------------------------------------------------*/
#if defined(__HW_TMPM4KNA)				

/*********************************************************************************************
=====================LED&g_fourWayValve Setting============================
	LED1---PA0;/LED2---PV1;/LED3---PV0;/LED4---PN2;/g_fourWayValve---PD4
************************************************************************************************/
#define cLED1_SHIFT 					0
#define cLED2_SHIFT 					1
#define cLED3_SHIFT 					0
#define cLED4_SHIFT 					2
#define cFWV_SHIFT						4
#define LED_CH1 						CG_FSYS_PERIPH_PORTA
#define LED_CH2 						CG_FSYS_PERIPH_PORTV
#define LED_CH3 						CG_FSYS_PERIPH_PORTV
#define LED_CH4 						CG_FSYS_PERIPH_PORTN
#define FWV_CG_CH						CG_FSYS_PERIPH_PORTD
#define LED1_CH 						TSB_PA
#define LED2_CH 						TSB_PV
#define LED3_CH 						TSB_PV
#define LED4_CH 						TSB_PN
#define FWV_CH							TSB_PD

#define LED1							TSB_PA_DATA_PA0
#define LED2							TSB_PV_DATA_PV1
#define LED3							TSB_PV_DATA_PV0
#define LED4							TSB_PN_DATA_PN2
#define FWVON							TSB_PD_DATA_PD4

/*************************************************************************************************
***************************************************************************************************
=====================PTC Setting============================
						PTC---PD5
***************************************************************************************************/
#if defined(__TYPE_HIGH_VOLT)			&& defined(__RELAY)

/* Relay Control Port Settings */
#define PO_RELAY_CG_FSYS_PERIPH_PORT	CG_FSYS_PERIPH_PORTD
#define PO_RELAY_INI					TSB_PD_DATA_PD5 = 0;\
		TSB_PD_CR_PD5C = 1;

#define PO_RELAY_OFF					TSB_PD_DATA_PD5 = 0;
#define PO_RELAY_ON 					TSB_PD_DATA_PD5 = 1;

#endif /* defined(__TYPE_HIGH_VOLT) && defined(__RELAY) */

/*****************************************************************************************************/

/******************************************************************************************************
=======================Temperature monitor Config==================
HVMOS---AINB3---REG0;/OUTROOM---AINC0---REG8;/OUTPIPE---AINC1---REG9
EXH---AINC2---REG10;/XQ---AINB4---REG8;/YJ---AINC3---REG11
********************************************************************************************************/
#define ADx0							TSB_ADB
#define ADx1							TSB_ADC
#define ADx2							TSB_ADC
#define ADx3							TSB_ADC
#define ADx4							TSB_ADB
#define ADx5							TSB_ADC

#define ADCx_REG0						ADC_REG0
#define ADCx_REG1						ADC_REG8
#define ADCx_REG2						ADC_REG9
#define ADCx_REG3						ADC_REG10
#define ADCx_REG4						ADC_REG1
#define ADCx_REG5						ADC_REG11

#define ADCx_AIN0						ADC_AIN3	/* HVMOS */
#define ADCx_AIN1						ADC_AIN0	/* OUT_ROOM */
#define ADCx_AIN2						ADC_AIN1	/* OUT_PIPE */
#define ADCx_AIN3						ADC_AIN2	/* EXH */
#define ADCx_AIN4						ADC_AIN4	/* Suction */
#define ADCx_AIN5						ADC_AIN3	/* Comp */

#define OUTDOORT_P						(TSB_ADC->REG8)
#define PIPE_P							(TSB_ADC->REG9)
#define EXH_P							(TSB_ADC->REG10)
#define HVMOS_P 						(TSB_ADB->REG0)

#define ST_SNGF0						TSB_ADC_ST_SNGF
#define ST_SNGF1						TSB_ADB_ST_SNGF
#define ST_SNGF2						TSB_ADA_ST_SNGF

/********************************************************************************************************/

/********************************************************************************************************
===============================PFC Config============================
IAC---AINC5;/PWM-OUTPUT-port-U02
VAC---AINC4;/PMD ch--PMD2
***********************************************************************************************************/
#define PFC_IAC_CH						(5)
#define PFC_VAC_CH						(4)
#define PFC_SAMPLE_CH					TSB_ADC
#define PFC_PWM_CH						TSB_PMD2
#define __U_PFCOUT

/**********************************************************************************************************/

/**********************************************END*******************************************************/
#elif (defined(__HW_TMPM4KLA))

/*********************************************************************************************
=====================LED&g_fourWayValve Setting============================
	LED1---PC2;/LED2---PC3;/LED3---PA2;/LED4---PU5;/g_fourWayValve---PG3
************************************************************************************************/
#define cLED1_SHIFT 					2
#define cLED2_SHIFT 					3
#define cLED3_SHIFT 					2
#define cLED4_SHIFT 					5

#define LED_CH1 						CG_FSYS_PERIPH_PORTC
#define LED_CH2 						CG_FSYS_PERIPH_PORTC
#define LED_CH3 						CG_FSYS_PERIPH_PORTA
#define LED_CH4 						CG_FSYS_PERIPH_PORTU

#define LED1_CH 						TSB_PC
#define LED2_CH 						TSB_PC
#define LED3_CH 						TSB_PA
#define LED4_CH 						TSB_PU

#define LED1							TSB_PC_DATA_PC2
#define LED2							TSB_PC_DATA_PC3
#define LED3							TSB_PA_DATA_PA2
#define LED4							TSB_PU_DATA_PU5


/***************************************************************************************************/

/***************************************************************************************************
=====================PTC Setting============================
						PTC---PL7
***************************************************************************************************/
#if defined(__TYPE_HIGH_VOLT)			&& defined(__RELAY)

/* Relay Control Port Settings */
#define PO_RELAY_CG_FSYS_PERIPH_PORT	CG_FSYS_PERIPH_PORTL
#define PO_RELAY_INI					TSB_PL_DATA_PL7 = 0;\
		TSB_PL_CR_PL7C = 1;

#define PO_RELAY_OFF					TSB_PL_DATA_PL7 = 0;
#define PO_RELAY_ON 					TSB_PL_DATA_PL7 = 1;

#endif /* defined(__TYPE_HIGH_VOLT) && defined(__RELAY) */

/********************************************************************************************************/

/******************************************************************************************************
=======================Temperature monitor Config==================
HVMOS---AINC2---REG8;/OUTROOM---AINB0---REG0;/OUTPIPE---AINB2---REG2
EXH---AINB1---REG1;/SU---AINA13---REG8;/Comp---AINA18---REG9
********************************************************************************************************/
#define ADx0							TSB_ADC
#define ADx1							TSB_ADB
#define ADx2							TSB_ADB
#define ADx3							TSB_ADB
#define ADx4							TSB_ADA
#define ADx5							TSB_ADA

#define ADCx_REG0						ADC_REG8
#define ADCx_REG1						ADC_REG0
#define ADCx_REG2						ADC_REG1
#define ADCx_REG3						ADC_REG2
#define ADCx_REG4						ADC_REG8
#define ADCx_REG5						ADC_REG9

#define ADCx_AIN0						ADC_AIN2	/* HVMOS */
#define ADCx_AIN1						ADC_AIN0	/* OUT_ROOM */
#define ADCx_AIN2						ADC_AIN2	/* OUT_PIPE */
#define ADCx_AIN3						ADC_AIN1	/* EXH */
#define ADCx_AIN4						ADC_AIN13	/* DIODE */
#define ADCx_AIN5						ADC_AIN18	/* IGBT */

#define OUTDOORT_P						(TSB_ADB->REG0)
#define PIPE_P							(TSB_ADB->REG1)
#define EXH_P							(TSB_ADB->REG2)
#define HVMOS_P 						(TSB_ADC->REG8)
#define IGBT_P							(TSB_ADA->REG9)
#define DIODE_P 						(TSB_ADA->REG8)

#define ST_SNGF0						(TSB_ADB_ST_SNGF)
#define ST_SNGF1						(TSB_ADC_ST_SNGF)
#define ST_SNGF2						(TSB_ADA_ST_SNGF)

/*******************************************************************************************************/

/********************************************************************************************************
===============================PFC Config========================
IAC---AINC5;/PWM-OUTPUT-port-U02
VAC---AINC4;/PMD ch--PMD2
***********************************************************************************************************/
#define PFC_IAC_CH						(1)
#define PFC_VAC_CH						(0)
#define PFC_SAMPLE_CH					TSB_ADC
#define PFC_PWM_CH						TSB_PMD2
#define __W_PFCOUT

/********************************************END*******************************************************/
#endif

/********************************************HW DEFINE END*******************************************************/

/*===================================================================*
	  Local Constant Data Definition
 *===================================================================*/
#define cV_RELAY_ON 					(100.0f * 0.9f)	/* [V] Relay On Voltage */
#define cRELAY_LEN						(1.0)			/* [s] Time of Voltage check */
#define cRELAY_LIM						(cRELAY_LEN*10)	/* [s] Time of limitation */
#define cPOWER_CNT						(4000)			/* [s] power on time */

/*===================================================================*
	  Local Variable Definition
 *===================================================================*/
EXTERN uint8_t relay_state;

/*===================================================================*
	  Interruption leve define
 *===================================================================*/
/* High   Low */
/*	0 ---- 7  */

#define	INT4KH_LEVEL			5				/* 4kH interval timer interrupt */
#define	INT_EMG0_LEVEL			1
#define	INT_EMG1_LEVEL			1
#define	INT_EMG2_LEVEL			1
#define	INT_VC_LEVEL			4				/* VE interrupt */
#define	INT_ADC_LEVEL			5				/* ADC interrupt */
#define	INT_DAC_LEVEL			6				/* SIO interrupt for Dac */
#define	INT_PMD1_LEVEL			5
#define	INT_UART_LEVEL			7



/*****************************************************************
 * Constant data set
 *****************************************************************/
#if defined(__A_PMD)
  #define PWMRate(x)			((uint16_t)((1 << 24U) / (cIMCLK_FRQ * x)))
  #define DeadTime(x)			((uint8_t)((cIMCLK_FRQ * 1000000U * x) / 1000000U / 4U))
#elif defined(__PMD)
  #define PWMPeriod(x)			((uint16_t)((cIMCLK_FRQ * x) / 2U))		/* triangle wave */
  #define DeadTime(x)			((uint8_t)((cIMCLK_FRQ * 1000000U * x)	/ 1000000U / 8U))
#endif

#define ADC_V2HEX(x)			((uint16_t)((uint16_t)((0xfff * x) / 5) << 4U))

/* TMRB */
#define	TMRB_DIV2_PRD(x)		(uint16_t)( (uint32_t)x * cIMCLK_FRQ / 2)
#define	TMRB_DIV8_PRD(x)		(uint16_t)( (uint32_t)x * cIMCLK_FRQ / 8)
#define	TMRB_DIV32_PRD(x)		(uint16_t)( (uint32_t)x * cIMCLK_FRQ / 32)
#define	TMRB_DIV64_PRD(x)		(uint16_t)( (uint32_t)x * cIMCLK_FRQ / 64)
#define	TMRB_DIV128_PRD(x)		(uint16_t)( (uint32_t)x * cIMCLK_FRQ / 128)
#define	TMRB_DIV256_PRD(x)		(uint16_t)( (uint32_t)x * cIMCLK_FRQ / 256)
#define	TMRB_DIV512_PRD(x)		(uint16_t)( (uint32_t)x * cIMCLK_FRQ / 512)

#define	T32A_DIV2_PRD(x)		(uint16_t)( (uint32_t)x * cIMCLK_FRQ /	  2ULL)
#define	T32A_DIV1_PRD(x)		(uint16_t)( (uint32_t)x * cIMCLK_FRQ)

#define SQT2					(0.707f)
#define cARRAY_MAX				(10)			/*Error array MAX Value*/

/****************************************************************************/
/*Error priority level setting*/
/***************************************************************************/
#define cFAN_I_HW_LEVEL 				1 /*fan hw overcurrent priority*/
#define cFAN_I_SW_LEVEL 				2 /*fan sw overcurrent priority*/
#define cCOMP_I_HW_LEVEL				1 /*comp hw overcurrent priority*/
#define cCOMP_I_SW_LEVEL				2 /*comp  sw overcurrent priority*/

#define cHPFC_ZC_LEVEL					1 /*hpfc zero cross detect priority*/
#define cHPFC_HZ_LEVEL					2 /*hpfc hz detect priority*/
#define cHPFC_I_HW_LEVEL				3 /*hpfc hw overcurrent priority*/
#define cHPFC_I_SW_LEVEL				4 /*hpfc sw overcurrent priority*/
#define cHPFC_OVER_VAC_LEVEL			5 /*hpfc over vac priority*/
#define cHPFC_UNDER_VAC_LEVEL			6 /*hpfc under vac priority*/
#define cHPFC_OVER_VDC_LEVEL			7 /*hpfc over vdc priority*/
#define cHPFC_UNDER_VDC_LEVEL			8 /*hpfc under vdc priority*/



#define cTEMP_OS_LEVEL					1 /*temperature sencor open/short display priority*/
#define cTEMP_OVER_LEVEL				2 /*temperature over display priority*/
#define cMOTOR_POWER_LEVEL				3/*over Power display priority*/

/*****************************************************************
 * Define The Typedef
 *****************************************************************/

typedef enum
{
	cStop		  = 0,
	cBootstrap	  = 1,
	cInitposition = 2,
	cForce		  = 3,
	cChange_up	  = 4,
	cSteady_A	  = 5,
	cEmergency	  = 6
} main_stage_e;

typedef enum
{
	cStep0		  = 0,
	cStep1		  = 1,
	cStep2		  = 2,
	cStep3		  = 3,
	cStep4		  = 4,
	cStepEnd	  = 5
} sub_stage_e;

typedef enum
{
	ciStop				= 0,
	ciBootstrap			= 1,
	ciInitposition_i	= 2,
	ciInitposition_v	= 3,
	ciForce_i			= 4,
	ciForce_v			= 5,
	ciChange_up			= 6,
	ciSteady_A			= 7,
	ciEmergency			= 8
} itr_stage_e;

typedef enum
{
	c1shunt		= 1,
	c3shunt		= 3,
	cPFCshunt	= 4
} shunt_type_e;

typedef enum
{
	cBoot_i		= 0,
	cBoot_v		= 1
} boot_type_e;

typedef enum
{
	cPHCVMD_SVM			= 0,
	cPHCVMD_I_CLARKE	= 1
} phcvmd_e;

typedef enum
{
	cTRGPOS_3SHUNT				= 0,
	cTRGPOS_1SHUNT_FIRST_HALF	= 1,
	cTRGPOS_1SHUNT_SECOND_HALF	= 2
} trgpos_e;

typedef struct
{
	main_stage_e main;
	sub_stage_e sub;
	itr_stage_e itr;
} stage_t;


typedef enum 
{
	cError_None_Err = 0, 
	cError_FAN_I_HW = 1, 
	cError_FAN_I_SW = 2, 
	cError_FAN_LostPos = 3, 
	cError_FAN_LostPha = 4, 
	cError_COMP_I_HW = 5, 
	cError_COMP_I_SW = 6, 
	cError_COMP_LostPos = 7, 
	cError_COMP_LostPha = 8, 
	cError_HPFC_I_HW = 9, 
	cError_HPFC_I_SW = 10, 
	cError_HPFC_VAC_OVER = 11, 
	cError_HPFC_VAC_UNDER = 12, 
	cError_HPFC_VDC_OVER = 13, 
	cError_HPFC_VDC_UNDER = 14, 
	cError_HPFC_HZ = 15, 
	cError_HPFC_ZC = 16, 
	cError_Temp_OS = 17, 
	cError_Over_Temp = 18, 
	cError_Uart = 19, 
	cError_Over_Power = 20, 
} error_type_e;


typedef struct 
{
	error_type_e type;
	uint8_t priority;
	uint8_t recovery_type;
} error_type_t;


typedef enum 
{
	cCOMP = 0, 
	cFAN = 1
} motortype_e;


typedef struct 
{
	struct 
	{
	    error_type_t error_type_buf[cARRAY_MAX];
	    uint8_t pt;
	} fan;


	struct 
	{
	    error_type_t error_type_buf[cARRAY_MAX];
	    uint8_t pt;
	} comp;


	struct 
	{
	    error_type_t error_type_buf[cARRAY_MAX];
	    uint8_t pt;
	} hpfc;


	struct 
	{
	    error_type_t error_type_buf[cARRAY_MAX];
	    uint8_t pt;
	} ac_logical;


	error_type_t fan_i_hw_error;
	error_type_t fan_i_sw_error;

	error_type_t comp_i_hw_error;
	error_type_t comp_i_sw_error;

	error_type_t hpfc_i_hw_error;
	error_type_t hpfc_i_sw_error;
	error_type_t hpfc_over_vac_error;
	error_type_t hpfc_under_vac_error;
	error_type_t hpfc_over_vdc_error;
	error_type_t hpfc_under_vdc_error;
	error_type_t hpfc_hz_error;
	error_type_t hpfc_zc_error;

	error_type_t ac_logical_temp_os_error;
	error_type_t ac_logical_temp_over_error;
	error_type_t ac_logical_uart_error;
	error_type_t ac_logical_power_error;
	error_type_t none_error;
} error_t;


#define	c3PHASE		0
#define	c2PHASE		1


typedef struct
{
	uint16_t reserve: 9;						/* reserve */
	uint16_t spwm: 2;							/* Shift PWM		0=off 					1=on								*/
	uint16_t spwm_pre: 2;						/* Shift PWM		0=off					1=on								*/
	uint16_t modul: 1;							/* PWM Moduration	0=3phase modulation 	1=2phase modulation 				*/
	uint16_t encoder: 1;						/* Position detect	0=Current				1=Encoder							*/
	uint16_t onoff: 1;							/* PWM output		0=off					1=on								*/
} command_t;

typedef struct
{
	uint16_t reserve: 9;						/* reserve */
	uint16_t F_vcomm_theta: 1;					/* Omega to Theta	0=command value 		1=Calculate the theta from omega.	*/
	uint16_t F_vcomm_omega: 1;					/* Omega by 		0=command value 		1=Result of Estimation position 	*/
	uint16_t F_vcomm_current: 1;				/* Current by		0=command value 		1=Result of Speed Control			*/
	uint16_t F_vcomm_volt: 1; 					/* Voltage by		0=command value 		1=Result of Current Control 		*/
	uint16_t F_vcomm_Edetect: 1;				/* Position detect	0=off					1=on								*/
	uint16_t F_vcomm_Idetect: 1;				/* Current detect	0=off					1=on								*/
	uint16_t F_vcomm_onoff: 1;					/* Motor output		0=off					1=on								*/
} vectorcmd_t;


typedef union
{
	struct
	{
    	uint16_t reserve: 8;					/* reserve */
    	uint16_t Loss_sync: 1;					/* 0:normal, 1: Loss of synchronism 	*/
    	uint16_t emg_I: 1;						/* 0:normal, 1: Current detect error	*/
    	uint16_t emg_S: 1;						/* 0:normal, 1: Over current(soft)		*/
    	uint16_t emg_H: 1;						/* 0:normal, 1: Over current(hard)		*/
    	uint16_t pfc_relate: 1;					/* 0:normal, 1: PFC error				*/
    	uint16_t Loss_phase: 1;					/* 0:normal, 1: Loss of phase					*/
    	uint16_t over_power: 1;					/* Power over */
    	uint16_t emg_reverse: 1;				/* UVW reverse*/
	} flg;
	uint16_t	all;
} state_t;

typedef union
{
	struct
	{
    	uint16_t reserve				: 14;		/* reserve */
    	uint16_t emg_temp_os			: 1;		/* 0:normal, 1: Loss of synchronism 			*/
    	uint16_t emg_temp_over			: 1;
	} flg;
    
	uint16_t all;
} t_state_t;


typedef enum 
{
	cERR_NOT = 0,									/* NOT  Err Nothing */
	cERR_IH_OVER = 1,								/* E10  Vac Over Voltage */
	cERR_IS_OVER = 2,								/* E11  Vac Under Voltage */
	cERR_LOST_POSITION = 3, 						/* E21  Vdc Over Voltage */
	cERR_LOST_PHASE = 4,							/* E22  Vdc Under Voltage */
} err_mode_e;


typedef struct
{
	stage_t				stage;					/* Motor Driver Stage */
	uint8_t up_down_speed_flg;						/*up/down speed flag 0:up;1:down*/
	uint8_t shift_flg;								/* */
	uint8_t Release_PI; 							/*PI Control Flag 0: Auto; 1:Mannual */
	uint8_t Error_flag_UW[2];						/*motor sample error check flag */
	uint8_t Error_flag_UV[2];						/*motor sample error check flag */
	uint8_t Error_flag[2];							/*motor sample error check flag array */
	uint32_t spd_restart_cnt;						/*3mins count for motor restart */
	uint16_t spwm_prepare;							/*Pwm mode switch count */
	uint8_t cntIa;									/*current sample filter count a */
	uint8_t cntIb;									/*current sample filter count b */
	uint8_t cntIc;									/*current sample filter count c */
	uint8_t filter_shift1;							/*current filter shift1 */
	uint8_t filter_shift2;							/*current filter shift1 */
	q15_t current_spd;							/*[rps]Motor current spd*/
	q15_t spd_power_limit;						/*[rps]Motor spd limit by power limit*/
	uint8_t s_valve_onoff;						/*STF on/off*/
	uint8_t valve_onoff;							/*STF status*/

	struct 
	{
    	uint16_t restart_count; 					/*Fan motor restart failture and restart count */
    	uint8_t spd_fan_pre;						/*[Hz]fan speed pre val */
    	uint8_t spd_fan;							/*[Hz]fan speed */
	} fan_control_spd;


#if defined(__MOTOR_DEBUG)					
	main_stage_e stage_his[5];						/*stage history */
#endif

	struct
	{
		q31_u			omega_com;				/* [Hz/maxHz] Command of Omega(electrical angle) */
		q31_u			omega;					/* [Hz/maxHz] Omega(speed): Electrical angle */
		q15_t			omega_dev;				/* [Hz/maxHz] Omega dev			 			*/

		q31_u			Id_com;					/* [A/maxA] Command of d-axis current */
		q31_u			Iq_com;					/* [A/maxA] Command of q-axis current */
		q15_t			Id_ref;					/* [A/maxA] Referense of d-axis current */
		q15_t			Iq_ref;					/* [A/maxA] Referense of q-axis current */
		q15_t Iq_ref_PI;							/* [A/maxA] Referense of q-axis current */
		q15_t R_Iq_ref1_Bak;						/*[A/maxA] Iq_ref backup */
		q15_t			Id;						/* [A/maxA] d-axis current */
		q15_t			Iq;						/* [A/maxA] q-axis current */
		q15_t Id_temp0; 							/* [A/maxA] d-axis current */
		q15_t Iq_temp0; 							/* [A/maxA] q-axis current */
		q15_t Id_temp1; 							/* [A/maxA] d-axis current */
		q15_t Iq_temp1; 							/* [A/maxA] q-axis current */

		q31_u			Iq_ref_I; 				/* [A/maxA] Referense of q-axis current (Integral) */

		uint16_t		theta_com;				/* [deg/360] Command of theta of motor */
		uint32_u		theta;					/* [deg/360] Theta(position): Electrical angle */
		uint32_u theta_pre; 						/* [deg/360] Theta(position): Electrical angle pre val */

		uint8_t sector; 							/*motor sector val */

		q15_t			Vdc;					/* [V/maxV] DC voltage */
		q31_u			Vdc_ave;				/* [V/maxV] DC voltage average */
		q15_t			Vdc_temp; 				/* [V/maxV]vdc tmp val */

		q31_u			Vd;						/* [V/maxV] d-axis voltage */
		q31_u			Vq;						/* [V/maxV] q-axis voltage */

		q15_t			Vdq;					/* [V/maxV] dq-axis voltage root3 */
		q31_u			Vdq_ave;				/* [V/maxV] dq-axis voltage root3 average */

		q31_t			Vd_out;					/* [V/maxV] d-axis output voltage for VOLT BOOT */

		q15_t			Ed;						/* [V/maxV] d-axis induced voltage	*/
		q15_t			Eq;						/* [V/maxV] q-axis induced voltage	*/

		q31_t			Ed_I;					/* d-axis induced voltage I */
		q31_t			Ed_PI;					/* d-axis induced voltage PI */

		state_t			state;					/* Motor error state */
		command_t		command;				/* Command instruction */
		vectorcmd_t		vector_cmd;				/* Vector control command */

		q15_t			spwm_threshold;			/* [Hz/maxHz] Shift-PWM threshold */
		uint16_t		chkpls;					/* [---] Check pulse width setting */
		uint8_t			idetect_error;			/* current detect error state */
		uint8_t idetect_error_pre;					/* current detect error state */

		q15_t			Ia_raw;					/* [A/maxA] a phase current Raw data */
		q15_t			Ib_raw;					/* [A/maxA] b phase current Raw data */
		q15_t			Ic_raw;					/* [A/maxA] c phase current Raw data */

		q15_t Ia_tst_pre1;							/* [A/maxA] a phase current Raw data */
		q15_t Ib_tst_pre1;							/* [A/maxA] b phase current Raw data */
		q15_t Ic_tst_pre1;							/* [A/maxA] c phase current Raw data */
		q15_t Ia_tst_pre2;							/* [A/maxA] a phase current Raw data */
		q15_t Ib_tst_pre2;							/* [A/maxA] b phase current Raw data */
		q15_t Ic_tst_pre2;							/* [A/maxA] c phase current Raw data */
		q15_t Ia_raw_pre1;							/* [A/maxA] a phase current */
		q15_t Ib_raw_pre1;							/* [A/maxA] b phase current */
		q15_t Ic_raw_pre1;							/* [A/maxA] c phase current */
		q15_t Ia_raw_pre2;							/* [A/maxA] a phase current */
		q15_t Ib_raw_pre2;							/* [A/maxA] b phase current */
		q15_t Ic_raw_pre2;							/* [A/maxA] c phase current */
		q15_t Ia_fpre;								/* [A/maxA] a phase current */
		q15_t Ib_fpre;								/* [A/maxA] b phase current */
		q15_t Ic_fpre;								/* [A/maxA] c phase current */
		q15_t Ia_tst;								/* [A/maxA] a phase current */
		q15_t Ib_tst;								/* [A/maxA] b phase current */
		q15_t Ic_tst;								/* [A/maxA] c phase current */
		q15_t Ia_rst;								/* [A/maxA] a phase current */
		q15_t Ib_rst;								/* [A/maxA] b phase current */
		q15_t Ic_rst;								/* [A/maxA] c phase current */

		q15_t			Ia;						/* [A/maxA] a phase current */
		q15_t			Ib;						/* [A/maxA] b phase current */
		q15_t			Ic;						/* [A/maxA] c phase current */

		uint16_t Ia1;								/* [A/maxA] a phase current */
		uint16_t Ib1;								/* [A/maxA] b phase current */
		uint16_t Ic1;								/* [A/maxA] c phase current */

		q15_t			Ia_t; 					/* [A/maxA] a phase current */
		q15_t			Ib_t; 					/* [A/maxA] b phase current */
		q15_t			Ic_t; 					/* [A/maxA] c phase current */

		q31_u			Iao_ave;				/* [A/maxA] a phase average of zero current */
		q31_u			Ibo_ave;				/* [A/maxA] b phase average of zero current */
		q31_u			Ico_ave;				/* [A/maxA] c phase average of zero current */

		uint16_t Iao;								/*no use */

		uint8_t			spdprd;					/* Speed pi period counter */

		/* Encoder setting */
#if defined(__USE_ENCODER_FAN)					|| defined(__USE_ENCODER_COMP)
		q15_t omega_enc;							/* [Hz/maxHz] Omega from encoder */
		q15_t omega_enc_raw;						/* [Hz/maxHz] Omega(Raw) from encoder */
		q31_u omega_enc_ave;						/* [Hz/maxHz] Omega(Ave) from encoder */
		uint32_t theta_enc; 						/* [deg/360] Theta from encoder */
		int16_t EnCnt;								/* Encoder counter value */
		int16_t EnCnt1; 							/* One previous encoder counter value */
		int16_t EnCnt_dev;							/* Deviation of the encoder counter */
#endif /* __USE_ENCODER */

#if !defined(__USE_VE_COMP) || !defined(__USE_VE_FAN)
		uint32_t* 		ADxREG0;				/* AD conversion result register address */
		uint32_t* 		ADxREG1;				/* AD conversion result register address */
		uint32_t* 		ADxREG2;				/* AD conversion result register address */
		uint32_t* 		ADxREG3;				/* AD conversion result register address */

		q15_t			Vdc_adc;				/* [V/maxV] DC voltage ADC value */
		uint16_t 		Ia_adc;					/* [A/maxA] U phase current ADC result */
		uint16_t 		Ib_adc;					/* [A/maxA] V phase current ADC result */
		uint16_t 		Ic_adc;					/* [A/maxA] W phase current ADC result */

		q15_t			Idc1_adc;				/* [A/maxA] 1st phase current ADC result (for 1shunt) */
		q15_t			Idc2_adc;				/* [A/maxA] 2nd phase current ADC result (for 1shunt) */
		q31_u			Idco_ave;				/* [A/maxA] average of zero current (for 1shunt) */

		q15_t			Ialpha;					/* [A/maxA] alpha-axis current */
		q15_t			Ibeta;					/* [A/maxA] beta-axis current */

		q15_t			Id_dev;					/* [A/maxA] d-axis current dev */
		q15_t			Iq_dev;					/* [A/maxA] q-axis current dev */

		q31_u			Vd_com;					/* [V/maxV] Command d-axis voltage */
		q31_u			Vq_com;					/* [V/maxV] Command q-axis voltage */

		q31_u			Vd_I;					/* [V/maxV] d-axis voltage value */
		q31_u			Vq_I;					/* [V/maxV] q-axis voltage value */

		q31_u			Valpha;					/* [V/maxV] alpha-axis voltage	*/
		q31_u			Vbeta;					/* [V/maxV] beta-axis voltage	*/

		q15_t 			DutyU;					/* U-phase PWM duty ratio */
		q15_t 			DutyV;					/* V-phase PWM duty ratio */
		q15_t 			DutyW;					/* W-phase PWM duty ratio */

		int32_t 		AdTrg0;					/* AD trigger timing 0 */
		int32_t 		AdTrg1;					/* AD trigger timing 1 */

		uint8_t			Sector0;				/* Sector0 */
		uint8_t			Sector1;				/* One previous sector */
#endif /* !defined(__USE_VE_COMP) || !defined(__USE_VE_FAN) */
	} drv;

	struct
	{
		q31_u			omega_user;				/* [Hz/maxHz] Target omega by user */

		q15_t			Id_st_user;				/* [A/maxA] Start d-axis current by user */
		q15_t			Iq_st_user;				/* [A/maxA] Start q-axis current by user */
		uint16_t		lambda_user;			/* [deg/360] Init position(Lambda) by user */

		q15_t Iq_adj_I1;							/*no use */
		command_t		com_user; 				/* Control command by User order */
		command_t		com_user_1;				/* before step Scom_user */

	} usr;

	struct
	{
		uint16_t		V_max; 					/* [V]V max */
		uint16_t I_max; 							/*[A]I max */
		uint16_t Hz_max;							/*[Hz]Hz max */
		q15_t			omega_min;				/* [Hz/maxHz] min omega */
		q15_t			omega_v2i;				/* [Hz/maxHz] Switching speed from the voltage type control to the current type control */
		q15_t			spwm_threshold;			/* [Hz/maxHz] Shift-PWM threshold */
		q31_t			vd_pos;					/* [V/maxV] d-axis Output Voltage */
		q31_t			spd_coef;				/* [---] speed coefficient */

		q31_u			sp_ud_lim_f; 			/* Limitation of the increase value (Forced Stage) */
		q31_u			sp_up_lim_s; 			/* Limitation of the increase value (Steady Stage) */
		q31_u			sp_dn_lim_s; 			/* Limitation of the decrease value (Steady Stage) */
		uint16_t		ctrlprd;				/* [---] Normalize data ctrlprd */


		struct
		{
			uint16_t	initpos;				/* Time of Positioning State */
			uint16_t	initpos2;				/* Time of Positioning Wait State */
			uint16_t	bootstp;				/* Time of Bootstrap State */
			uint16_t	go_up;					/* Time of Change-up state */
		} time;

		q31_t			iq_lim;					/* [A/maxA] Limitation of Iq */
		q31_t			id_lim;					/* [A/maxA] Limitation of Id */
		q15_t			err_ovc;				/* [A/maxA] Over current */

		struct
		{
			q31_t		kp;						/* [---] Normalize data Pos Kp */
			q31_t		ki;						/* [---] Normalize data Pos Ki */
			int32_t		ctrlprd;				/* [---] Normalize data ctrlprd */
		} pos;

		struct
		{
			q31_t		kp;						/* [---] Normalize data Spd Kp */
			q31_t		ki;						/* [---] Normalize data Spd Ki */
			uint8_t		pi_prd;					/* [cnt] Speed PI period count */
		} spd;

		struct
		{
			q31_t		dkp;					/* [---] Normalize data Current(Id) Kp */
			q31_t		dki;					/* [---] Normalize data Current(Id) Ki */
			q31_t		qkp;					/* [---] Normalize data Current(Iq) Kp */
			q31_t		qki;					/* [---] Normalize data Current(Iq) Ki */
		} crt;

		struct
		{
			q31_t		r;						/* [---] Normalize data motor R */
			q31_t		Lq;						/* [---] Normalize data motor Lq */
			q31_t		Ld;						/* [---] Normalize data motor Ld */
		} motor;


#if defined(__USE_ENCODER_FAN)					|| defined(__USE_ENCODER_COMP)
		struct 
		{
			uint32_t pls2theta; 					/* [---] Convert pulse to theta */
			q15_t pls2omega;						/* [---] Convert pulse to omega */
			int32_t plsnum; 						/* [---] Encoder pulse number */
			int32_t ctrlprd;						/* [---] Control period */
			uint32_t deg_adjust;					/* [---] Theta adjust */
		} enc;
#endif /* __USE_ENCODER */

		int32_t			delta_lambda;			/* [---] Change_up stage lamda */

		uint16_t		chkpls;					/* [---] Check pulse width setting */

		q15_t Filter_limit;

#if !defined(__USE_VE_COMP) || !defined(__USE_VE_FAN)
		int16_t			TrgCompen;				/* Trigger timing correction value */
		trgpos_e		TrgPosMd;				/* Trigger position mode of Current detection 0: 3shunt, 1: 1shunt(first half), 2: 1shunt(second half) */
		phcvmd_e		PhCvMd;					/* Phase conversion mode 0:SVM,  1:InvClarke */
#endif /* !defined(__USE_VE_COMP) || !defined(__USE_VE_FAN) */
	} para;

	uint32_t			stage_counter;			/* Stage counter */
	shunt_type_e		shunt_type;				/* Shunt type */
	boot_type_e			boot_type;				/* Boot type */
	motortype_e			motor_type; 			/* Motor Type: 0: Compressor; 1: Fan */

	uint8_t Iac_eff_filter_shift3;				/* Iac eff  filter shift3 */
	uint8_t Vac_eff_filter_shift4;				/* Vac eff filter shift1 */
	q15_t target_spd;							/* [rps]Motor target spd*/

	struct
	{
	    t_state_t state;
	} temp_t;

	struct
	{
		uint16_t apparent_power_compensate; 		/* [W]Power compensate*/
		uint16_t apparent_power;					/* [W]Power val */
		uint16_t apparent_power_tmp;				/* [W]Power val tmp */
		uint16_t power_cnt; 						/* power filter count */
		uint16_t power_cnt1;						/* power filter count */
		uint8_t power_para; 						/* power filter count */
		uint8_t fdecrease;							/* power limit decrease speed flag */
		uint16_t cnt_t1;							/* Power decrease speed count1*/
		uint8_t cnt_t0; 							/* Power decrease speed count0*/

		uint16_t Iac_eff_temp;						/* [A/MaxA]Iac eff tmp */
		uint16_t Iac_eff;							/* [A/MaxA]Iac eff */
		uint16_t Vac_eff_temp;						/* [V/MaxV]Vac eff tmp */
		uint16_t Vac_eff;							/* [V/MaxV]Vac eff */
		uint16_t Iac_temp;							/* [A/MaxA]Iac tmp */
		uint16_t Vac_temp;							/* [V/MaxV]Vac */
		uint32_t Iac_temp_sum;						/* [A/MaxA]Iac tmp sum */
		uint32_t Vac_temp_sum;						/* [V/MaxV]Vac tmp */
		uint16_t sum_cnt;							/* cal sum count */
		uint8_t start_power_flag;					/* power cal start flag */
		uint8_t power_filter_shift5;				/* power filter cal filter shift */
		uint16_t pfc_max_current;					/* [A*10]PFC current max */
		uint16_t Vac_max;							/* [V] Vac max val */
	} power;


	struct 
	{
		uint8_t constant_filter1;					/*filter constant */
		uint8_t constant_filter2;					/*filter constant */
		uint8_t constant_filter3;					/*filter constant */
		uint8_t constant_filter5;					/*filter constant */
	} constant;


} vector_t;


typedef enum 
{
	cVENTILATING_UART = 0, 
	cHEATING_UART = 1, 
	cCOOLING_UART = 2, 
	cDRYING_UART = 3, 
	cFORCECOOL_UART = 4, 
	cFORCEHEAT_UART = 5, 
	cAUTOCLEANING_UART = 8, 
	cNONE_MODE_UART = 7, 
} main_mode_e;


typedef enum 
{
	cVENTILATING = 0, 
	cHEATING_MODE = 1, 
	cCOOLING_MODE = 2, 
	cDRYING = 3, 
	cNONE_MODE = 4, 
} common_mode_e;


typedef enum 
{
	cNONE_FUNC = 0, 
	cAUTOCLEAN_FUNC = 1, 
} special_func_e;


typedef enum 
{
	cNONE = 0, 
	cDEFROST = 1, 
} main_sub_e;


/*****************************************************************
 * Definition of variable
 *****************************************************************/
EXTERN uint16_t		M_Main_Counter; 		/* Main loop counter */

#if defined(__CONTROL_MOTOR_COMP)
EXTERN vector_t		Motor_comp;				/* Motor data for ch1 */
#endif
#if defined(__CONTROL_MOTOR_FAN)
EXTERN vector_t		Motor_fan;				/* Motor data for ch0 */
#endif

/*===================================================================*
  global functions.
 *===================================================================*/

/****************************************************************************************
 * Module	: calc_ave
 *---------------------------------------------------------------------------------------
 * Function : cals Average
 * Input	: ave, val, shift
 * Output	: Average
 * Note 	:
 ****************************************************************************************/
static __INLINE int32_t calc_ave(int32_t ave, int32_t val, int32_t shift)
{
	return (ave + ((val - ave) >> shift));
}


#endif /* _IPDEFINE_H */

/*************************** END OF FILE **************************************/
