/**
 *******************************************************************************
 * @file    ipdrv_t32a.c
 * @brief   This file provides API functions for M4Kx T32A driver.
 * @version V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 *
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "ipdrv_t32a.h"

/** @addtogroup TX04_Periph_Driver
  * @{
  */

/** @defgroup T32A
  * @brief T32A driver modules
  * @{
  */

/** @defgroup T32A_Private_Defines
  * @{
  */

#define MOD_HALT_CLEAR        ((uint32_t)0xFFFFFFFD)
#define MOD_MODE32_CLEAR      ((uint32_t)0xFFFFFFFE)

#define RUN_RUNFLG_MASK       ((uint32_t)0x00000010)
#define RUN_RUNFLG_STOP       ((uint32_t)0x00000000)
#define RUN_RUN_SET           ((uint32_t)0x00000001)
#define RUN_RUN_CLEAR         ((uint32_t)0xFFFFFFFE)

#define CR_WBF_SET            ((uint32_t)0x00100000)
#define CR_WBF_CLEAR          ((uint32_t)0xFFEFFFFF)
#define CR_START_CLEAR        ((uint32_t)0xFFFFFFF8)
#define CR_STOP_CLEAR         ((uint32_t)0xFFFFFF8F)
#define CR_RELD_CLEAR         ((uint32_t)0xFFFFF8FF)
#define CR_CLK_PRSCL_CLEAR    ((uint32_t)0x88FFFFFF)
#define CR_UPDN_CLEAR         ((uint32_t)0xFFFCFFFF)
#define CR_UPDN_PULSECNT_SET ((uint32_t)0x00030000)

#define PLSCR_PMODE_MASK      ((uint32_t)0x00000001)
#define PLSCR_PMODE_CLEAR     ((uint32_t)0xFFFFFFFE)
#define PLSCR_PDIR_CLEAR      ((uint32_t)0xFFFFFFFD)
#define PLSCR_NF_CLEAR        ((uint32_t)0xFFFFFFCF)
#define PLSCR_PUP_CLEAR       ((uint32_t)0xFFFFF8FF)
#define PLSCR_PDN_CLEAR       ((uint32_t)0xFFFF8FFF)

/**
  * @brief the base address of T32A timer.
  */
static const uint32_t T32A_TIMER[TOTAL_TIMER_UNIT] = {
    0x00000040U, 0x00000080U, 0x000000C0U
};

/** @} */
/* End of group T32A_Private_Defines */

/** @defgroup T32A_Private_FunctionPrototypes
  * @{
  */

/** @} */
/* End of group T32A_Private_FunctionPrototypes */

/** @defgroup T32A_Private_Functions
  * @{
  */

/** @} */
/* End of group T32A_Private_Functions */

/** @defgroup T32A_Exported_Functions
  * @{
  */

/**
  * @brief  Set the operation mode.
  * @param  T32Ax: Select the T32A channel.
  *   This parameter can be one of the following values:
  *   TSB_T32A0, TSB_T32A1, TSB_T32A2, TSB_T32A3,
  *   TSB_T32A4, TSB_T32A5.
  * @param  Mode: Select the operation mode.
  *   This parameter can be MODE_16 or MODE_32.
  * @retval Success or not
  *   The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result T32A_SetMode(TSB_T32A_TypeDef * T32Ax, uint32_t Mode)
{
    FunctionalState state_TimerA = DISABLE;
    FunctionalState state_TimerB = DISABLE;
    FunctionalState state_TimerC = DISABLE;
    Result retval = ERROR;
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_T32A_ALL_PERIPH(T32Ax));
    assert_param(IS_T32A_MODE(Mode));

    state_TimerA = T32A_GetTimerState(T32Ax, TIMER_A);
    state_TimerB = T32A_GetTimerState(T32Ax, TIMER_B);
    state_TimerC = T32A_GetTimerState(T32Ax, TIMER_C);
    if ((state_TimerA == DISABLE) && (state_TimerB == DISABLE) && (state_TimerC == DISABLE)) {
        /* Can change MOD register when all timers stop */
        tmp = T32Ax->MOD;
        tmp &= MOD_MODE32_CLEAR;
        tmp |= Mode;
        T32Ax->MOD = tmp;
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Enable Timer unit of the specified T32A channel.
  * @param  T32Ax: Select the T32A channel.
  *   This parameter can be one of the following values:
  *   TSB_T32A0, TSB_T32A1, TSB_T32A2, TSB_T32A3,
  *   TSB_T32A4, TSB_T32A5.
  * @param  Unit: Select Timer unit.
  *   This parameter can be one of the following values:
  *   TIMER_A, TIMER_B, TIMER_C.
  * @retval Success or not
  *   The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result T32A_Enable(TSB_T32A_TypeDef * T32Ax, TimerUnit Unit)
{
    TSB_T32A_TIMER_TypeDef *TIMER = 0U;
    Result retval = ERROR;

    /* Check the parameters */
    assert_param(IS_T32A_ALL_PERIPH(T32Ax));
    assert_param(IS_T32A_ALL_UNIT(Unit));

    TIMER = (TSB_T32A_TIMER_TypeDef *) ((uint32_t)T32Ax + T32A_TIMER[Unit]);

    if (T32A_GetTimerState(T32Ax, Unit) == DISABLE) {
        /* Before enable Timer, check whether Timer is stop */
        TIMER->RUN |= RUN_RUN_SET;
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Start or stop counter by software of the specified T32A channel and unit.
  * @param  T32Ax: Select the T32A channel.
  *   This parameter can be one of the following values:
  *   TSB_T32A0, TSB_T32A1, TSB_T32A2, TSB_T32A3,
  *   TSB_T32A4, TSB_T32A5.
  * @param  Unit: Select Timer unit.
  *   This parameter can be one of the following values:
  *   TIMER_A, TIMER_B, TIMER_C.
  * @param  Cmd: The command for the counter.
  *   This parameter can be T32A_START_BY_SW or T32A_STOP_BY_SW.
  * @retval None
  */
void T32A_SetRunStateBySW(TSB_T32A_TypeDef * T32Ax, TimerUnit Unit, uint32_t Cmd)
{
    TSB_T32A_TIMER_TypeDef *TIMER = 0U;

    /* Check the parameters */
    assert_param(IS_T32A_ALL_PERIPH(T32Ax));
    assert_param(IS_T32A_ALL_UNIT(Unit));
    assert_param(IS_T32A_CMD(Cmd));

    TIMER = (TSB_T32A_TIMER_TypeDef *) ((uint32_t)T32Ax + T32A_TIMER[Unit]);

    /* Write command into RUN register */
    TIMER->RUN |= Cmd;
}

/**
  * @brief  Get the state of Timer.
  * @param  T32Ax: Select the T32A channel.
  *   This parameter can be one of the following values:
  *   TSB_T32A0, TSB_T32A1, TSB_T32A2, TSB_T32A3,
  *   TSB_T32A4, TSB_T32A5.
  * @param  Unit: Select Timer unit.
  *   This parameter can be one of the following values:
  *   TIMER_A, TIMER_B, TIMER_C.
  * @retval State of Timer
  *   The value returned can be one of the following values:
  *   ENABLE or DISABLE
  */
FunctionalState T32A_GetTimerState(TSB_T32A_TypeDef * T32Ax, TimerUnit Unit)
{
    TSB_T32A_TIMER_TypeDef *TIMER = 0U;
    FunctionalState state = DISABLE;
    uint32_t runflg = 0U;

    /* Check the parameters */
    assert_param(IS_T32A_ALL_PERIPH(T32Ax));
    assert_param(IS_T32A_ALL_UNIT(Unit));

    TIMER = (TSB_T32A_TIMER_TypeDef *) ((uint32_t)T32Ax + T32A_TIMER[Unit]);

    runflg = TIMER->RUN & RUN_RUNFLG_MASK;
    if (runflg == RUN_RUNFLG_MASK) {
        state = ENABLE;
    } else {
        /* Do nothing */
    }

    return state;
}

/**
  * @brief  Set the reload condition of the counter.
  * @param  T32Ax: Select the T32A channel.
  *   This parameter can be one of the following values:
  *   TSB_T32A0, TSB_T32A1, TSB_T32A2, TSB_T32A3,
  *   TSB_T32A4, TSB_T32A5.
  * @param  Unit: Select Timer unit.
  *   This parameter can be one of the following values:
  *   TIMER_A, TIMER_B, TIMER_C.
  * @param  ReloadCond: The condition for reload.
  *   This parameter can be one of the following values:
  *   When not use pulse count mode:
  *   T32A_COND_NONE, T32A_COND_INTERNAL_TRIGGER,
  *   T32A_COND_RISING_IN0, T32A_COND_FALLING_IN0,
  *   T32A_COND_RISING_OTHER, T32A_COND_FALLING_OTHER,
  *   T32A_COND_SYNC_CHANNEL, T32A_COND_MATCH_COMPARE1.
  *   When use pulse count mode (Timer C):
  *   T32A_COND_NONE, T32A_COND_INTERNAL_TRIGGER,
  *   T32A_COND_RISING_OTHER, T32A_COND_FALLING_OTHER,
  *   T32A_COND_SYNC_CHANNEL, T32A_COND_MATCH_COMPARE1.
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result T32A_SetReloadCond(TSB_T32A_TypeDef * T32Ax, TimerUnit Unit, uint32_t ReloadCond)
{
    TSB_T32A_TIMER_TypeDef *TIMER = 0U;
    uint32_t tmp = 0U;
    Result retval = ERROR;

    /* Check the parameters */
    assert_param(IS_T32A_ALL_PERIPH(T32Ax));
    assert_param(IS_T32A_ALL_UNIT(Unit));
    if ((Unit == TIMER_C) && ((T32Ax->CRC & CR_UPDN_PULSECNT_SET) == CR_UPDN_PULSECNT_SET)) {
        assert_param(IS_T32A_RELOAD_COND_PULSECNT(ReloadCond));
    } else {
        assert_param(IS_T32A_RELOAD_COND(ReloadCond));
    }

    if (T32A_GetTimerState(T32Ax, Unit) == DISABLE) {
        TIMER = (TSB_T32A_TIMER_TypeDef *) ((uint32_t)T32Ax + T32A_TIMER[Unit]);
        tmp = TIMER->CR;
        tmp &= CR_RELD_CLEAR;
        tmp |= (ReloadCond << 8U);
        TIMER->CR = tmp;
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}


/**
  * @brief  Initialize the specified T32A channel.
  * @param  T32Ax: Select the T32A channel.
  *   This parameter can be one of the following values:
  *   TSB_T32A0, TSB_T32A1, TSB_T32A2, TSB_T32A3,
  *   TSB_T32A4, TSB_T32A5.
  * @param  Unit: Select Timer unit.
  *   This parameter can be one of the following values:
  *   TIMER_A, TIMER_B, TIMER_C.
  * @param  InitStruct: The structure containing basic T32A configuration.
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result T32A_Init(TSB_T32A_TypeDef * T32Ax, TimerUnit Unit, T32A_InitTypeDef * InitStruct)
{
    TSB_T32A_TIMER_TypeDef *TIMER = 0U;
    uint32_t tmp = 0U;
    uint32_t pmode = 0U;
    Result retval = ERROR;

    /* Check the parameters */
    assert_param(IS_T32A_ALL_PERIPH(T32Ax));
    assert_param(IS_T32A_ALL_UNIT(Unit));
    assert_param(IS_POINTER_NOT_NULL(InitStruct));
    if (Unit == TIMER_C) {
        assert_param(IS_T32A_TIMERC_CNT_MOD(InitStruct->CounterMode));
    } else {
        assert_param(IS_T32A_TIMERAB_CNT_MOD(InitStruct->CounterMode));
        assert_param(IS_T32A_TIMERAB_CNT_VAL(InitStruct->Compare0Value));
        assert_param(IS_T32A_TIMERAB_CNT_VAL(InitStruct->Compare1Value));
        assert_param(IS_T32A_TIMERAB_CNT_VAL(InitStruct->ReloadValue));
    }
    assert_param(IS_T32A_CLK_SRC(InitStruct->ClkSrc));
    assert_param(IS_T32A_CLK_DIV(InitStruct->ClkDiv));

    if (T32A_GetTimerState(T32Ax, Unit) == DISABLE) {
        TIMER = (TSB_T32A_TIMER_TypeDef *) ((uint32_t)T32Ax + T32A_TIMER[Unit]);

        /* Configure source clock for T32Ax */
        tmp = TIMER->CR;
        tmp &= CR_CLK_PRSCL_CLEAR;
        tmp |= (InitStruct->ClkDiv << 28U);
        tmp |= (InitStruct->ClkSrc << 24U);

        /* Set counter operation running mode */
        tmp &= CR_UPDN_CLEAR;
        if ((InitStruct->CounterMode == T32A_CNT_MOD_PULSE_1) ||
            (InitStruct->CounterMode == T32A_CNT_MOD_PULSE_2)) {
            /* Select Pulse counter mode */
            pmode = T32Ax->PLSCR;
            pmode &= PLSCR_PMODE_CLEAR;
            pmode |= (InitStruct->CounterMode & PLSCR_PMODE_MASK);
            T32Ax->PLSCR = pmode;
            /* Set Pulse counter mode */
            tmp |= CR_UPDN_PULSECNT_SET;
        } else {
            tmp |= (InitStruct->CounterMode << 16U);
        }
        TIMER->CR = tmp;

        /* Write compare0 timing into RG0 */
        TIMER->RG0 = InitStruct->Compare0Value;

        /* Write compare1 timing into RG1 */
        TIMER->RG1 = InitStruct->Compare1Value;

        /* Write reload value into RELD */
        TIMER->RELD = InitStruct->ReloadValue;

        retval = SUCCESS;
    } else {
            /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Mask some T32A interrupt.
  * @param  T32Ax: Select the T32A channel.
  *   This parameter can be one of the following values:
  *   TSB_T32A0, TSB_T32A1, TSB_T32A2, TSB_T32A3,
  *   TSB_T32A4, TSB_T32A5.
  * @param  Unit: Select Timer unit.
  *   This parameter can be one of the following values:
  *   TIMER_A, TIMER_B, TIMER_C.
  * @param  INTMask: Select the mask of T32A interrupt.
  *   This parameter can be one of the following values:
  *   If Unit is TIMER_A or TIMER_B:
  *   T32A_INTMASK_NONE, T32A_INTMASK_MATCH_COMPARE0, T32A_INTMASK_MATCH_COMPARE1,
  *   T32A_INTMASK_OVERFLOW, T32A_INTMASK_UNDERFLOW or combination.
  *   If Unit is TIMER_C:
  *   T32A_INTMASK_NONE, T32A_INTMASK_MATCH_COMPARE0, T32A_INTMASK_MATCH_COMPARE1,
  *   T32A_INTMASK_OVERFLOW, T32A_INTMASK_UNDERFLOW, T32A_INTMASK_TRANSITION_ERROR or combination.
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result T32A_SetINTMask(TSB_T32A_TypeDef * T32Ax, TimerUnit Unit, uint32_t INTMask)
{
    TSB_T32A_TIMER_TypeDef *TIMER = 0U;
    Result retval = ERROR;

    /* Check the parameters */
    assert_param(IS_T32A_ALL_PERIPH(T32Ax));
    assert_param(IS_T32A_ALL_UNIT(Unit));
    if (Unit == TIMER_C) {
        assert_param(IS_T32A_TIMERC_INTMASK(INTMask));
    } else {
        assert_param(IS_T32A_TIMERAB_INTMASK(INTMask));
    }

    if (T32A_GetTimerState(T32Ax, Unit) == DISABLE) {
        TIMER = (TSB_T32A_TIMER_TypeDef *) ((uint32_t)T32Ax + T32A_TIMER[Unit]);

        /* Mask the specified interrupt */
        TIMER->IM = INTMask;
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/** @} */
/* End of group T32A_Exported_Functions */

/** @} */
/* End of group T32A */

/** @} */
/* End of group TX04_Periph_Driver */

