/**
 ****************************************************************************
 * @file    HPFC_drv.c
 * @brief   Interleaved Power Factor Correction control driver Source File
 * @version V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2023
 *****************************************************************************
 */

#include <stdlib.h>
#include "ipdefine.h"
#define DEFINE_APP
#include "HPFC_drv.h"
#undef  DEFINE_APP
#include "ipdrv_t32a.h"
#include "HPFC_Para.h"
#include "mcuip_drv.h"
#include "E_Sub.h"
#include "ipdrv_t32a.h"
#include "ipdrv_cg.h"
#include "ipdrv_enc.h"
#include "usercon.h"
#include <limits.h>
#include "Dac_drv.h"

/**
 *********************************************************************************************
 * @brief       Initial setting of PMD OWM.
 *
 * @param       pfc_t* const    _pfc
 *
 * @return      none
 *********************************************************************************************
 */
void pfc_InitSetting_PWM(pfc_t * const _pfc)
{
#if defined(__CONTROL_PFC)
    PMD_InitTypeDef pmdini;

    /******** Setting for PFC *********************************************/
    pmdini.shunt = cPFC_SHUNT;
    pmdini.poll = cPOLL_PFC;
    pmdini.polh = cPOLH_PFC;
    pmdini.pwmrate = PWMRate(cPWMPRD_PFC);
    pmdini.deadtime = DeadTime(0);
    pmdini.busmode = WBUF_BUS;

    IP_PMD_init(TSB_PMD2, &pmdini);
#endif  /* __CONTROL_PFC */
}

/**
 *********************************************************************************************
 * @brief       Initial setting of ADC.
 *
 * @param       pfc_t* const    _pfc
 *
 * @return      none
 *********************************************************************************************
 */
void pfc_InitSetting_ADC(pfc_t * const _pfc)
{
#if defined(__CONTROL_PFC)
    AD_InitTypeDef adini;

    /* Set AD Unit */
    _pfc->mcu_unit.ADx = HPFC_ADUNIT;

    adini.shunt = cPFC_SHUNT;
    adini.idcch = cADCH_CURRENT_IAC;
    adini.vdcch = cADCH_VOLTAGE_VAC;
    adini.pmd_ch = cPMD2;
    adini.pints = cPINTS_C;
    IP_ADC_init(HPFC_ADUNIT, &adini);

    NVIC_SetPriority(INTADCPDB_IRQn, HPFC_ADTRG_IRQ_LEVEL);
    NVIC_ClearPendingIRQ(INTADCPDB_IRQn);
    NVIC_EnableIRQ(INTADCPDB_IRQn);

#endif  /* __CONTROL_PFC */
}

/**
 *********************************************************************************************
 * @brief       Initialize HPFC Control
 *
 * @return      none
 *
 * @note        This function, please call After calling init_PMDen() function.
 *********************************************************************************************
 */
void init_HPFC_Control(void)
{
    pfc_t *const _pfc = &Hpfc;

    /**********Initialize PI Gain parameter*******************/
    /* Voltage control PI */
    _pfc->para.vol.kp = (q31_t)(FIX_12 / 2 * cVOL_KP_PFC * cV_MAX_PFC / cA_MAX_PFC);
    _pfc->para.vol.ki = (q31_t)(FIX_12 / 2 * cVOL_KI_PFC * cVOL_PI_CTRL_PRD_HPFC * cV_MAX_PFC / cA_MAX_PFC);

    /* Current control PI */
    _pfc->para.crt.kp = (q31_t)(FIX_12 / 2 * cCRT_KP_PFC * cA_MAX_PFC / 100);
    _pfc->para.crt.ki = (q31_t)(FIX_12 / 2 * cCRT_KI_PFC * cCRT_PI_CTRL_PRD_HPFC * cA_MAX_PFC / 100);

    /***************************************************/

    _pfc->para.pfc1 = (q15_t)(cPFC_TRG0);
    _pfc->para.pfc2 = (q15_t)(cPFC_TRG1);

    _pfc->max = 0;
    _pfc->min = 0xffff;

    _pfc->para.filter = cFILTER_PFC / cVAC_SAMPLE_FREQ;

    /* Limit of Iac amp */
    _pfc->para.Iac_amp_lim_up = cIAC_LIM_UP_PFC;
    _pfc->para.Iac_amp_lim_dw = cIAC_LIM_DW_PFC;
    _pfc->para.Iac_amp_lim = (q15_t)(cCURRENT_LIM_PFC * FIX_12 / 2 / cA_MAX_PFC);

    /* Limit of Duty */
    _pfc->para.duty_lim.min = (q15_t)(FIX_15 * cPWM_DUTY_MIN_PFC / 100);
    _pfc->para.duty_lim.max = (q15_t)(FIX_15 * cPWM_DUTY_MAX_PFC / 100);

    /* Over current */
    _pfc->judgevac.Vac_balance = (FIX_12 / 2);
    _pfc->judgevac.Vac_balance_temp = (FIX_12 / 2);
    _pfc->Vac.balance = (FIX_12 / 2);
    _pfc->para.Vdc_Limit = cVDC_LIM;
    _pfc->para.Vac_max = cVAC_MAX;
    _pfc->para.volt_up_lim.word = (uint32_t)(FIX_32 * cVdc_Cycle * cVOLT_UP_LIM_PFC / cV_MAX_PFC);
    _pfc->para.volt_dw_lim.word = (uint32_t)(FIX_32 * cVdc_Cycle * cVOLT_DW_LIM_PFC / cV_MAX_PFC);

	_pfc->drv.Iao = FIX_12 / 2;
	_pfc->para.Iac_lim_up =  cIAC_LIM_UP_PFC;
	_pfc->para.Vac_sample_time = cVAC_SAMPLE_FREQ;
	_pfc->para.pwm_prd_time = cPWMPRD_PFC_T;
	_pfc->para.trg_switch_point = (uint16_t)cTRG_SWITCH_CONSTANT;
	_pfc->para.stup_cnt = cPFC_STUP_CNT;
	_pfc->para.Vdc_target_cnt = cPFC_VDCTARGET_CNT;
	_pfc->para.Vadd = cPFC_VADD;
	_pfc->para.Vdc_tgt_cycle = cPFC_VDC_TGT_CTRCYCLE;
	_pfc->para.Vdc_dw_limt = cPFC_VDCDWLIMIT;

    /**** Initialize PWM Output BY PMD2 ****/
    pfc_InitSetting_PWM(_pfc);

    /**** HPFC ADC Setting  ****/
    pfc_InitSetting_ADC(_pfc);
}
/**
 *********************************************************************************************
 * @brief       HPFC Control at interrupt.
 *
 * @param       pfc_t* const    _pfc
 *
 * @return      none
 *********************************************************************************************
 */
void HPFC_Control_Int(void)
{

    pfc_t *const _pfc = &Hpfc;

    /**** Calculation of Phase Current ****/
    HPFC_INT_ERROR_CHECK();
    if (_pfc->drv.state.all == 0)
    {
        if (_pfc->control_flag >= 1)
        {

            HPFC_ControlCurrent(_pfc, cPhaseA);
            HPFC_INT_pwmA();
            /**** Calculation of PWM register set value ****/
            __disable_irq();
            TSB_PMD2->CMPW = _pfc->drv.Duty_a_reg0;
            TSB_PMD2->TRGCMP0 = _pfc->drv.Duty_a_trg;
            __enable_irq();
        }
    }
    else
    {
        HPFC_PWMOutputOff(_pfc);
    }

}

/**
 *********************************************************************************************
 * @brief       Set PFC mode
 *
 * @param       _pfc    _mode   Set mode
 *
 * @return      bool    TRUE:Mode update / FALSE:Mode no update
 *********************************************************************************************
 */
bool HPFC_SetPFC_Mode(pfc_mode_e _mode)
{
    pfc_t *const _pfc = &Hpfc;
    bool ret = false;

    if (_mode == cPFC_ON)
    {
        /* If Frequency of AC detected and No error, then PFC ON. */
        if ((_pfc->drv.ac.power_mode == cPowerON) && (_pfc->drv.state.all == 0))
        {
            if (!_pfc->pwm_mode)
            {
                _pfc->usr.pfc_mode = cPFC_ON;
                _pfc->pwm_mode = cPWM_ON_REQ;
                ret = true;
            }
        }
        else
        {
            _pfc->pwm_mode = cPWM_OFF;
            _pfc->Error.Err_mode_counter = 0;
            _pfc->Error.Err_disp_counter = 0;
            HPFC_PWMOutputOff(_pfc);
        }
    }
    else
    {
        _pfc->usr.pfc_mode = cPFC_OFF;
        ret = false;
    }
    return (ret);
}

/**
 *********************************************************************************************
 * @brief       HPFC_INT_ERROR_CHECK  Iac error check, every pfc control cycle
                check the error status
 *
 * @param       void
 *
 * @return      void
 *********************************************************************************************
 */
void HPFC_INT_ERROR_CHECK(void)
{
    pfc_t *const _pfc = &Hpfc;
    vector_t *_motor = &Motor_comp;

    if (_pfc->drv.Iac > cVAL_ERR_I_OVER)
    {
        if (_pfc->Error.Err_mode_counter < cM_ERR_MODE_NUM)
        {
            _pfc->drv.state.flg.emg_S = SET;
            _motor->drv.state.flg.pfc_relate = SET;
		}
    }
    else
    {
        /*no content*/
    }
}

/**
 *********************************************************************************************
 * @brief       Interrupt of ADC conversion triggered by timer is finished.
 *
 * @param       pfc_t* const    _pfc
 *
 * @return      none
 *********************************************************************************************
 */
void HPFC_INT_ADC_Fin(void)
{
    pfc_t *const _pfc = &Hpfc;

    /* Get ADC value */
    HPFC_GetPhaseCurrentADC(_pfc);

    /* Interleaved PFC control */
    HPFC_Control_Int();
}

/**
 *********************************************************************************************
 * @brief       PWM Output On.
 *
 * @param       pfc_t* const    _pfc
 *
 * @return      none
 *********************************************************************************************
 */
void HPFC_PWMOutputOn(pfc_t * const _pfc)
{
    if (_pfc->drv.state.all == 0)
    {
        PMD_SetOutputMode(TSB_PMD2, cMDOUT_PFC_ON);     /* cMDOUT_OFF */
        PMD_RegDataSet(TSB_PMD2, 0, 0, 0, 0X3FFF, 0);   /* W duty=0, trg=0 */
    }
}

/**
 *********************************************************************************************
 * @brief       PWM Output On.
 *
 * @param       pfc_t* const    _pfc
 *
 * @return      none
 *********************************************************************************************
 */
void HPFC_PWMOutputOn_pre(pfc_t * const _pfc)
{
    PMD_RegDataSet(TSB_PMD2, 0, 0, 0, 0X3FFF, 0);   /* W duty=0, trg=0 */
}

/**
 *********************************************************************************************
 * @brief       PWM Output Off.
 *
 * @param       pfc_t* const    _pfc
 *
 * @return      none
 *********************************************************************************************
 */
void HPFC_PWMOutputOff(pfc_t * const _pfc)
{
    PMD_SetOutputMode(TSB_PMD2, cMDOUT_OFF);
    PMD_RegDataSet(TSB_PMD2, 0, 0, 0, 0X3FFF, 0);   /* W duty=0, trg=0 */
}

/**
 *********************************************************************************************
 * @brief       Get Phase Current ADC value.
 *
 * @param       pfc_t* const    _pfc
 *
 * @return      none
 *********************************************************************************************
 */
void HPFC_GetPhaseCurrentADC(pfc_t * const _pfc)
{
    ADC_Result i_la;
    ADC_Result i_la1;
    ADC_Result i_la2;

    /* Get IAC AD value and IAC filter */
    i_la.All = TSB_ADC->REG0;
    i_la1.All = TSB_ADC->REG1;
    i_la2.All = TSB_ADC->REG2;

    _pfc->drv.Iac_adc0 = i_la.Bit.ADResult;
    _pfc->drv.Iac_adc1 = i_la1.Bit.ADResult;
    _pfc->drv.Iac_adc2 = i_la2.Bit.ADResult;

    _pfc->drv.Iac_adc = (uint16_t)E_Med3(i_la.Bit.ADResult, i_la1.Bit.ADResult, i_la2.Bit.ADResult);

    if (_pfc->drv.Iac_adc > _pfc->drv.Iao)
    {
        _pfc->drv.Iac_raw = _pfc->drv.Iac_adc - _pfc->drv.Iao;
    }
    else
    {
        _pfc->drv.Iac_raw = 0;
    }

    _pfc->drv.Iac += (_pfc->drv.Iac_raw - _pfc->drv.Iac) >> 1;

}

/**
 *********************************************************************************************
 * @brief       Calculation PWM Duty register value.
 *
 * @param       pfc_t* const    _pfc
 *
 * @return      none
 *********************************************************************************************
 */
void HPFC_CalPWM(pfc_t * const _pfc)
{

#if defined(__U_PFCOUT)
    PMD_RegDataSet(PFC_PWM_CH, _pfc->drv.Duty_a_reg0, 0, 0, _pfc->drv.Duty_a_trg, 0);
#elif defined(__V_PFCOUT)
    PMD_RegDataSet(PFC_PWM_CH, 0, _pfc->drv.Duty_a_reg0, 0, _pfc->drv.Duty_a_trg, 0);
#elif defined(__W_PFCOUT)
    PMD_RegDataSet(PFC_PWM_CH, 0, 0, _pfc->drv.Duty_a_reg0, _pfc->drv.Duty_a_trg, 0);
#endif
}

/**
 *********************************************************************************************
 * @brief       Calculation Zero Current.
 *
 * @param       pfc_t* const    _pfc
 *
 * @return      none
 *********************************************************************************************
 */
void HPFC_CalZeroCurrent(pfc_t * const _pfc)
{
    if (_pfc->drv.Iac_adc > _pfc->drv.Iao)
    {
        _pfc->drv.Iao += (_pfc->drv.Iac_adc - _pfc->drv.Iao) >> 2;
    }
    else
    {
        _pfc->drv.Iao -= (_pfc->drv.Iao - _pfc->drv.Iac_adc) >> 2;
    }
}

/**
 *********************************************************************************************
 * @brief       Set MotorControl data for user.
 *
 * @return      none
 *********************************************************************************************
 */
uint16_t  g_PfcOnTime, g_PfcOffTime;
void HPFC_UserControl(void)
{
    vector_t *_motor = &Motor_comp;
    pfc_t *_pfc = &Hpfc;

    if ((_pfc->pfc_mode_req == cPFC_ON) && (_motor->power.Iac_eff > cPFC_ON_CUR * 1000) && ((_motor->stage.main == cSteady_A)||(_motor->stage.main == cForce))&& _pfc->drv.ac.power_mode != cPowerON)
    {
        if (g_PfcOnTime++ > 200)
        {
            g_PfcOnTime = 0;
            _pfc->drv.ac.power_mode = cPowerON; /* Start PFC Control */
        }
        g_PfcOffTime = 0;
    }
    else
    {
        if ((_pfc->pfc_mode_req == cPFC_OFF) && (_motor->power.Iac_eff < (1.0f) * 1000) && (_pfc->drv.ac.power_mode == cPowerON))
        {
            if (g_PfcOffTime++ > 2000)
            {
                g_PfcOffTime = 0;
                _pfc->drv.ac.power_mode = cPowerOFF;
            }
        }
        g_PfcOnTime = 0;
    }

    /* PFC ON/OFF */
    HPFC_SetPFC_Mode(cPFC_ON);
}

/**
 *********************************************************************************************
 * @brief       HPFC Control at Main.
 *
 * @return      none
 *********************************************************************************************
 */

void HPFC_Control_Main(void)
{
    pfc_t *const _pfc = &Hpfc;
    vector_t *_motor = &Motor_comp;

    if (_pfc->drv.state.all == 0x00)
    {
        switch (_pfc->pwm_mode)
        {
            case cPWM_OFF:
                if ((*((__I uint32_t *) BITBAND_PERI(&TSB_PMD2->EMGSTA, B_EMGI))) == 1)
                {
                    (*((__O uint32_t *) BITBAND_PERI(&TSB_PMD2->EMGCR, B_EMGRS))) = 1;
                }

                _pfc->control_flag = 0;
                _pfc->pfc_signal.update_pfc = 1;
                _pfc->pfc_signal.status = 3;
                _pfc->drv.val_tgt_Vdc_temp =
                    (uint16_t)((((_pfc->Vac.high_fix_filter_mid * 2 * _pfc->para.Vac_max / FIX_12) + 0) * FIX_12) / (_motor->para.V_max));

                _pfc->drv.val_tgt_Vdc = _pfc->drv.val_tgt_Vdc + ((_pfc->drv.val_tgt_Vdc_temp - _pfc->drv.val_tgt_Vdc) >> 3);

                if (_pfc->drv.val_tgt_Vdc > ((_pfc->para.Vdc_Limit * FIX_12) / (_motor->para.V_max)))
                {
                    _pfc->drv.val_tgt_Vdc = (uint16_t)((_pfc->para.Vdc_Limit * FIX_12) / (_motor->para.V_max));
                }

                _pfc->judgevac.Vdq_sum = 0;
                _pfc->judgevac.cnt_vdq = 0;
                _pfc->drv.Vdc_ref = 0;
                _pfc->drv.target_Vdc = 0;
                _pfc->drv.target_Vdc_buf = 0;
                _pfc->drv.target_Vdc_now = 0;
                _pfc->drv.target_Vdc_now_tmp = 0;
                _pfc->drv.start_point_Vdc = 0;
                _pfc->drv.delta_point_Vdc = 0;
                _pfc->drv.Iac_amp_ref = 0;
                _pfc->drv.Iac_amp_ref_I = 0;
                _pfc->drv.Iac_amp_ref_P = 0;
                break;

            case cPWM_ON_REQ:
                _pfc->control_flag = 0;
                _pfc->drv.tempcr.temp_i_ki = 0;
                _pfc->drv.tempcr.temp_i_kp = 0;

                _pfc->pfc_signal.stup_cnt = 100;
                _pfc->drv.target_Vdc = _pfc->drv.Vdc_lpf;
                _pfc->drv.start_point_Vdc = _pfc->drv.Vdc_lpf;
                _pfc->drv.delta_point_Vdc = _pfc->drv.val_tgt_Vdc - _pfc->drv.Vdc_lpf;

                _pfc->pwm_mode = cPFC_INIT2;

                _pfc->pfc_signal.update_pfc = 1;
                _pfc->pfc_signal.status = 2;
                break;

            case cPFC_INIT2:
                _pfc->control_flag = 1;
                _pfc->drv.target_Vdc =
                    _pfc->drv.start_point_Vdc +
                    (uint16_t)((_pfc->drv.delta_point_Vdc * _pfc->pfc_signal.stup_cnt) / _pfc->para.stup_cnt);
                _pfc->pfc_signal.stup_cnt++;
                if (_pfc->pfc_signal.stup_cnt >= _pfc->para.stup_cnt)
                {
                    _pfc->pfc_signal.stup_cnt = 0;
                    _pfc->drv.target_Vdc = _pfc->drv.val_tgt_Vdc;
                    _pfc->drv.target_Vdc_now = _pfc->drv.target_Vdc;
                    _pfc->drv.target_Vdc_now_tmp = _pfc->drv.target_Vdc_now;
                    _pfc->pwm_mode = cPWM_ON;
                    _pfc->pfc_signal.stop = 0;
                }
                break;

            case cPWM_ON:
                _pfc->control_flag = 2;

                HPFC_VdcTarget(_pfc, _motor);
                break;

            default:
                _pfc->pwm_mode = cPWM_OFF;
                break;
        }
    }
    else
    {
        _pfc->pfc_signal.update_pfc = 1;
        _pfc->pfc_signal.status = 3;
        _pfc->pwm_mode = cPWM_OFF;
        HPFC_PWMOutputOff(_pfc);
    }
}


/**
 *********************************************************************************************
 * @brief       Calculation Vdc Filter value.
 *
 * @param       pfc_t* const    _pfc
 *
 * @return      none
 *********************************************************************************************
 */
void HPFC_CalVdcFilter_16k(pfc_t * const _pfc)
{
    vector_t *_motor_fan = &Motor_fan;
    vector_t *_motor_comp = &Motor_comp;

    _pfc->drv.Vdc_lpf0 = (*_motor_fan->drv.ADxREG3 & 0xFFF0) >> 4;
    if (_pfc->drv.Vdc_lpf0 > (FIX_12 * cOVV_HPFC / cV_MAX_PFC))
    {
        _pfc->over_dc_cnt_test++;
        if (_pfc->over_dc_cnt_test > 2)
        {
            _pfc->over_dc_cnt_test = 0;
            _pfc->drv.state.flg.emg_DC = SET;
            _motor_comp->drv.state.flg.pfc_relate = 1;
        }
    }
    else
    {
        _pfc->over_dc_cnt_test = 0;
    }

    _pfc->drv.Vdc_lpf += (_pfc->drv.Vdc_lpf0 - _pfc->drv.Vdc_lpf) >> 5;
}
/**
 *********************************************************************************************
 * @brief       PFC Error Check
                PFC Error Check Cycle < PFC Control Cycle
 *
 * @param       void
 *
 * @return      void
 *********************************************************************************************
 */
void HPFC_INT_ERROR_CHECK_16k(void)
{
    pfc_t *const _pfc = &Hpfc;
    vector_t *_motor = &Motor_comp;

    if (_motor->drv.state.all != 0)
    {
        _pfc->drv.state.flg.emg_comp = SET;
    }

	/* Vdc Over voltage detect */
    if (_pfc->dcave.Vdc_ave > cVAL_ERR_VDC_OVER)
    {
        if (_pfc->Error.Err_mode_counter < cM_ERR_MODE_NUM)
        {

            _pfc->drv.state.flg.over_Vdc = 1;
            _motor->drv.state.flg.pfc_relate = SET;
        }
    }

    /* Vdc Under voltage detect */
    if (_pfc->dcave.Vdc_ave < cVAL_ERR_VDC_UNDER)
    {
        if (_pfc->Error.Err_mode_counter < cM_ERR_MODE_NUM)
        {
            _pfc->drv.state.flg.under_Vdc = 1;
            _motor->drv.state.flg.pfc_relate = SET;
        }
    }

    /* Vac Over voltage detect */
    if (_pfc->Vac.high_fix_value > cVAL_ERR_VAC_OVER)
    {
        if (_pfc->Error.Err_mode_counter < cM_ERR_MODE_NUM)
        {
            _pfc->drv.state.flg.over_Vac = 1;
            _motor->drv.state.flg.pfc_relate = SET;
        }
    }

    /* Vac Under voltage detect */
    if (_pfc->Vac.high_fix_value < cVAL_ERR_VAC_UNDER)
    {
        if (_pfc->Error.Err_mode_counter < cM_ERR_MODE_NUM)
        {
            _pfc->drv.state.flg.under_Vac = 1;
            _motor->drv.state.flg.pfc_relate = SET;
        }
    }
    
    /* AC frequecy protect */
    if ((_pfc->Vac.Hz_data < cFREQ_ERR_OVER) || (_pfc->Vac.Hz_data > cFREQ_ERR_UNDER))
    {
        if (_pfc->Error.Err_mode_counter < cM_ERR_MODE_NUM)
        {
            _pfc->drv.state.flg.emg_Hz = 1;
        }
    }

    /* AC zero cross protect */
    if (_pfc->Vac.zero_flag == 0)
    {
        _pfc->Vac.zero_protect_cnt++;
        if (_pfc->Vac.zero_protect_cnt > cZERO_ERROR_CYCLE)
        {
            _pfc->drv.state.flg.emg_ZC = 1;
            _pfc->Vac.zero_protect_cnt = 0;
        }
    }
    else
    {
        _pfc->Vac.zero_protect_cnt = 0;
    }
}

/**
 *********************************************************************************************
 * @brief       HPFC_VacVdc_Handle
 *
 * @param       none
 *
 * @return      none
 *********************************************************************************************
 */
void HPFC_VacVdc_Handle(pfc_t* const _pfc)
{
    /* PFC control */
    ADC_Result v_ac;
    v_ac.All = TSB_ADC->REG12;
    _pfc->drv.Vac_adc = v_ac.Bit.ADResult;
    _pfc->drv.Vac +=  (_pfc->drv.Vac_adc - _pfc->drv.Vac) >> 1;
    HPFC_CalVdcFilter_16k(_pfc);
    HPFC_CalVacFilter_16k(_pfc);
    HPFC_INT_ERROR_CHECK_16k();

    if (_pfc->drv.state.all == 0)
    {
        if (_pfc->control_flag >= 1)
        {
            HPFC_ControlVoltage_16k(_pfc);
        }
    }
    _pfc->drv.Iac_ref = _pfc->drv.Iac_ref_buff;
}

/**
 *********************************************************************************************
 * @brief       Control Current.
 *
 * @param       pfc_t* const    _pfc
 * @param       phase           _mode
 *
 * @return      none
 *********************************************************************************************
 */
void HPFC_ControlCurrent(pfc_t * const _pfc, phase_mode_e _mode)
{
    q15_t error;

    error = _pfc->drv.Iac_ref - _pfc->drv.Iac;
    _pfc->drv.tempcr.temp_i_kp = ((error * _pfc->para.crt.kp) >> 10);
    _pfc->drv.tempcr.temp_i_ki += ((error * _pfc->para.crt.ki) >> 10);

    if (_pfc->drv.tempcr.temp_i_ki > _pfc->para.Iac_lim_up)
    {
        _pfc->drv.tempcr.temp_i_ki = _pfc->para.Iac_lim_up;
    }
    else
    {
        if (_pfc->drv.tempcr.temp_i_ki < 0)
        {
            _pfc->drv.tempcr.temp_i_ki = 0;
        }
    }

    _pfc->drv.Duty_a_reg = _pfc->drv.tempcr.temp_i_kp + _pfc->drv.tempcr.temp_i_ki;
    _pfc->drv.Duty_a_reg0 = (q31_t)(FIX_15 * _pfc->drv.Duty_a_reg / _pfc->para.pwm_prd_time);

    if (_pfc->drv.Duty_a_reg0 > _pfc->para.duty_lim.max)
    {
        _pfc->drv.Duty_a_reg0 = _pfc->para.duty_lim.max;
    }
    else
    {
        if (_pfc->drv.Duty_a_reg0 < _pfc->para.duty_lim.min)
        {
            _pfc->drv.Duty_a_reg0 = 0;
        }
    }
}

/**
 *********************************************************************************************
 * @brief       HPFC_VdcTarget
 *
 * @param       pfc_t* const    _pfc
 * @param       none
 *
 * @return      none
 *********************************************************************************************
 */
void HPFC_VdcTarget(pfc_t * const _pfc, vector_t* const _motor)
{
	if ((_pfc->judgevac.Vdq_ave * (_motor->para.V_max) / FIX_15) > _pfc->para.Vdc_dw_limt)
    {
        g_CntTgtVdc3 = 0;

        g_CntTgtVdc1++;
        if (g_CntTgtVdc1 > 100)
        {
            g_CntTgtVdc1 = 101;
            _pfc->drv.target_Vdc_now_tmp =
                (uint16_t)((((_pfc->Vac.high_fix_filter_mid * 2 * _pfc->para.Vac_max / FIX_12) +
                              (_pfc->judgevac.Vdq_ave * (_motor->para.V_max) / FIX_15)) * FIX_12) / (_motor->para.V_max));
        }
    }
    else
    {
        g_CntTgtVdc1 = 0;

        g_CntTgtVdc3++;
        if (g_CntTgtVdc3 > 100)
        {
            g_CntTgtVdc3 = 101;
            _pfc->drv.target_Vdc_now_tmp =
                (uint16_t)((((_pfc->Vac.high_fix_filter_mid * 2 * _pfc->para.Vac_max / FIX_12) + 0) * FIX_12) / (_motor->para.V_max));
        }
    }
    _pfc->drv.target_Vdc_now = _pfc->drv.target_Vdc_now_tmp;

    /* vdc now and taget gap > 5V, add start */
    if (abs(_pfc->drv.target_Vdc_now - _pfc->drv.target_Vdc) >= ((_pfc->para.Vadd) * FIX_12 / (_motor->para.V_max)))
    {
        g_SubTgtVdc = (_pfc->drv.target_Vdc_now - _pfc->drv.target_Vdc) / (_pfc->para.Vdc_target_cnt);
    }

    g_CntTgtVdc2++;
    if (g_CntTgtVdc2 >= _pfc->para.Vdc_tgt_cycle)
    {
        g_CntTgtVdc2 = 0;
        _pfc->drv.target_Vdc = (uint16_t)(_pfc->drv.target_Vdc + g_SubTgtVdc);
    }

    if (g_SubTgtVdc >= 0)
    {
        _pfc->drv.target_Vdc = ((_pfc->drv.target_Vdc > _pfc->drv.target_Vdc_now)? _pfc->drv.target_Vdc_now : _pfc->drv.target_Vdc);
    }
    else
    {
        _pfc->drv.target_Vdc = ((_pfc->drv.target_Vdc > _pfc->drv.target_Vdc_now)? _pfc->drv.target_Vdc : _pfc->drv.target_Vdc_now);
    }

    if (_pfc->drv.target_Vdc > ((_pfc->para.Vdc_Limit * FIX_12) / (_motor->para.V_max)))
    {
        _pfc->drv.target_Vdc = (uint16_t)(((_pfc->para.Vdc_Limit * FIX_12) / (_motor->para.V_max)));
    }
}

/**
 *********************************************************************************************
 * @brief       Control Voltage.
 *
 * @param       pfc_t* const    _pfc
 *
 * @return      none
 *********************************************************************************************
 */
void HPFC_ControlVoltage(pfc_t * const _pfc)
{
    /* Determination of Iac_amp_ref */
    if (_pfc->para.vol_loop == 1)
    {
        q31_t error;
        _pfc->para.vol_loop = 0;
        _pfc->drv.Vdc_ref = _pfc->drv.target_Vdc;

        /* Error calculation of Vdc */
        error = (q15_t)(_pfc->drv.Vdc_ref - _pfc->dcave.Vdc_ave);
        /* Integral term calculation */
        _pfc->drv.Iac_amp_ref_P = ((error * _pfc->para.vol.kp) >> 10);
        _pfc->drv.Iac_amp_ref_I += ((error * _pfc->para.vol.ki) >> 10);

        /* Limit value excess */
        if (_pfc->drv.Iac_amp_ref_I < _pfc->para.Iac_amp_lim_dw)
        {
            _pfc->drv.Iac_amp_ref_I = _pfc->para.Iac_amp_lim_dw;
        }
        else if (_pfc->drv.Iac_amp_ref_I > _pfc->para.Iac_amp_lim_up)
        {
            _pfc->drv.Iac_amp_ref_I = (q15_t)(_pfc->para.Iac_amp_lim_up);
        }

        _pfc->drv.Iac_amp_ref = (_pfc->drv.Iac_amp_ref_P + _pfc->drv.Iac_amp_ref_I);

        /* Limit value excess */
        if (_pfc->drv.Iac_amp_ref < 0)
        {
            _pfc->drv.Iac_amp_ref = 0;
        }
    }
}

/**
 *********************************************************************************************
 * @brief       Interrupt of PWM A phase.
 *
 * @param       pfc_t* const    _pfc
 *
 * @return      none
 *********************************************************************************************
 */
void HPFC_INT_pwmA(void)
{
    pfc_t *const _pfc = &Hpfc;
    uint16_t temp_timer;

    if (_pfc->drv.Duty_a_reg0 >= _pfc->para.duty_lim.max)
    {
        temp_timer = _pfc->para.duty_lim.max;
    }
    else if (_pfc->drv.Duty_a_reg0 <= _pfc->para.duty_lim.min)
    {
        temp_timer = _pfc->para.duty_lim.min;
    }
    else
    {
        temp_timer = _pfc->drv.Duty_a_reg0;
    }

    if (temp_timer >= _pfc->para.trg_switch_point)
    {
        /* Sample at PWM high level, the fixed sampleing time is used to ensure that the interrupt function can be completed within a period  */
        _pfc->drv.Duty_a_trg = FIX_15 - _pfc->para.pfc1;
    }
    else
    {
        /* Sample at PWM low level */
        _pfc->drv.Duty_a_trg = ((FIX_15 - temp_timer) >> 1) - _pfc->para.pfc2;
    }
}

/**
 *********************************************************************************************
 * @brief       Calculation Vac Filter value (16k)
 *
 * @param       pfc_t* const    _pfc
 *
 * @return      none
 *********************************************************************************************
 */
void HPFC_CalVacFilter_16k(pfc_t * const _pfc)
{
    vector_t *_motor = &Motor_comp;

    _pfc->Vac.cnt++;

    if (_pfc->drv.Vac < _pfc->Vac.balance)   /* Bottom judgement */
    {
        /* Zero cross judgement */
        if (_pfc->Vac.high_flag == 1)
        {
            _pfc->Vac.zero_cnt++;

            if (_pfc->Vac.zero_cnt >= 4)
            {
                _pfc->Vac.high_flag = 0;
                _pfc->Vac.zero_flag = 1;    /* Zero cross */
                _pfc->para.vol_loop = 1;
                _pfc->Vac.zero_cnt = 0;
            }
            else
            {
                _pfc->Vac.zero_flag = 0;    /* Make sure that te next cycle flag is cleared */
            }
        }
        else
        {
            _pfc->Vac.zero_flag = 0;        /* Clear zero flag */
            _pfc->Vac.zero_cnt = 0;         /* Make sure the count is incremented continuously */
        }

        if (_pfc->drv.Vac <= _pfc->Vac.low_value)
        {
            _pfc->Vac.low_value = _pfc->drv.Vac;
            _pfc->Vac.bottom_cnt = _pfc->Vac.cnt;
            _pfc->Vac.bottom_judge_filter = 0;
        }
        else
        {
            if (_pfc->Vac.bottom_judge_flag == 0)
            {
                if (++_pfc->Vac.bottom_judge_filter > _pfc->para.filter)
                {
                    _pfc->Vac.low_fix_value = _pfc->Vac.low_value;
                    _pfc->Vac.bottom_judge_flag = 1;
                    _pfc->Vac.peak_judge_flag = 0;
                    _pfc->Vac.bottom_judge_filter = 0;
                }
            }
        }
    }
    else if (_pfc->drv.Vac >= _pfc->Vac.balance)
    {
        /* Zero cross judgement */
        if (_pfc->Vac.high_flag == 0)
        {
            _pfc->Vac.zero_cnt++;
            if (_pfc->Vac.zero_cnt >= 4)
            {
                _pfc->Vac.high_flag = 1;
                _pfc->Vac.zero_flag = 1;    /* Zero cross */
                _pfc->para.vol_loop = 1;
                _pfc->Vac.zero_cnt = 0;
            }
            else
            {
                _pfc->Vac.zero_flag = 0;    /* Make sure that te next cycle flag is cleared */
            }
        }
        else
        {
            _pfc->Vac.zero_flag = 0;        /* Clear zero flag */
            _pfc->Vac.zero_cnt = 0;         /* Make sure the count is incremented continuously */
        }

        if (_pfc->drv.Vac >= _pfc->Vac.high_value)
        {
            _pfc->Vac.peak_cnt = _pfc->Vac.cnt;
            if (_pfc->Vac.bottom_judge_flag == 1)
            {
                _pfc->Vac.high_value = _pfc->drv.Vac;
                _pfc->Vac.peak_judge_filter = 0;
            }
        }
        else
        {
            if (_pfc->Vac.peak_judge_flag == 0)
            {
                if (++_pfc->Vac.peak_judge_filter > _pfc->para.filter)
                {
                    _motor->power.start_power_flag = 1;
                    _pfc->Vac.peak_judge_filter = 0;
                    _pfc->Vac.peak_judge_flag = 1;

                    if (_pfc->Vac.bottom_judge_flag == 1)
                    {
                        _pfc->Vac.bottom_judge_flag = 0;
                        _pfc->Vac.Hz_data_tmp = _pfc->Vac.peak_cnt - _pfc->Vac.bottom_cnt;
                        _pfc->Vac.Hz_data += (_pfc->Vac.Hz_data_tmp  - _pfc->Vac.Hz_data) >> 2; /* filter */
                        _pfc->Vac.cnt = 0;
                        _pfc->Vac.balance = (_pfc->Vac.high_value + _pfc->Vac.low_value) / 2;

                        _pfc->Vac.high_fix_value = _pfc->Vac.high_value - _pfc->Vac.balance;

                        if (_pfc->Vac.high_fix_filter_cnt <= 2)
                        {
                            _pfc->Vac.high_fix_filter[_pfc->Vac.high_fix_filter_cnt] = _pfc->Vac.high_fix_value;
                            if (_pfc->Vac.high_fix_filter[_pfc->Vac.high_fix_filter_cnt] > _pfc->Vac.high_fix_filter_max)
                            {
                                _pfc->Vac.high_fix_filter_max = _pfc->Vac.high_fix_filter[_pfc->Vac.high_fix_filter_cnt];
                            }
                            if (_pfc->Vac.high_fix_filter[_pfc->Vac.high_fix_filter_cnt] < _pfc->Vac.high_fix_filter_min)
                            {
                                _pfc->Vac.high_fix_filter_min = _pfc->Vac.high_fix_filter[_pfc->Vac.high_fix_filter_cnt];
                            }
                            _pfc->Vac.high_fix_filter_sum += _pfc->Vac.high_fix_filter[_pfc->Vac.high_fix_filter_cnt];
                            _pfc->Vac.high_fix_filter_cnt++;
                            if (_pfc->Vac.high_fix_filter_cnt == 3)
                            {
                                _pfc->Vac.high_fix_filter_cnt = 0;
                                _pfc->Vac.high_fix_filter_mid = _pfc->Vac.high_fix_filter_sum - _pfc->Vac.high_fix_filter_max - _pfc->Vac.high_fix_filter_min;
                                _pfc->Vac.high_fix_filter_min = 0xffff;
                                _pfc->Vac.high_fix_filter_max = 0;
                                _pfc->Vac.high_fix_filter_sum = 0;
                            }
                        }
                        _pfc->Vac.high_value = _pfc->Vac.balance;
                        _pfc->Vac.low_value = _pfc->Vac.balance;
                    }
                }
            }
        }
    }

    if (_pfc->drv.Vac > _pfc->Vac.balance)
    {
        _pfc->drv.m_Vac = _pfc->drv.Vac - _pfc->Vac.balance;
    }
    else
    {
        _pfc->drv.m_Vac = _pfc->Vac.balance - _pfc->drv.Vac;
    }

    /* Vdc_ave */
    if (_pfc->Vac.zero_flag == 1)
    {
        _pfc->dcave.Vdc_ave = _pfc->dcave.Vdc_add / _pfc->dcave.update_cnt;
        _pfc->dcave.Vdc_add = 0;
        _pfc->dcave.update_cnt = 0;
    }
    else
    {
        _pfc->dcave.update_cnt++;
        _pfc->dcave.Vdc_add += _pfc->drv.Vdc_lpf;
    }

}
/**
 *********************************************************************************************
 * @brief       Control Voltage.
 *
 * @param       pfc_t* const    _pfc
 *
 * @return      none
 *********************************************************************************************
 */
void HPFC_ControlVoltage_16k(pfc_t * const _pfc)
{
    /* Determination of Iac_amp_ref */
    if (_pfc->para.vol_loop == 1)
    {

        q31_t error;
        _pfc->para.vol_loop = 0;
        _pfc->drv.Vdc_ref = _pfc->drv.target_Vdc;

        /* Error calculation of Vdc */
        error = (q15_t)(_pfc->drv.Vdc_ref - _pfc->dcave.Vdc_ave);

        /* Integral term calculation */
        _pfc->drv.Iac_amp_ref_P = ((error * _pfc->para.vol.kp) >> 10);
        _pfc->drv.Iac_amp_ref_I += ((error * _pfc->para.vol.ki) >> 10);

        /* Limit value excess */
        if (_pfc->drv.Iac_amp_ref_I  < _pfc->para.Iac_amp_lim_dw)
        {
            _pfc->drv.Iac_amp_ref_I = _pfc->para.Iac_amp_lim_dw;
        }
        else if (_pfc->drv.Iac_amp_ref_I > _pfc->para.Iac_amp_lim_up)
        {
            _pfc->drv.Iac_amp_ref_I = (q15_t)(_pfc->para.Iac_amp_lim_up);
        }
        _pfc->drv.Iac_amp_ref = (_pfc->drv.Iac_amp_ref_P + _pfc->drv.Iac_amp_ref_I);
        /* Limit value excess */
        if (_pfc->drv.Iac_amp_ref  < 0)
        {
            _pfc->drv.Iac_amp_ref = 0;
        }
    }

    /* New add */
    _pfc->drv.Iac_ref_buff = ((_pfc->drv.m_Vac * _pfc->drv.Iac_amp_ref) >> 10);
    if (_pfc->drv.Iac_ref_buff >= _pfc->para.Iac_amp_lim)
    {
        _pfc->drv.Iac_ref_buff = _pfc->para.Iac_amp_lim;
    }
}

/*************************************************End**********************************************************/
