/**
 ****************************************************************************
 * @file    ipdrv_adc.h
 * @brief   This file provides all the functions prototypes for M4Kx ADC driver.
 * @version V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2023
 *****************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __IPDRV_ADC_H
#define __IPDRV_ADC_H

#ifdef __cplusplus
extern "C" {
#endif

/* Includes ------------------------------------------------------------------*/
#include "TMPM4KyA.h"
#include "ipdrv_common.h"

/** @addtogroup TX04_Periph_Driver
  * @{
  */

/** @addtogroup ADC
  * @{
  */

/** @addtogroup ADC_Exported_types
  * @{
  */

#define ADC_NUM_MAX                 ((uint32_t)16)            /*!< Max Num of conversion. */

#define IS_ADC_UNIT(param)          ((param) == TSB_ADA || (param) == TSB_ADB || (param) == TSB_ADC)

/* macros to set Analog Input channel or Program number together with its Enable/Disable setting for the trigger registers */
#define TRG_ENABLE(AINx_ProgX)      ((uint8_t)(0x80U | (AINx_ProgX)))
#define TRG_DISABLE(AINx_ProgX)     ((uint8_t)(AINx_ProgX))

/* ADC Sampling Period */
#define ADC_SAMPLING_PERIOD_1           ((uint32_t)0x00000000)    /*!< SCLK Period*NN*1    */
#define ADC_SAMPLING_PERIOD_2           ((uint32_t)0x00000008)    /*!< SCLK Period*NN*2    */
#define ADC_SAMPLING_PERIOD_3           ((uint32_t)0x00000010)    /*!< SCLK Period*NN*3    */
#define ADC_SAMPLING_PERIOD_4           ((uint32_t)0x00000018)    /*!< SCLK Period*NN*4    */
#define ADC_SAMPLING_PERIOD_16          ((uint32_t)0x00000028)    /*!< SCLK Period*NN*16   */
#define ADC_SAMPLING_PERIOD_64          ((uint32_t)0x00000038)    /*!< SCLK Period*NN*64   */
#define ADC_SAMPLING_PERIOD_128         ((uint32_t)0x00000040)    /*!< SCLK Period*NN*128  */
#define ADC_SAMPLING_PERIOD_256         ((uint32_t)0x00000048)    /*!< SCLK Period*NN*256  */
#define ADC_SAMPLING_PERIOD_512         ((uint32_t)0x00000050)    /*!< SCLK Period*NN*512  */
#define ADC_SAMPLING_PERIOD_1024        ((uint32_t)0x00000058)    /*!< SCLK Period*NN*1024 */
#define IS_ADC_SAMPLING_PERIOD(param)   (((param) == ADC_SAMPLING_PERIOD_1) || \
                                        ((param) == ADC_SAMPLING_PERIOD_2) || \
                                        ((param) == ADC_SAMPLING_PERIOD_3) || \
                                        ((param) == ADC_SAMPLING_PERIOD_4) || \
                                        ((param) == ADC_SAMPLING_PERIOD_16) || \
                                        ((param) == ADC_SAMPLING_PERIOD_64) || \
                                        ((param) == ADC_SAMPLING_PERIOD_128) || \
                                        ((param) == ADC_SAMPLING_PERIOD_256) || \
                                        ((param) == ADC_SAMPLING_PERIOD_512) || \
                                        ((param) == ADC_SAMPLING_PERIOD_1024))

/* Select AD prescaler output (SCLK) */
#define ADC_SCLK_1                      ((uint32_t)0x00000000)    /*!< ADCLK/1    */
#define ADC_SCLK_2                      ((uint32_t)0x00000001)    /*!< ADCLK/2    */
#define ADC_SCLK_4                      ((uint32_t)0x00000002)    /*!< ADCLK/4    */
#define ADC_SCLK_8                      ((uint32_t)0x00000003)    /*!< ADCLK/8    */
#define ADC_SCLK_16                     ((uint32_t)0x00000004)    /*!< ADCLK/16   */
#define IS_ADC_PRESCALER(param)     ((param) <= ADC_SCLK_16)


    typedef enum {
        ADC_TRG_SINGLE = 0U,
        ADC_TRG_CONTINUE = 1U
    } ADC_TrgType;
#define IS_ADC_TRGTYPE(param)       ((param) <= ADC_TRG_CONTINUE)
#define IS_ADC_START(param)         ((param) <= ADC_TRG_CONTINUE)


#if (defined(__TMPM4KNA_H__)||defined(__TMPM4KLA_H__))
    typedef enum {
        ADC_CLKMOD_160M = 0U,
        ADC_CLKMOD_120M = 1U,
        ADC_CLKMOD_80M = 0U
    } ADC_ClkMod;
#define IS_ADC_CLKMOD(param)        ((param) <= ADC_CLKMOD_120M)
#else
    typedef enum {
        ADC_CLKMOD_0_5 = 0U,
        ADC_CLKMOD_0_85 = 1U
    } ADC_ClkMod;
#define IS_ADC_CLKMOD(param)        ((param) <= ADC_CLKMOD_0_85)
#endif

    typedef enum {
        ADC_REFSEL_VREFH = 0U,
        ADC_REFSEL_AIN = 1U
    } ADC_RefSel;
#define IS_ADC_REFSEL(param)        ((param) <= ADC_REFSEL_AIN)

    typedef enum {
        ADC_RCUT_NORMAL = 0U,
        ADC_RCUT_LOWPOWER = 1U
    } ADC_RcutMode;
#define IS_ADC_RCUTMODE(param)      ((param) <= ADC_RCUT_LOWPOWER)


	typedef enum {
        ADC_REG0 = 0U,
        ADC_REG1 = 1U,
        ADC_REG2 = 2U,
        ADC_REG3 = 3U,
        ADC_REG4 = 4U,
        ADC_REG5 = 5U,
        ADC_REG6 = 6U,
        ADC_REG7 = 7U,
        ADC_REG8 = 8U,
        ADC_REG9 = 9U,
        ADC_REG10 = 10U,
        ADC_REG11 = 11U,
        ADC_REG12 = 12U,
        ADC_REG13 = 13U,
        ADC_REG14 = 14U,
        ADC_REG15 = 15U,
        ADC_REG16 = 16U,
        ADC_REG17 = 17U,
        ADC_REG18 = 18U,
        ADC_REG19 = 19U,
        ADC_REG20 = 20U,
        ADC_REG21 = 21U,
        ADC_REG22 = 22U,
        ADC_REG23 = 23U,
        ADC_REG24 = 24U,
        ADC_REG25 = 25U,
        ADC_REG26 = 26U,
        ADC_REG27 = 27U,
        ADC_REG28 = 28U,
        ADC_REG29 = 29U,
        ADC_REG30 = 30U,
        ADC_REG31 = 31U
    } ADC_REGx;
#define IS_ADC_REGx(param)          ((param) <= ADC_REG31)


    typedef enum {
        PMD_PROG0 = 0U,
        PMD_PROG1 = 1U,
        PMD_PROG2 = 2U,
        PMD_PROG3 = 3U,
        PMD_PROG4 = 4U,
        PMD_PROG5 = 5U,
        PMD_PROG6 = 6U,
        PMD_PROG7 = 7U,
        PMD_PROG8 = 8U
    } PMD_PROGx;
#define IS_PMD_PROGx(param)   ((param) <= PMD_PROG8)


    typedef enum {
        PMD_INTNONE = 0U,
        PMD_INTADPDA = 1U,
        PMD_INTADPDB = 2U,
        PMD_INTADPDC = 3U,
        PMD_INTADPDD = 4U
    } PMD_INT_NAME;
#define IS_PMD_INT_NAME(param)   ((param) <= PMD_INTADPDD)


	/*!< define for Analog Input channel */
    typedef enum {
        ADC_AIN0 = 0U,
        ADC_AIN1 = 1U,
        ADC_AIN2 = 2U,
        ADC_AIN3 = 3U,
        ADC_AIN4 = 4U,
        ADC_AIN5 = 5U,
        ADC_AIN6 = 6U,
        ADC_AIN7 = 7U,
        ADC_AIN8 = 8U,
        ADC_AIN9 = 9U,
        ADC_AIN10 = 10U,
        ADC_AIN11 = 11U,
        ADC_AIN12 = 12U,
        ADC_AIN13 = 13U,
        ADC_AIN14 = 14U,
        ADC_AIN15 = 15U,
        ADC_AIN16 = 16U,
        ADC_AIN17 = 17U,
        ADC_AIN18 = 18U,
        ADC_AIN19 = 19U,
        ADC_AIN20 = 20U,
        ADC_AIN21 = 21U,
        ADC_AIN22 = 22U,
        ADC_AIN23 = 23U
    } ADC_AINx;

    typedef enum {
        GEN_TRG_DISABLE = 0U,
        GEN_TRG_CONTINUOUS = 1U,
        GEN_TRG_SINGLE = 2U,
        GEN_TRG_TRIGGER = 3U
    } ADC_GeneralTrgMode;

    typedef enum {
        VE_PHASE_NONE = 0U,
        VE_PHASE_U = 1U,
        VE_PHASE_V = 2U,
        VE_PHASE_W = 3U
    } VE_PHASE;
#define IS_VE_PHASE(param)      ((param) <= VE_PHASE_W)


/**
  * @brief Union to store ADC result
  */
    typedef union {
        uint32_t All;
        struct {
            uint32_t Stored:1;  /*!< bit0, AD result has been stored */
            uint32_t OverRun:1; /*!< bit1, OverRun, new AD result is stored before the old one been read */
            uint32_t Reserved1:2;       /*!< bit2 to bit3, reserved */
            uint32_t ADResult:12;       /*!< bit4 to bit15, store AD result */
            uint32_t Reserved2:16;      /*!< bit16 to bit31, reserved */
        } Bit;
    } ADC_Result;

/** @} */
/* End of group ADC_Exported_types */

/** @defgroup ADC_Exported_FunctionPrototypes
  * @{
  */
    void ADC_SetClk(TSB_AD_TypeDef * ADx, uint32_t Sampling_Period, uint32_t Prescaler_Output);
    void ADC_Enable(TSB_AD_TypeDef * ADx, ADC_ClkMod ClkMod, ADC_RefSel Sel, ADC_RcutMode Rcut);
    void ADC_Start(TSB_AD_TypeDef * ADx, ADC_TrgType Trg);
    WorkState ADC_GetConvertState(TSB_AD_TypeDef * ADx, ADC_TrgType Trg);
    ADC_Result ADC_GetConvertResult(TSB_AD_TypeDef * ADx, ADC_REGx ResultREGx);
    void ADC_SetSWTrg(TSB_AD_TypeDef * ADx, ADC_REGx ResultREGx, uint8_t MacroAINx);

/** @} */
/* End of group ADC_Exported_FunctionPrototypes */

/** @} */
/* End of group ADC */

/** @} */
/* End of group TX04_Periph_Driver */
#ifdef __cplusplus
}
#endif                          /* __cplusplus */
#endif                          /*__IPDRV_ADC_H */
