/**
 ****************************************************************************
 * @file    ipdrv_t32a.h
 * @brief   This file provides all the functions prototypes for M4Kx T32A driver.
 * @version V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2023
 *****************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __IPDRV_T32A_H
#define __IPDRV_T32A_H

#ifdef __cplusplus
extern "C" {
#endif                          /* __cplusplus */

/* Includes ------------------------------------------------------------------*/
#include "TMPM4KyA.h"
#include "ipdrv_common.h"

/** @addtogroup TX04_Periph_Driver
  * @{
  */
/** @addtogroup T32A
  * @{
  */
/** @defgroup T32A_Exported_Types
  * @{
  */

/** @brief The TimerUnit enum
  * Note for porting:
  * the timer value order from low to high with '1' step
  * and begin with "0".
  */
    typedef enum {
        TIMER_A = 0U,
        TIMER_B = 1U,
        TIMER_C = 2U,
        TOTAL_TIMER_UNIT = 3U,
    } TimerUnit;

    typedef struct {
        __IO uint32_t RUN;              /*!< T32A Run Register                           */
        __IO uint32_t CR;               /*!< T32A Control Register                       */
        __IO uint32_t CAPCR;            /*!< T32A Capture Control Register               */
        __O  uint32_t OUTCR0;           /*!< T32A Output Control Register 0              */
        __IO uint32_t OUTCR1;           /*!< T32A Output Control Register 1              */
        __IO uint32_t ST;               /*!< T32A Status Register                        */
        __IO uint32_t IM;               /*!< T32A Interrupt Mask Register                */
        __I  uint32_t TMR;              /*!< T32A Counter Capture Register               */
        __IO uint32_t RELD;             /*!< T32A Reload Register                        */
        __IO uint32_t RG0;              /*!< T32A Timer Register 0                       */
        __IO uint32_t RG1;              /*!< T32A Timer Register 1                       */
        __I  uint32_t CAP0;             /*!< T32A Capture Register 0                     */
        __I  uint32_t CAP1;             /*!< T32A Capture Register 1                     */
        __IO  uint32_t DMA;              /*!< T32A DMA Request Enable Register            */
    } TSB_T32A_TIMER_TypeDef;

/**
  * @brief T32A Init Structure definition
  */
    typedef struct {
        uint32_t CounterMode;         /*!< Select T32A counter mode between up, down, updown, 1-phase pulse 
                                           counter mode and 2-phase pulse counter mode */
        uint32_t ClkSrc;              /*!< Select the T32A source clock */
        uint32_t ClkDiv;              /*!< Select the division for T32A source clock */
        uint32_t Compare0Value;       /*!< Specify the compare0 value to be written 
                                           into T32AxRG0 register */
        uint32_t Compare1Value;       /*!< Specify the compare1 value to be written 
                                           into T32AxRG1 register */
        uint32_t ReloadValue;         /*!< Specify the reload value to be written 
                                           into T32AxRELD register */
    } T32A_InitTypeDef;

/** @} */
/* End of group T32A_Exported_Types */

/** @defgroup T32A_Exported_Constants
  * @{
  */
#define IS_T32A_ALL_PERIPH(param)    (((param) == TSB_T32A0) || \
                                      ((param) == TSB_T32A1) || \
                                      ((param) == TSB_T32A2) || \
                                      ((param) == TSB_T32A3) || \
                                      ((param) == TSB_T32A4) || \
                                      ((param) == TSB_T32A5))

#define IS_T32A_ALL_UNIT(param)      (((param) == TIMER_A) || \
                                      ((param) == TIMER_B) || \
                                      ((param) == TIMER_C))
#define IS_T32A_OUTPUT_CTRL_BY_SW(param)     (((param) == T32A_OUTPUT_NOCHANGE) || \
                                              ((param) == T32A_OUTPUT_SET) || \
                                              ((param) == T32A_OUTPUT_CLEAR) || \
                                              ((param) == T32A_OUTPUT_REVERSE))
#define MODE_16                      ((uint32_t)0x00000000)
#define MODE_32                      ((uint32_t)0x00000001)
#define IS_T32A_MODE(param)          (((param) == MODE_16) || \
                                      ((param) == MODE_32))

#define T32A_COND_NONE               ((uint32_t)0x00000000)
#define T32A_COND_INTERNAL_TRIGGER   ((uint32_t)0x00000001)
#define T32A_COND_RISING_IN0         ((uint32_t)0x00000002)
#define T32A_COND_FALLING_IN0        ((uint32_t)0x00000003)
#define T32A_COND_RISING_OTHER       ((uint32_t)0x00000004)
#define T32A_COND_FALLING_OTHER      ((uint32_t)0x00000005)
#define T32A_COND_SYNC_CHANNEL       ((uint32_t)0x00000006)
#define T32A_COND_MATCH_COMPARE1     ((uint32_t)0x00000007)
#define IS_T32A_RELOAD_COND(param)   (((param) == T32A_COND_NONE) || \
                                      ((param) == T32A_COND_INTERNAL_TRIGGER) || \
                                      ((param) == T32A_COND_RISING_IN0) || \
                                      ((param) == T32A_COND_FALLING_IN0) || \
                                      ((param) == T32A_COND_RISING_OTHER) || \
                                      ((param) == T32A_COND_FALLING_OTHER) || \
                                      ((param) == T32A_COND_SYNC_CHANNEL) || \
                                      ((param) == T32A_COND_MATCH_COMPARE1))
#define IS_T32A_RELOAD_COND_PULSECNT(param)    (((param) == T32A_COND_NONE) || \
                                      ((param) == T32A_COND_INTERNAL_TRIGGER) || \
                                      ((param) == T32A_COND_RISING_OTHER) || \
                                      ((param) == T32A_COND_FALLING_OTHER) || \
                                      ((param) == T32A_COND_SYNC_CHANNEL) || \
                                      ((param) == T32A_COND_MATCH_COMPARE1))

#define T32A_CNT_MOD_UP               ((uint32_t)0x00000000)
#define T32A_CNT_MOD_DN               ((uint32_t)0x00000001)
#define T32A_CNT_MOD_UPDN             ((uint32_t)0x00000002)
#define T32A_CNT_MOD_PULSE_1          ((uint32_t)0x00000003)
#define T32A_CNT_MOD_PULSE_2          ((uint32_t)0x00000004)
#define IS_T32A_TIMERC_CNT_MOD(param) (((param) == T32A_CNT_MOD_UP) || \
                                       ((param) == T32A_CNT_MOD_DN) || \
                                       ((param) == T32A_CNT_MOD_UPDN) || \
                                       ((param) == T32A_CNT_MOD_PULSE_1) || \
                                       ((param) == T32A_CNT_MOD_PULSE_2))
#define IS_T32A_TIMERAB_CNT_MOD(param) (((param) == T32A_CNT_MOD_UP) || \
                                        ((param) == T32A_CNT_MOD_DN) || \
                                        ((param) == T32A_CNT_MOD_UPDN))

#define T32A_CLK_SRC_PRESCALER         ((uint32_t)0x00000000)
#define T32A_CLK_SRC_INTERNAL_TRIGGER  ((uint32_t)0x00000001)
#define T32A_CLK_SRC_RISING_OTHER      ((uint32_t)0x00000002)
#define T32A_CLK_SRC_FALLING_OTHER     ((uint32_t)0x00000003)
#define T32A_CLK_SRC_RISING_IN0        ((uint32_t)0x00000004)
#define T32A_CLK_SRC_FALLING_IN0       ((uint32_t)0x00000005)
#define IS_T32A_CLK_SRC(param)         (((param) == T32A_CLK_SRC_PRESCALER) || \
                                        ((param) == T32A_CLK_SRC_INTERNAL_TRIGGER) || \
                                        ((param) == T32A_CLK_SRC_RISING_OTHER) || \
                                        ((param) == T32A_CLK_SRC_FALLING_OTHER) || \
                                        ((param) == T32A_CLK_SRC_RISING_IN0) || \
                                        ((param) == T32A_CLK_SRC_FALLING_IN0))

#define T32A_CLK_DIV_1                 ((uint32_t)0x00000000)
#define T32A_CLK_DIV_2                 ((uint32_t)0x00000001)
#define T32A_CLK_DIV_8                 ((uint32_t)0x00000002)
#define T32A_CLK_DIV_32                ((uint32_t)0x00000003)
#define T32A_CLK_DIV_128               ((uint32_t)0x00000004)
#define T32A_CLK_DIV_256               ((uint32_t)0x00000005)
#define T32A_CLK_DIV_512               ((uint32_t)0x00000006)
#define T32A_CLK_DIV_1024              ((uint32_t)0x00000007)
#define IS_T32A_CLK_DIV(param)         (((param) == T32A_CLK_DIV_1) || \
                                        ((param) == T32A_CLK_DIV_2) || \
                                        ((param) == T32A_CLK_DIV_8) || \
                                        ((param) == T32A_CLK_DIV_32) || \
                                        ((param) == T32A_CLK_DIV_128) || \
                                        ((param) == T32A_CLK_DIV_256) || \
                                        ((param) == T32A_CLK_DIV_512) || \
                                        ((param) == T32A_CLK_DIV_1024))

#define T32A_START_BY_SW                    ((uint32_t)0x00000002)
#define T32A_STOP_BY_SW                     ((uint32_t)0x00000004)
#define IS_T32A_CMD(param)                  (((param) == T32A_START_BY_SW) || ((param) == T32A_STOP_BY_SW))

#define T32A_INTMASK_NONE                   ((uint32_t)0x00000000)
#define T32A_INTMASK_MATCH_COMPARE0         ((uint32_t)0x00000001)
#define T32A_INTMASK_MATCH_COMPARE1         ((uint32_t)0x00000002)
#define T32A_INTMASK_OVERFLOW               ((uint32_t)0x00000004)
#define T32A_INTMASK_UNDERFLOW              ((uint32_t)0x00000008)
#define T32A_INTMASK_TRANSITION_ERROR       ((uint32_t)0x00000010)
#define T32A_TIMERAB_INTMASK_ALL            ((uint32_t)0x0000000F)
#define T32A_TIMERC_INTMASK_ALL             ((uint32_t)0x0000001F)
#define IS_T32A_TIMERAB_INTMASK(param)      (((param) <= T32A_TIMERAB_INTMASK_ALL))
#define IS_T32A_TIMERC_INTMASK(param)       (((param) <= T32A_TIMERC_INTMASK_ALL))

#define IS_T32A_TIMERAB_CNT_VAL(param)       ((param) <= 0xFFFFU)
/*--------------------------------PFC ------------------------------------------*/
#define OUTCR1_OCRCAP0_CLEAR  ((uint32_t)0xFFFFFFCF)
#define OUTCR1_OCRCAP1_CLEAR  ((uint32_t)0xFFFFFF3F)
#define T32A_OUTPUT_REVERSE                  ((uint32_t)0x00000003)
#define T32A_OUTPUT_FACTOR_CAPTURE_0         ((uint8_t)0x02)
#define T32A_OUTPUT_FACTOR_CAPTURE_1         ((uint8_t)0x03)
#define T32A_OUTPUT_NOCHANGE                 ((uint32_t)0x00000000)
#define T32A_OUTPUT_SET                      ((uint32_t)0x00000001)
#define T32A_OUTPUT_CLEAR                    ((uint32_t)0x00000002)
#define T32A_OUTPUT_FACTOR_COMPARE_0         ((uint8_t)0x00)
#define T32A_OUTPUT_FACTOR_COMPARE_1         ((uint8_t)0x01)
#define IS_T32A_OUTPUT_FACTOR(param)         (((param) == T32A_OUTPUT_FACTOR_COMPARE_0) || \
                                              ((param) == T32A_OUTPUT_FACTOR_COMPARE_1) || \
                                              ((param) == T32A_OUTPUT_FACTOR_CAPTURE_0) || \
                                              ((param) == T32A_OUTPUT_FACTOR_CAPTURE_1))
#define IS_T32A_OUTPUT_CTRL(param)           (((param) == T32A_OUTPUT_SET) || \
                                              ((param) == T32A_OUTPUT_CLEAR) || \
                                              ((param) == T32A_OUTPUT_REVERSE))
                                              #define OUTCR1_OCRCMP0_CLEAR  ((uint32_t)0xFFFFFFFC)
#define OUTCR1_OCRCMP1_CLEAR  ((uint32_t)0xFFFFFFF3)


/*------------------------------------------------------------------------------*/
/** @} */
/* End of group T32A_Exported_Constants */

/** @defgroup T32A_Exported_FunctionPrototypes
  * @{
  */

    Result T32A_SetMode(TSB_T32A_TypeDef * T32Ax, uint32_t Mode);
    Result T32A_Enable(TSB_T32A_TypeDef * T32Ax, TimerUnit Unit);
    void T32A_SetRunStateBySW(TSB_T32A_TypeDef * T32Ax, TimerUnit Unit, uint32_t Cmd);
    FunctionalState T32A_GetTimerState(TSB_T32A_TypeDef * T32Ax, TimerUnit Unit);
	void T32A_SetOutputCtrlBySW(TSB_T32A_TypeDef * T32Ax, TimerUnit Unit, uint32_t OutputCtrl);
	Result T32A_SetOutputCtrl(TSB_T32A_TypeDef * T32Ax, TimerUnit Unit, uint8_t OutputFactor, uint32_t OutputCtrl);
    Result T32A_SetReloadCond(TSB_T32A_TypeDef * T32Ax, TimerUnit Unit, uint32_t ReloadCond);
    Result T32A_Init(TSB_T32A_TypeDef * T32Ax, TimerUnit Unit, T32A_InitTypeDef * InitStruct);
    Result T32A_SetINTMask(TSB_T32A_TypeDef * T32Ax, TimerUnit Unit, uint32_t INTMask);
void T32A_ChangeCompare1Value(TSB_T32A_TypeDef * T32Ax, TimerUnit Unit, uint32_t Compare1Value);
void T32A_ChangeCompare0Value(TSB_T32A_TypeDef * T32Ax, TimerUnit Unit, uint32_t Compare1Value);
Result T32A_SetDoubleBuf(TSB_T32A_TypeDef * T32Ax, TimerUnit Unit, FunctionalState NewState);

/** @} */
/* End of group T32A_Exported_FunctionPrototypes */

/** @} */
/* End of group T32A */

/** @} */
/* End of group TX04_Periph_Driver */

#ifdef __cplusplus
}
#endif                          /* __cplusplus */
#endif                          /* __IPDRV_T32A_H */
