/**
 ****************************************************************************
 * @file	mcuip_drv.c
 * @brief	MCU IP driver for A-VE Plus
 * @version	V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2023
 *****************************************************************************
 */
 #include <stdlib.h>
#include "ipdefine.h"
#include "HPFC_Para.h"
#include "ipdrv_adc.h"
#include "ipdrv_cg.h"
#include "HPFC_drv.h"
#define DEFINE_APP
#include "mcuip_drv.h"
#undef DEFINE_APP
#include "D_Driver.h"
#include "E_sub.h"
/**
 *********************************************************************************************
 * @brief		Get EMGstatus
 *
 * @param		TSB_PMD_TypeDef*	PMDx	Select the PMD channel.
 *
 * @return		emg_status_e				EMG status
 *********************************************************************************************
 */
emg_status_e PMD_GetEMG_Status(TSB_PMD_TypeDef* const PMDx)
{
	return ((emg_status_e)(*((__I uint32_t*)BITBAND_PERI(&PMDx->EMGSTA, B_EMGST))));
}

/**
 *********************************************************************************************
 * @brief		Release EMG protection.
 *
 * @param		TSB_PMD_TypeDef*	PMDx	Select the PMD channel.
 *
 * @return		none
 *********************************************************************************************
 */
void PMD_ReleaseEMG_Protection(TSB_PMD_TypeDef* const PMDx)
{
	emg_status_e	emgst;

	emgst = PMD_GetEMG_Status(PMDx);

	if (emgst == cEMGProtected)
	{
		if ((*((__I uint32_t*)BITBAND_PERI(&PMDx->EMGSTA, B_EMGI))) == 1)
		{
			if (PMDx == TSB_PMD1)
			{
			#if defined(__CONTROL_MOTOR_COMP)
				#if !defined(__USE_VE_COMP)
				(*((__O uint32_t*)BITBAND_PERI(&PMDx->EMGCR, B_EMGRS))) = 1;
				#endif
			#endif
			}
			else
			{
			#if defined(__CONTROL_MOTOR_FAN)
				#if !defined(__USE_VE_FAN)
				(*((__O uint32_t*)BITBAND_PERI(&PMDx->EMGCR, B_EMGRS))) = 1;
				#else
				TSB_VE0->EMGRS = 1;						/*Release EMG protection */
				#endif
			#endif
			}
		}
	}

}

/**
 *********************************************************************************************
 * @brief		PMD_RegDataSet
 *
 * @param		TSB_PMD_TypeDef* const	PMDx		Select the PMD channel.
 * @param		 uint16_t  duty_u					U-phase PWM duty ratio
 * @param		 uint16_t  duty_v					V-phase PWM duty ratio
 * @param		 uint16_t  duty_w					W-phase PWM duty ratio
 * @param		 q15_t  adtrg0						AD trigger timing 0 position
 * @param		 q15_t  adtrg1						AD trigger timing 1 position
 *
 * @return		none	
 *********************************************************************************************
 */
void PMD_RegDataSet(TSB_PMD_TypeDef* const PMDx, uint16_t duty_u, uint16_t duty_v, uint16_t duty_w, q15_t adtrg0, q15_t adtrg1)
{
	/* TRGCMPx = 0 to 0x7FFF */
	int32_t trgtmp0, trgtmp1;

	if(PMDx==TSB_PMD2)
	{
		PMDx->TRGCMP0 = adtrg0;
		PMDx->TRGCMP1 = adtrg1;
	}
	else
	{
		trgtmp0 = ((int32_t)adtrg0 + 32768) / 2;
		if (trgtmp0 >= 0x8000) trgtmp0 = 0x7fff;
		trgtmp1 = ((int32_t)adtrg1 + 32768) / 2;
		if (trgtmp1 >= 0x8000) trgtmp1 = 0x7fff;

		PMDx->TRGCMP0 = trgtmp0;
		PMDx->TRGCMP1 = trgtmp1;
	}

	/* When <CMPx> >= 0x8000, the duty ratio is 100%.*/
	PMDx->CMPU = duty_u;
	__disable_irq();
	PMDx->CMPV = duty_v;
	PMDx->CMPW = duty_w;
	__enable_irq();
}



/**
 ********************************************************************************************
 * @brief		Set OutputMode Register.
 *
 * @param[in]	TSB_PMD_TypeDef* const	PMDx	.
 * @param[in]	uint32_t				_mdout	.
 *
 * @return		none
 ********************************************************************************************
 */
void PMD_SetOutputMode(TSB_PMD_TypeDef* const PMDx, uint32_t _mdout)
{
	PMDx->MDOUT = _mdout;
	
}


/**
 ********************************************************************************************
 * @brief		Set Trigger select Register.
 *
 * @param[in]	TSB_PMD_TypeDef* const	PMDx	.
 * @param[in]	uint32_t				_trgsel	.
 *
 * @return		none
 ********************************************************************************************
 */
void PMD_SetTriggerSelect(TSB_PMD_TypeDef* const PMDx, uint32_t _trgsel)
{
	PMDx->TRGSEL = _trgsel;
	
}


/**
 *********************************************************************************************
 * @brief		VE start
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 *
 * @return		none
 *********************************************************************************************
 */
void VE_Start(TSB_VE_TypeDef* const VEx)
{
	VEx->CPURUNTRG = VE_START;

}

/**
 *********************************************************************************************
 * @brief		VE_GetTask
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 *
 * @return		uint32_t						Task No.
 *********************************************************************************************
 */
uint32_t VE_GetTask(const TSB_VE_TypeDef* const VEx)
{
	return((VEx->SCHTASKRUN >> 1U) & 0x0000000f);
}

/**
 *********************************************************************************************
 * @brief		Get Phase Current from VE register.
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 * @param		q15_t*					_ia		U-phase current value store address.
 * @param		q15_t*					_ib		V-phase current value store address.
 * @param		q15_t*					_ic		W-phase current value store address.
 *
 * @return		none
 *********************************************************************************************
 */
void VE_GetPhaseCurrent(TSB_VE_TypeDef* const VEx, q15_t* _ia, q15_t* _ib, q15_t* _ic)
{
	*_ia = VEx->IA >> 16;
	*_ib = VEx->IB >> 16;
	*_ic = VEx->IC >> 16;
}

/**
 *********************************************************************************************
 * @brief		Get current adc data from VE register.
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 * @param		uint32_t*				_adc_ia	U-phase current ad value store address.
 * @param		uint32_t*				_adc_ib	V-phase current ad value store address.
 * @param		uint32_t*				_adc_ic	W-phase current ad value store address.
 *
 * @return		none
 *********************************************************************************************
 */
void VE_GetCurrentAdcData(TSB_VE_TypeDef* const VEx, uint32_t* _adc_ia, uint32_t* _adc_ib, uint32_t* _adc_ic)
{

	*_adc_ia = VEx->IAADC;
	*_adc_ib = VEx->IBADC;
	*_adc_ic = VEx->ICADC;

}

/**
 *********************************************************************************************
 * @brief		Get data from VE registers.
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 * @param		vector_t* const			_motor	Structure address of vector control variable.
 *
 * @return		none
 *********************************************************************************************
 */
void VE_GetdataFromVEreg(TSB_VE_TypeDef* const VEx, vector_t* const _motor)
{
	_motor->drv.Vdc_temp = VEx->VDC;
	_motor->drv.Vdc += (_motor->drv.Vdc_temp-_motor->drv.Vdc)>>5;
	VEx->VDC = _motor->drv.Vdc;

	_motor->drv.Vd.word = VEx->VD;								/* Read Vd */
	_motor->drv.Vq.word = VEx->VQ;								/* Read Vd */
	if (_motor->drv.idetect_error == 0)
	{
		_motor->drv.Id = (q15_t)(VEx->ID >> 16);				/* Read Id */
		_motor->drv.Iq = (q15_t)(VEx->IQ >> 16);				/* Read Iq */
	}
	else
	{
		VEx->ID = (q31_t)_motor->drv.Id << 16;					/* Set Id */
		VEx->IQ = (q31_t)_motor->drv.Iq << 16;					/* Set Iq */
	}
}

/**
 *********************************************************************************************
 * @brief		VE_GetPWM_DutyU
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 *
 * @return		uint32_t						U Phase duty.
 *********************************************************************************************
 */
uint32_t VE_GetPWM_DutyU(TSB_VE_TypeDef* const VEx)
{
	return(VEx->CMPU);
}

/**
 *********************************************************************************************
 * @brief		VE_GetPWM_DutyV
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 *
 * @return		uint32_t						V Phase duty.
 *********************************************************************************************
 */
uint32_t VE_GetPWM_DutyV(TSB_VE_TypeDef* const VEx)
{
	return(VEx->CMPV);
}

/**
 *********************************************************************************************
 * @brief		VE_GetPWM_DutyW
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 *
 * @return		uint32_t						W Phase duty.
 *********************************************************************************************
 */
uint32_t VE_GetPWM_DutyW(TSB_VE_TypeDef* const VEx)
{
	return(VEx->CMPW);
}

/**
 *********************************************************************************************
 * @brief		Get medium duty from VE registers.
 *
 * @param		TSB_VE_TypeDef* const	VEx		Select the VE channel.
 *
 * @return		uint32_t						Medium duty value.
 *********************************************************************************************
 */
uint32_t VE_GetPWM_DutyMed(TSB_VE_TypeDef* const VEx)
{
	uint32_t	a,b,c,ret;

	a = VEx->CMPU;
	b = VEx->CMPV;
	c = VEx->CMPW;

	if (a >= b)
	{
		ret = (c < b)? b : (c > a)? a : c;
	}
	else
	{
		ret = (c < a)? a : (c > b)? b : c;
	}

	return ret;

}

/**
 *********************************************************************************************
 * @brief		Get medium duty from VE registers.
 *
 * @param		TSB_VE_TypeDef* const	VEx		Select the VE channel.
 *
 * @return		uint32_t						Medium duty value.
 *********************************************************************************************
 */
uint32_t VE_GetPWM_DutyMin(TSB_VE_TypeDef* const VEx)
{
	uint32_t	a,b,c,ret;

	a = VEx->CMPU;
	b = VEx->CMPV;
	c = VEx->CMPW;

	if (a >= b)
	{
		ret = (c < b)? c : b;
	}
	else
	{
		ret = (c < a)? c : a;
	}

	return ret;

}

/**
 *********************************************************************************************
 * @brief		VE_GetPWM_Modulation
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 *
 * @return		int								PWM modulation 0:3Phase, 1:2Phase
 *********************************************************************************************
 */
int	VE_GetPWM_Modulation(TSB_VE_TypeDef* const VEx)
{
	int	ret = 0;

	if ((VEx->FMODE & (1 << B_C2PEN)) != 0)
	{
		ret = 1;
	}

	return(ret);

}

/**
 *********************************************************************************************
 * @brief		VE_GetSector
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 *
 * @return		uint32_t						Sector0
 *********************************************************************************************
 */
uint32_t VE_GetSector(TSB_VE_TypeDef* const VEx)
{
	return(VEx->SECTOR);
}

/**
 *********************************************************************************************
 * @brief		VE_GetShiftPWMState
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 *
 * @return		int								PWM Shift state 0:Normal PWM, 1;Shift PWM
 *********************************************************************************************
 */
int	VE_GetShiftPWMState(TSB_VE_TypeDef* const VEx)
{
	int	ret = 0;

	if ((VEx->MCTLF & (1 << B_LAVF)) != 0)
	{
		ret = 1;
	}

	return(ret);
}

/**
 *********************************************************************************************
 * @brief		VE_GetOutputMode
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 *
 * @return		int	
 *********************************************************************************************
 */
int	VE_GetOutputMode(TSB_VE_TypeDef* const VEx)
{
	return(((VEx->MODE >> 2U) & 0x00000003));
}



/**
 *********************************************************************************************
 * @brief		Set data to VE registers in Stop stage.
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 * @param		vector_t* const			_motor	Structure address of vector control variable.
 *
 * @return		none
 *********************************************************************************************
 */
void VE_SetdataToVEreg_Stop(TSB_VE_TypeDef* const VEx, const vector_t* const _motor)
{
	VEx->TASKAPP = SET_TAPP(TAPP_OUT_CON);						/* VE start from output control */
	VEx->ACTSCH = SET_ACTSCH(SCH_9);							/* Schedule 9 */

	VEx->MODE = VEMODE_STOP;									/* Output OFF */

	VEx->ID = 0;
	VEx->IQ = 0;
	VEx->VD = 0;
	VEx->VQ = 0;
	VEx->VDIH = 0;												/* Clear Current PI Control Integral Term Registers */
	VEx->VDILH = 0;
	VEx->VQIH = 0;
	VEx->VQILH = 0;
	VEx->IDREF = 0;
	VEx->IQREF = 0;
	VEx->THETA = 0;
	VEx->OMEGA = 0;
	VEx->SECTOR = 0;
	VEx->SECTORM = 0;
	VEx->MINPLS = 0;

}

/**
 *********************************************************************************************
 * @brief		Set data to VE registers in Bootstrap stage.
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 * @param		vector_t* const			_motor	Structure address of vector control variable.
 *
 * @return		none
 *********************************************************************************************
 */
void VE_SetdataToVEreg_Bootstrap(TSB_VE_TypeDef* const VEx, const vector_t* const _motor)
{

	VEx->TASKAPP = SET_TAPP(TAPP_SINCOS);						/* VE start from SIN/COS */
	VEx->ACTSCH = SET_ACTSCH(SCH_4);							/* Schedule 4 */

	VEx->MODE = VEMODE_BOOT;									/* Output ON */

}

/**
 *********************************************************************************************
 * @brief		Set data to VE registers in Initposition stage for current control.
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 * @param		vector_t* const			_motor	Structure address of vector control variable.
 *
 * @return		none
 *********************************************************************************************
 */
void VE_SetdataToVEreg_Initposition_i(TSB_VE_TypeDef* const VEx, const vector_t* const _motor)
{

	VEx->TASKAPP = SET_TAPP(TAPP_CRT_CON);						/* VE start from current control */
	VEx->ACTSCH = SET_ACTSCH(SCH_1);							/* Schedule 1 */

	VEx->MODE = (uint32_t)(VEMODE_INITPOS);						/* Output ON */

	VEx->IDREF = _motor->drv.Id_ref;							/* Set Id */
	VEx->IQREF = _motor->drv.Iq_ref; 							/* Set Iq */
	VEx->THETA = _motor->drv.theta.half[1];
	VEx->OMEGA = _motor->drv.omega.half[1];
	if (_motor->drv.command.spwm == OFF)
	{
		VEx->MINPLS = _motor->drv.chkpls;
	}
	else
	{
		VEx->MINPLS = _motor->drv.chkpls / 2;
	}
}

/**
 *********************************************************************************************
 * @brief		Set data to VE registers in Initposition stage for voltage control.
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 * @param		vector_t* const			_motor	Structure address of vector control variable.
 *
 * @return		none
 *********************************************************************************************
 */
void VE_SetdataToVEreg_Initposition_v(TSB_VE_TypeDef* const VEx, const vector_t* const _motor)
{
	VEx->TASKAPP = SET_TAPP(TAPP_SINCOS);						/* VE start from SIN/COS */
	VEx->ACTSCH = SET_ACTSCH(SCH_4);							/* Schedule 4 */

	VEx->MODE = VEMODE_INITPOS;									/* Output ON */

	VEx->VD = _motor->drv.Vd_out;								/* Vd */
	VEx->VQ = 0;
	VEx->THETA = _motor->drv.theta.half[1];
	VEx->OMEGA = _motor->drv.omega.half[1];
	if (_motor->drv.command.spwm == OFF)
	{
		VEx->MINPLS = _motor->drv.chkpls;
	}
	else
	{
		VEx->MINPLS = _motor->drv.chkpls / 2;
	}

}

/**
 *********************************************************************************************
 * @brief		Set data to VE registers in Force stage for current control.
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 * @param		vector_t* const			_motor	Structure address of vector control variable.
 *
 * @return		none
 *********************************************************************************************
 */
void VE_SetdataToVEreg_Force_i(TSB_VE_TypeDef* const VEx, const vector_t* const _motor)
{
	VEx->TASKAPP = SET_TAPP(TAPP_CRT_CON);						/* VE start from current control */
	VEx->ACTSCH = SET_ACTSCH(SCH_1);							/* Schedule 1 */

	VEx->MODE = (uint32_t)(VEMODE_FORCE);						/* Output ON */

	VEx->IDREF = _motor->drv.Id_ref;							/* Set Id */
	VEx->IQREF = _motor->drv.Iq_ref; 							/* Set Iq */
	VEx->THETA = _motor->drv.theta.half[1];
	VEx->OMEGA = _motor->drv.omega.half[1];
	if(_motor->drv.command.spwm == OFF)
	{
		VEx->MINPLS = _motor->drv.chkpls;
	}
	else
	{
		VEx->MINPLS = _motor->drv.chkpls / 2;
	}
}

/**
 *********************************************************************************************
 * @brief		Set data to VE registers in Force stage for voltage control.
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 * @param		vector_t* const			_motor	Structure address of vector control variable.
 *
 * @return		none
 *********************************************************************************************
 */
void VE_SetdataToVEreg_Force_v(TSB_VE_TypeDef* const VEx, const vector_t* const _motor)
{
	VEx->TASKAPP = SET_TAPP(TAPP_SINCOS);						/* VE start from SIN/COS */
	VEx->ACTSCH = SET_ACTSCH(SCH_4);							/* Schedule 4 */

	VEx->MODE = VEMODE_FORCE;									/* Output ON */

	VEx->VD = _motor->drv.Vd_out;								/* Vd */
	VEx->VDIH = _motor->drv.Vd_out;								/* Vd Integral value initialization */
	VEx->VQ = 0; 												/* Vq = 0 */
	VEx->THETA = _motor->drv.theta.half[1];						/* Theta set */
	VEx->OMEGA = _motor->drv.omega.half[1];						/* Omega set */
	if (_motor->drv.command.spwm == OFF)
	{
		VEx->MINPLS = _motor->drv.chkpls;
	}
	else
	{
		VEx->MINPLS = _motor->drv.chkpls / 2;
	}
}

/**
 *********************************************************************************************
 * @brief		Set data to VE registers in Change_up stage.
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 * @param		vector_t* const			_motor	Structure address of vector control variable.
 *
 * @return		none
 *********************************************************************************************
 */
void VE_SetdataToVEreg_Change_up(TSB_VE_TypeDef* const VEx, const vector_t* const _motor)
{
	VEx->TASKAPP = SET_TAPP(TAPP_CRT_CON);						/* VE start from current control */
	VEx->ACTSCH = SET_ACTSCH(SCH_1);							/* Schedule 1 */

	VEx->MODE = (uint32_t)(VEMODE_CHGUP);						/* Output ON */

	VEx->IDREF = _motor->drv.Id_ref;							/* Set Id */
	VEx->IQREF = _motor->drv.Iq_ref; 							/* Set Iq */
	VEx->THETA = _motor->drv.theta.half[1];
	VEx->OMEGA = _motor->drv.omega.half[1];
	if (_motor->drv.command.spwm == OFF)
	{
		VEx->MINPLS = _motor->drv.chkpls;
	}
	else
	{
		VEx->MINPLS = _motor->drv.chkpls / 2;
	}

}

/**
 *********************************************************************************************
 * @brief		Set data to VE registers in Steady_A stage.
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 * @param		vector_t* const			_motor	Structure address of vector control variable.
 *
 * @return		none
 *********************************************************************************************
 */
void VE_SetdataToVEreg_Steady_A(TSB_PMD_TypeDef* const PMDx,TSB_VE_TypeDef* const VEx, const vector_t* const _motor)
{
	VEx->TASKAPP = SET_TAPP(TAPP_CRT_CON);						/* VE start from current control */
	VEx->ACTSCH = SET_ACTSCH(SCH_1);							/* Schedule 1 */

	VEx->MODE = VEMODE_STEADY;									/* Output ON */

	VEx->IDREF = _motor->drv.Id_ref;							/* Id set */
	VEx->IQREF = _motor->drv.Iq_ref;							/* Iq set */
	VEx->THETA = _motor->drv.theta.half[1];						/* Theta set */
	VEx->OMEGA = _motor->drv.omega.half[1];						/* Omega set */

	if (_motor->drv.command.spwm == OFF)
	{
		VEx->MINPLS = _motor->drv.chkpls;
	}
	else
	{
		VEx->MINPLS = _motor->drv.chkpls / 2;
	}
}

/**
 *********************************************************************************************
 * @brief		Set data to VE registers in Emergency stage.
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 * @param		vector_t* const			_motor	Structure address of vector control variable.
 *
 * @return		none
 *********************************************************************************************
 */
void VE_SetdataToVEreg_Emergency(TSB_VE_TypeDef* const VEx, const vector_t* const _motor)
{
	VEx->TASKAPP = SET_TAPP(TAPP_OUT_CON);						/* VE start from output control */
	VEx->ACTSCH = SET_ACTSCH(SCH_9);							/* Schedule 9 */

	VEx->MODE = VEMODE_EMG;										/* Output OFF */
}

/**
 *********************************************************************************************
 * @brief		Set zero current data to VE register.
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 * @param		uint32_t				_z_ia	U Phase value.
 * @param		uint32_t				_z_ib	V Phase value.
 * @param		uint32_t				_z_ic	W Phase value.
 *
 * @return		none
 *********************************************************************************************
 */
void VE_SetZeroCurrentData(TSB_VE_TypeDef* const VEx, uint32_t _z_ia, uint32_t _z_ib, uint32_t _z_ic)
{
	VEx->IAO = _z_ia;
	VEx->IBO = _z_ib;
	VEx->ICO = _z_ic;

}

/**
 *********************************************************************************************
 * @brief		Set VEVDC reg.
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 * @param		q15_t					_dat	Power supply voltage(Q15)
 *
 * @return		none
 *********************************************************************************************
 */
void VE_SetVDCreg(TSB_VE_TypeDef* const VEx, q15_t _dat)
{
	VEx->VDC = _dat;
}

/**
 *********************************************************************************************
 * @brief		Set shift pwm_mode.
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 * @param		uint8_t					_dat	Shift pwm mode.
 *
 * @return		none
 *********************************************************************************************
 */
void VE_SetSpwmMode(TSB_VE_TypeDef* const VEx, uint8_t _dat)
{
	(*((__IO uint32_t*)BITBAND_PERI(&VEx->FMODE, B_SPWMEN))) = _dat;
}

/**
 *********************************************************************************************
 * @brief		Set modulation type.
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 * @param		uint8_t					_dat	modulation type.
 *
 * @return		none
 *********************************************************************************************
 */
void VE_SetModulType(TSB_VE_TypeDef* const VEx, uint8_t _dat)
{
	(*((__IO uint32_t*)BITBAND_PERI(&VEx->FMODE, B_C2PEN))) = _dat;
}

/**
 *********************************************************************************************
 * @brief		Acquire AD at the position in the first half of the PWM cycle.
 *
 * @param		vector_t* const	_motor			The structure address for motor vector control.
 *
 * @return		none
 *********************************************************************************************
 */
void	AD_GetIdcFirstHalf(vector_t* const _motor)
{
#if (defined(__CONTROL_MOTOR_COMP) && !defined(__USE_VE_COMP)) || (defined(__CONTROL_MOTOR_FAN) && !defined(__USE_VE_FAN))
	_motor->drv.Idc1_adc = (q15_t)(*_motor->drv.ADxREG1 & 0xFFF0);
	_motor->drv.Idc2_adc = (q15_t)(*_motor->drv.ADxREG0 & 0xFFF0);
#endif /* (defined(__CONTROL_MOTOR_COMP) && !defined(__USE_VE_CH1)) || (defined(__CONTROL_MOTOR_FAN) && !defined(__USE_VE_FAN)) */
}

/**
 *********************************************************************************************
 * @brief		Acquire AD at the position in the second half of the PWM cycle.
 *
 * @param		vector_t* const	_motor			The structure address for motor vector control.
 *
 * @return		none
 *********************************************************************************************
 */
void	AD_GetIdcSecondHalf(vector_t* const _motor)
{
#if (defined(__CONTROL_MOTOR_COMP) && !defined(__USE_VE_COMP)) || (defined(__CONTROL_MOTOR_FAN) && !defined(__USE_VE_FAN))
	_motor->drv.Idc1_adc = (q15_t)(*_motor->drv.ADxREG0 & 0xFFF0);
	_motor->drv.Idc2_adc = (q15_t)(*_motor->drv.ADxREG1 & 0xFFF0);
#endif /* (defined(__CONTROL_MOTOR_COMP) && !defined(__USE_VE_CH1)) || (defined(__CONTROL_MOTOR_FAN) && !defined(__USE_VE_FAN)) */
}

/**
 *********************************************************************************************
 * @brief		PMD IP Initialize
 *
 * @param		TSB_PMD_TypeDef* const	PMDx		Select the PMD channel.
 * @param		PMD_InitTypeDef* const	_initdata	The structure containing basic PMD configuration.
 *
 * @return		none
 *********************************************************************************************
 */
void IP_PMD_init(TSB_PMD_TypeDef* const PMDx, PMD_InitTypeDef* const _initdata)
{
	if (PMDx == TSB_PMD1)
	{
		CG_SetFsysSupplyPeriphB(CG_FSYS_PERIPH_PMD1, ENABLE);
	}
	else if(PMDx == TSB_PMD0)
	{
		CG_SetFsysSupplyPeriphB(CG_FSYS_PERIPH_PMD0, ENABLE);
	}
	else if(PMDx == TSB_PMD2)
	{
		CG_SetFsysSupplyPeriphB(CG_FSYS_PERIPH_PMD2, ENABLE);
	}

	PMDx->MODESEL = _initdata->busmode;

	/***** active setting *****/
	PMDx->MDPOT = cMDPOT_INI;
	if (_initdata->poll == 1)
	{
		PMDx->MDPOT |= (1U << B_POLL);
	}
	else
	{
		PMDx->MDPOT &= ~(1U << B_POLL);
	}
	if (_initdata->polh == 1)
	{
		PMDx->MDPOT |= (1U << B_POLH);
	}
	else
	{
		PMDx->MDPOT &= ~(1U << B_POLH);
	}

	PMDx->MDOUT = cMDOUT_OFF;									/* Output off */
	PMDx->CMPU = 0;
	PMDx->CMPV = 0;
	PMDx->CMPW = 0;

	/***** EMG setting *****/
#if (1)
	PMDx->EMGCR = cEMG_ENA; 									/* EMG Enable */
	PMDx->EMGCR = cEMG_Release; 								/* EMG Release */
#else
	PMDx->EMGREL = 0x5A;										/* Disable code */
	PMDx->EMGREL = 0xA5;
	PMDx->EMGCR = cEMG_DIS; 									/* EMG Disable */
#endif

	/* OVV disable set */
	PMDx->EMGREL = 0x5A;										/* Disable code */
	PMDx->EMGREL = 0xA5;
	PMDx->OVVCR = cOVV_DIS;

	PMDx->PORTMD = cPORTMD_INI;									/* Tool break */
	PMDx->MDCR = cMDCR_INI;
	if(PMDx == TSB_PMD2)
	{
		PMDx->MDCR = cMDCR_INI_PFC;
	}
	
	PMDx->RATE = _initdata->pwmrate;							/* PWM rate */
	PMDx->DTR = _initdata->deadtime; 							/* Dead timewr set */
	PMDx->MDEN = 0x01;											/* PMD enable (Set after set MDOUT,MDPOT) */

	/***** port setting *****/
	if (PMDx == TSB_PMD1)
	{
		CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_PORTE, ENABLE);
		TSB_PE->FR6 |= BIT8(0, 0, 1, 1, 1, 1, 1, 1); 			/* PMD1 PORT output Enable */
		TSB_PE->CR  |= BIT8(0, 0, 1, 1, 1, 1, 1, 1); 			/* PMD1 PORT output Enable */

		if ((*((__IO uint32_t*)BITBAND_PERI(&PMDx->EMGCR, B_EMGEN))) == 1)	/* EMG enable? */
		{
			TSB_PE->FR6 |= BIT8(0, 1, 0, 0, 0, 0, 0, 0);		/* PMD1 PORT Enable */
			TSB_PE->IE  |= BIT8(0, 1, 0, 0, 0, 0, 0, 0); 		/* PMD1 PORT Enable */
		}

	#if defined(__MOTOR_PMDDBG)	/* for debug */
	/* PMD DBG for COMP */
		CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_PORTC, ENABLE);
		TSB_PC_CR_PC3C = SET;
		TSB_PC_FR7_PC3F7 = SET;
		PMDx->DBGOUTCR = DBGOUT_ENA_ADC;								/* ADC convert timing monitor */
	#endif
	}
	else if (PMDx == TSB_PMD0)
	{
		CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_PORTB, ENABLE);
		TSB_PB->FR4 |= BIT8(0, 0, 1, 1, 1, 1, 1, 1); 					/* PMD0 PORT output Enable */
		TSB_PB->CR  |= BIT8(0, 0, 1, 1, 1, 1, 1, 1); 					/* PMD0 PORT output Enable */
		if ((*((__IO uint32_t*)BITBAND_PERI(&PMDx->EMGCR, B_EMGEN))) == 1)	/* EMG enable? */
		{
			TSB_PB->FR4 |= BIT8(0, 1, 0, 0, 0, 0, 0, 0); 				/* PMD0 PORT Enable */
			TSB_PB->IE  |= BIT8(0, 1, 0, 0, 0, 0, 0, 0); 				/* PMD0 PORT Enable */
		}

		if ((*((__IO uint32_t*)BITBAND_PERI(&PMDx->OVVCR, B_OVVEN))) == 1)	/* OVV enable? */
		{
			TSB_PB->FR4 |= BIT8(1, 0, 0, 0, 0, 0, 0, 0); 				/* PMD0 PORT Enable */
			TSB_PB->IE  |= BIT8(1, 0, 0, 0, 0, 0, 0, 0); 				/* PMD0 PORT Enable */
		}

	#if defined(__MOTOR_PMDDBG)	/* for debug */
	/* PMD DBG for FAN */
		CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_PORTC, ENABLE);
		TSB_PC_CR_PC2C = SET;
		TSB_PC_FR7_PC2F7 = SET;
		PMDx->DBGOUTCR = DBGOUT_ENA_ADC;								/* ADC convert timing monitor */
	#endif

	}
	else if (PMDx == TSB_PMD2)
	{
		CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_PORTU, ENABLE);
		#if defined(__HW_TMPM4KNA)
		TSB_PU->FR6 |= BIT8(0, 0, 1, 1, 1, 1, 1, 1); 					/* PMD2 PORT output Enable */
		#elif defined(__HW_TMPM4KLA)
		TSB_PU->FR6 |= BIT8(0, 0, 0, 1, 0, 0, 0, 0); 
		#endif
#if defined(__W_PFCOUT)
		TSB_PU->CR  |= BIT8(0, 0, 0, 1, 0, 0, 0, 0); 					/* PMD2 PORT output Enable */
#elif defined(__V_PFCOUT)
		TSB_PU->CR  |= BIT8(0, 0, 0, 0, 0, 1, 0, 0);
#elif defined(__U_PFCOUT)
		TSB_PU->CR  |= BIT8(0, 0, 0, 1, 0, 0, 0, 1);
#endif 
		if ((*((__IO uint32_t*)BITBAND_PERI(&PMDx->EMGCR, B_EMGEN))) == 1)	/* EMG enable? */
		{
			TSB_PU->FR6 |= BIT8(0, 1, 0, 0, 0, 0, 0, 0); 				/* PMD2 PORT Enable */
			TSB_PU->IE  |= BIT8(0, 1, 0, 0, 0, 0, 0, 0); 				/* PMD2 PORT Enable */
		}

		if ((*((__IO uint32_t*)BITBAND_PERI(&PMDx->OVVCR, B_OVVEN))) == 1)		/* OVV enable? */
		{
			TSB_PU->FR6 |= BIT8(1, 0, 0, 0, 0, 0, 0, 0); 				/* PMD2 PORT Enable */
			TSB_PU->IE  |= BIT8(1, 0, 0, 0, 0, 0, 0, 0); 				/* PMD2 PORT Enable */
		}

	#if defined(__MOTOR_PMDDBG)	/* for debug */
	/* PMD DBG for PFC */
		TSB_PU_CR_PU7C = SET;
		TSB_PU_FR7_PU7F7 = SET;*/

		CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_PORTA, ENABLE);
		TSB_PA_CR_PA2C = SET;
		TSB_PA_FR6_PA2F6 = SET;
		PMDx->DBGOUTCR = DBGOUT_ENA_ADC;								/* ADC convert timing monitor */
	#endif

	}

	/***** Release EMG protection *****/
	if ((*((__I uint32_t*)BITBAND_PERI(&PMDx->EMGSTA, B_EMGI))) == 1)
	{
		(*((__O uint32_t*)BITBAND_PERI(&PMDx->EMGCR, B_EMGRS))) = 1;
	}

	/***** Triger setting *****/
	PMDx->TRGCMP0 = 0; 													/* TRG position clear */
	PMDx->TRGCMP1 = 0;
	PMDx->TRGCMP2 = 0;
	PMDx->TRGCMP3 = 0;

	if (_initdata->shunt == c3shunt)
	{
		PMDx->TRGCR = cTRG_3SHUNT;
		PMDx->TRGMD = cTRGMD_3SHUNT;
	}
	else if (_initdata->shunt == c1shunt)
	{
		if (PMDx == TSB_PMD0)
		{
			PMDx->TRGCR = cTRG_1SHUNT_FAN_NORAML;
			PMDx->TRGCMP3 = (uint32_t)(FIX_15 * (cPWMPRD_FAN/2 - 4.0f)/ cPWMPRD_FAN);   /* Used for VAC sample in shift-2 mode */
		}
		else if (PMDx == TSB_PMD1)
		{
			PMDx->TRGCR = cTRG_1SHUNT;
		}
		PMDx->TRGMD = cTRGMD_1SHUNT;
	}
	else if(_initdata->shunt == cPFCshunt)
	{
		PMDx->TRGCR = cTRG_PFC;
		PMDx->TRGMD = cTRGMD_SHUNT_PFC;
        PMDx->TRGCMP0 = 0x3FFF;
		PMDx->TRGCMP1 = 0;
	}	
	PMDx->TRGSEL = 0;												/* TRG No = Sector0 = 0 */

	PMDx->TRGSYNCR = cTRGSYNCR_INI;								/* TRG update at PWM end */ 
}
/**
 *********************************************************************************************
 * @brief		IP_PMD_Timer_Start
 *
 * @param		TSB_PMD_TypeDef* const	PMDx		Select the PMD channel.
 * @return		none
 *********************************************************************************************
 */
void IP_PMD_Timer_Start(TSB_PMD_TypeDef* const PMDx)
{
	PMDx->MDEN = 0x01;											/* PMD enable (Set after set MDOUT,MDPOT) */
}
/**
 *********************************************************************************************
 * @brief		IP_ADC_TTRG_ENST_REG9
 *
 * @param		TSB_AD_TypeDef* const ADx		Select the AD channel. 
 * @param		 uint32_t	_dat
 * @return		none
 *********************************************************************************************
 */
void IP_ADC_TTRG_ENST_REG9(TSB_AD_TypeDef* const ADx, uint32_t	_dat)
{
	if(_dat == SET)
	{
		ADx->TSET9 |= (3U << 5);
	}
	else
	{
		ADx->TSET9 &= ~(3U << 5);
	}
}

void IP_ADC_TTRG_ENST_REG10(TSB_AD_TypeDef* const ADx, uint32_t	_dat)
{
	if(_dat == SET)
	{
		ADx->TSET10 |= (1U << 7);
	}
	else
	{
		ADx->TSET10 &= ~(1U << 7);
	}
}

void IP_ADC_TTRG_ENST_REG11(TSB_AD_TypeDef* const ADx, uint32_t	_dat)
{
	if(_dat == SET)
	{
		ADx->TSET11 |= (3U << 5);
	}
	else
	{
		ADx->TSET11 &= ~(3U << 5);
	}
}

/**
 *********************************************************************************************
 * @brief		IP_VE_init
 *
 * @param		TSB_VE_TypeDef* const	VEx			Select the VE channel.
 * @param		VE_InitTypeDef* const	_initdata	The structure containing basic VE configuration
 *
 * @return		none
 *********************************************************************************************
 */
void IP_VE_init(TSB_VE_TypeDef* const VEx, VE_InitTypeDef* const _initdata)
{
	CG_SetFsysSupplyPeriphB(CG_FSYS_PERIPH_A_VE_P, ENABLE);

	/*********************** VE Control Register ********************************/
	VEx->EN = 0x01; 											/* VE Enable */

	VEx->TASKAPP = SET_TAPP(TAPP_OUT_CON);						/* VE start from output control */
	VEx->ACTSCH = SET_ACTSCH(SCH_9); 							/* Schedule 9 */

	VEx->REPTIME = SET_REPTIME(_initdata->reptime);
	VEx->TRGMODE = SET_TRGMODE(_initdata->trgmode);

	VEx->ERRINTEN = SET_ERRINTEN(cVERREN_INI);

	VEx->TMPREG0 = 0;											/* Temporary register clear */
	VEx->TMPREG1 = 0;
	VEx->TMPREG2 = 0;
	VEx->TMPREG3 = 0;
	VEx->TMPREG4 = 0;
	VEx->TMPREG5 = 0;

	VEx->TADC = (uint16_t)(cTADC * cIMCLK_FRQ * 1e6);			/* for 1shunt shift pwm.  (AD convert time) */

	/*********************** VE Register for Channel 1 **************************/
	VEx->MODE = VEMODE_STOP;

	if (_initdata->shunt == c3shunt)
	{
		VEx->FMODE = cFMODE_INI_3SHUNT;
	}
	else
	{
		VEx->FMODE = cFMODE_INI_1SHUNT;
	}

	VEx->TPWM = _initdata->tpwm;

	VEx->OMEGA = 0;
	VEx->THETA = 0;

	VEx->COS = 0;
	VEx->SIN = 0;
	VEx->COSM = 0;
	VEx->SINM = 0;

	VEx->IDREF = 0;
	VEx->IQREF = 0;

	VEx->VD = 0;
	VEx->VQ = 0;

	VEx->CIDKI = _initdata->idki;
	VEx->CIDKP = _initdata->idkp;
	VEx->CIDKG = 0;
	VEx->CIQKI = _initdata->iqki;
	VEx->CIQKP = _initdata->iqkp;
	VEx->CIQKG = 0;

	VEx->VDIH = 0;
	VEx->VDILH = 0;
	VEx->VQIH = 0;
	VEx->VQILH = 0;

	VEx->MCTLF = 0x00;

	VEx->FPWMCHG = 0x7fff;										/* Hard control off */

	VEx->MINPLS = 0;

	VEx->SECTOR = 0;
	VEx->SECTORM = 0;

	VEx->IAO = _initdata->zero_offset;
	VEx->IBO = _initdata->zero_offset;
	VEx->ICO = _initdata->zero_offset;

	VEx->IAADC = 0;
	VEx->IBADC = 0;
	VEx->ICADC = 0;

	VEx->VDC = 0;
	VEx->ID = 0;
	VEx->IQ = 0;

	VEx->CMPU = 0;
	VEx->CMPV = 0;
	VEx->CMPW = 0;

	VEx->OUTCR = 0x0000;										/* Output UP/DOWN = OFF/OFF */

	VEx->TRGCRC = 0;

	VEx->TRGCMP0 = 0;
	VEx->TRGCMP1 = 0;

	VEx->TRGSEL = 0;

	VEx->EMGRS = 0;

	VEx->CPURUNTRG = VE_START;
}

/**
 *********************************************************************************************
 * @brief		ADC Initial set
 *
 * @param		TSB_AD_TypeDef* const	ADx			Select the ADC channel.
 * @param		AD_InitTypeDef* const	_initdata	The structure containing basic ADC configuration
 *
 * @return		none
 *********************************************************************************************
 */
void IP_ADC_init(TSB_AD_TypeDef* const ADx, AD_InitTypeDef* const _initdata)
{
	uint32_t data32;
	uint32_t data33;
	/***** PMD TRG Start mode set *****/
	if(_initdata->shunt != cPFCshunt)
	{
		CG_SetFcSupplyPeriph(CG_FC_PERIPH_DNF_A, ENABLE);
		CG_SetFcSupplyPeriph(CG_FC_PERIPH_DNF_B, ENABLE);
		CG_SetFcSupplyPeriph(CG_FC_PERIPH_DNF_C, ENABLE);
		CG_SetAdcClkSupply(TSB_ADA, ENABLE);
		CG_SetAdcClkSupply(TSB_ADB, ENABLE);
		CG_SetAdcClkSupply(TSB_ADC, ENABLE);
		CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_PORTL, ENABLE);	/* Unit A */
		CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_PORTM, ENABLE);	/* Unit A */
		CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_PORTK, ENABLE);	/* Unit B */
		CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_PORTJ, ENABLE);	/* Unit C */
		CG_SetFsysSupplyPeriphB(CG_FSYS_PERIPH_ADC_A, ENABLE);
		CG_SetFsysSupplyPeriphB(CG_FSYS_PERIPH_ADC_B, ENABLE);
		CG_SetFsysSupplyPeriphB(CG_FSYS_PERIPH_ADC_C, ENABLE);
	}

	if (_initdata->shunt == c3shunt)
	{
		/* ADx Program set */
		if (_initdata->pmd_ch == cPMD0)
		{
			ADx->PSEL0  = TRG_ENABLE(PMD_PROG0);		/* PMD0 TRG0:BProg0 at Sector0	3-shunt */
			ADx->PSEL1  = TRG_ENABLE(PMD_PROG1);		/* PMD0 TRG1:BProg1 at Sector1	3-shunt */
			ADx->PSEL2  = TRG_ENABLE(PMD_PROG2);		/* PMD0 TRG2:BProg2 at Sector2	3-shunt */
			ADx->PSEL3  = TRG_ENABLE(PMD_PROG3);		/* PMD0 TRG3:BProg3 at Sector3	3-shunt */
			ADx->PSEL4  = TRG_ENABLE(PMD_PROG4);		/* PMD0 TRG4:BProg4 at Sector4	3-shunt */
			ADx->PSEL5  = TRG_ENABLE(PMD_PROG5);		/* PMD0 TRG5:BProg5 at Sector5	3-shunt */
		}
		else if (_initdata->pmd_ch == cPMD1)
		{
			ADx->PSEL6  = TRG_ENABLE(PMD_PROG0);		/* PMD1 TRG0:BProg0 at Sector0	3-shunt */
			ADx->PSEL7  = TRG_ENABLE(PMD_PROG1);		/* PMD1 TRG1:BProg1 at Sector1	3-shunt */
			ADx->PSEL8  = TRG_ENABLE(PMD_PROG2);		/* PMD1 TRG2:BProg2 at Sector2	3-shunt */
			ADx->PSEL9  = TRG_ENABLE(PMD_PROG3);		/* PMD1 TRG3:BProg3 at Sector3	3-shunt */
			ADx->PSEL10 = TRG_ENABLE(PMD_PROG4);		/* PMD1 TRG4:BProg4 at Sector4	3-shunt */
			ADx->PSEL11 = TRG_ENABLE(PMD_PROG5);		/* PMD1 TRG5:BProg5 at Sector5	3-shunt */
		}
		
		if (_initdata->pmd_ch == cPMD0)
		{
			/* Getting AD convert registers address. */
			*_initdata->p_adreg0 = (uint32_t *)&ADx->REG0 + cPMD0_ADREG_OFFSET;
			*_initdata->p_adreg1 = (uint32_t *)&ADx->REG1 + cPMD0_ADREG_OFFSET;
			*_initdata->p_adreg2 = (uint32_t *)&ADx->REG2 + cPMD0_ADREG_OFFSET;
			*_initdata->p_adreg3 = (uint32_t *)&ADx->REG3 + cPMD0_ADREG_OFFSET;
			/* Setting AD convert registers to PRG0-5. */
			ADx->PREGS = cPMD0_PREGS_3SHUNT;
		}
		else if (_initdata->pmd_ch == cPMD1)
		{
			/* Getting AD convert registers address. */
			*_initdata->p_adreg0 = (uint32_t *)&ADx->REG0 + cPMD1_ADREG_OFFSET;
			*_initdata->p_adreg1 = (uint32_t *)&ADx->REG1 + cPMD1_ADREG_OFFSET;
			*_initdata->p_adreg2 = (uint32_t *)&ADx->REG2 + cPMD1_ADREG_OFFSET;
			*_initdata->p_adreg3 = (uint32_t *)&ADx->REG3 + cPMD1_ADREG_OFFSET;
			/* Setting AD convert registers to PRG0-5. */
			ADx->PREGS = cPMD1_PREGS_3SHUNT;
		}

		if (_initdata->pints == cPINTS_A)
		{
			ADx->PINTS0 = PMD_INTADPDA;					/* End of all prog generate INTADA */
			ADx->PINTS1 = PMD_INTADPDA;
			ADx->PINTS2 = PMD_INTADPDA;
			ADx->PINTS3 = PMD_INTADPDA;
			ADx->PINTS4 = PMD_INTADPDA;
			ADx->PINTS5 = PMD_INTADPDA;
			ADx->PINTS6 = PMD_INTNONE;
			ADx->PINTS7 = PMD_INTNONE;
		}
		else if (_initdata->pints == cPINTS_B)
		{
			ADx->PINTS0 = PMD_INTADPDB;					/* End of all prog generate INTADB */
			ADx->PINTS1 = PMD_INTADPDB;
			ADx->PINTS2 = PMD_INTADPDB;
			ADx->PINTS3 = PMD_INTADPDB;
			ADx->PINTS4 = PMD_INTADPDB;
			ADx->PINTS5 = PMD_INTADPDB;
			ADx->PINTS6 = PMD_INTNONE;
			ADx->PINTS7 = PMD_INTNONE;
		}
		else if (_initdata->pints == cPINTS_C)
		{
			ADx->PINTS0 = PMD_INTADPDC;					/* End of all prog generate INTADC */
			ADx->PINTS1 = PMD_INTADPDC;
			ADx->PINTS2 = PMD_INTADPDC;
			ADx->PINTS3 = PMD_INTADPDC;
			ADx->PINTS4 = PMD_INTADPDC;
			ADx->PINTS5 = PMD_INTADPDC;
			ADx->PINTS6 = PMD_INTNONE;
			ADx->PINTS7 = PMD_INTNONE;
		}
		else
		{
			ADx->PINTS0 = PMD_INTADPDD;					/* End of all prog generate INTADD */
			ADx->PINTS1 = PMD_INTADPDD;
			ADx->PINTS2 = PMD_INTADPDD;
			ADx->PINTS3 = PMD_INTADPDD;
			ADx->PINTS4 = PMD_INTADPDD;
			ADx->PINTS5 = PMD_INTADPDD;
			ADx->PINTS6 = PMD_INTNONE;
			ADx->PINTS7 = PMD_INTNONE;
		}

		/* ADx->PSET5,PSET0	REG0:V, REG1:W, REG2:U, REG3:Vdc */
		data32 = (((0x80 | (VE_PHASE_V    << 5U) | _initdata->ivch)) \
		        | ((0x80 | (VE_PHASE_W    << 5U) | _initdata->iwch)  << 8U) \
		        | ((0x80 | (VE_PHASE_U    << 5U) | _initdata->iuch)  << 16U) \
		        | ((0x80 | (VE_PHASE_NONE << 5U) | _initdata->vdcch) << 24U));
		ADx->PSET5 = data32;
		ADx->PSET0 = data32;

		/* ADx->PSET1,PSET2	REG0:W, REG1:U, REG2:V, REG3:Vdc */
		data32 = (((0x80 | (VE_PHASE_W    << 5U) | _initdata->iwch)) \
		        | ((0x80 | (VE_PHASE_U    << 5U) | _initdata->iuch)  << 8U) \
		        | ((0x80 | (VE_PHASE_V    << 5U) | _initdata->ivch)  << 16U) \
		        | ((0x80 | (VE_PHASE_NONE << 5U) | _initdata->vdcch) << 24U));
		ADx->PSET1 = data32;
		ADx->PSET2 = data32;

		/* ADx->PSET3,PSET4	REG0:U, REG1:V, REG2:W, REG3:Vdc */
		data32 = (((0x80 | (VE_PHASE_U    << 5U) | _initdata->iuch)) \
		        | ((0x80 | (VE_PHASE_V    << 5U) | _initdata->ivch)  << 8U) \
		        | ((0x80 | (VE_PHASE_W    << 5U) | _initdata->iwch)  << 16U) \
		        | ((0x80 | (VE_PHASE_NONE << 5U) | _initdata->vdcch) << 24U));
		ADx->PSET3	= data32;
		ADx->PSET4	= data32;

		ADx->PSET6	= 0;
		ADx->PSET7	= 0;

	}
	else if (_initdata->shunt == c1shunt)
	{
		/* ADx Program set */
		if (_initdata->pmd_ch == cPMD0)
		{
			ADx->PSEL0  = TRG_ENABLE(PMD_PROG0);		/* PMD0 TRG0:BProg0 1-shunt */
			ADx->PSEL1  = TRG_ENABLE(PMD_PROG1);		/* PMD0 TRG1:BProg1 1-shunt */
			ADx->PSEL2  = TRG_ENABLE(PMD_PROG2);		/* PMD0 TRG2:BProg2 */
			ADx->PSEL3  = TRG_DISABLE(PMD_PROG0);		/* PMD0 TRG3:BProg3 */
			ADx->PSEL4  = TRG_DISABLE(PMD_PROG0);		/* PMD0 TRG4:BProg4 */
			ADx->PSEL5  = TRG_DISABLE(PMD_PROG0);		/* PMD0 TRG5:BProg5 */
		}
		else if (_initdata->pmd_ch == cPMD1)
		{
			ADx->PSEL6  = TRG_ENABLE(PMD_PROG0);		/* PMD1 TRG0:BProg0 1-shunt */
			ADx->PSEL7  = TRG_ENABLE(PMD_PROG1);		/* PMD1 TRG1:BProg1 1-shunt */
			ADx->PSEL8  = TRG_ENABLE(PMD_PROG2);		/* PMD1 TRG2:BProg2 */
			ADx->PSEL9  = TRG_DISABLE(PMD_PROG0);		/* PMD1 TRG3:BProg3 */
			ADx->PSEL10 = TRG_DISABLE(PMD_PROG0);		/* PMD1 TRG4:BProg4 */
			ADx->PSEL11 = TRG_DISABLE(PMD_PROG0);		/* PMD1 TRG5:BProg5 */
		}

		if (_initdata->pmd_ch == cPMD0)
		{
			/* Getting AD convert registers address. */
			*_initdata->p_adreg0 = (uint32_t *)&ADx->REG0 + cPMD0_ADREG_OFFSET;
			*_initdata->p_adreg1 = (uint32_t *)&ADx->REG1 + cPMD0_ADREG_OFFSET;
			*_initdata->p_adreg2 = (uint32_t *)&ADx->REG2 + cPMD0_ADREG_OFFSET;
			*_initdata->p_adreg3 = (uint32_t *)&ADx->REG3 + cPMD0_ADREG_OFFSET;
			/* Setting AD convert registers to PRG0-2. */
			ADx->PREGS = cPMD0_PREGS_1SHUNT;
		}
		else if (_initdata->pmd_ch == cPMD1)
		{
			/* Getting AD convert registers address. */
			*_initdata->p_adreg0 = (uint32_t *)&ADx->REG0 + cPMD1_ADREG_OFFSET;
			*_initdata->p_adreg1 = (uint32_t *)&ADx->REG1 + cPMD1_ADREG_OFFSET;
			*_initdata->p_adreg2 = (uint32_t *)&ADx->REG2 + cPMD1_ADREG_OFFSET;
			*_initdata->p_adreg3 = (uint32_t *)&ADx->REG3 + cPMD1_ADREG_OFFSET;
			/* Setting AD convert registers to PRG0-2. */
			ADx->PREGS = cPMD1_PREGS_1SHUNT;
		}

		if (_initdata->pints == cPINTS_A)
		{
			ADx->PINTS0 = PMD_INTNONE; 					/* End of all prog generate INTADA */
			ADx->PINTS1 = PMD_INTNONE;
			ADx->PINTS2 = PMD_INTADPDA;
			ADx->PINTS3 = PMD_INTNONE;
			ADx->PINTS4 = PMD_INTNONE;
			ADx->PINTS5 = PMD_INTNONE;
			ADx->PINTS6 = PMD_INTNONE;
			ADx->PINTS7 = PMD_INTNONE;
		}
		else if (_initdata->pints == cPINTS_B)
		{
			ADx->PINTS0 = PMD_INTNONE; 					/* End of all prog generate INTADB */
			ADx->PINTS1 = PMD_INTNONE;
			ADx->PINTS2 = PMD_INTADPDB;
			ADx->PINTS3 = PMD_INTNONE;
			ADx->PINTS4 = PMD_INTNONE;
			ADx->PINTS5 = PMD_INTNONE;
			ADx->PINTS6 = PMD_INTNONE;
			ADx->PINTS7 = PMD_INTNONE;
		}
		else if (_initdata->pints == cPINTS_C)
		{
			ADx->PINTS0 = PMD_INTNONE; 					/* End of all prog generate INTADC */
			ADx->PINTS1 = PMD_INTNONE;
			ADx->PINTS2 = PMD_INTADPDC;
			ADx->PINTS3 = PMD_INTNONE;
			ADx->PINTS4 = PMD_INTNONE;
			ADx->PINTS5 = PMD_INTNONE;
			ADx->PINTS6 = PMD_INTNONE;
			ADx->PINTS7 = PMD_INTNONE;
		}
		else
		{
			ADx->PINTS0 = PMD_INTNONE; 					/* End of all prog generate INTADD */
			ADx->PINTS1 = PMD_INTNONE;
			ADx->PINTS2 = PMD_INTADPDD;
			ADx->PINTS3 = PMD_INTNONE;
			ADx->PINTS4 = PMD_INTNONE;
			ADx->PINTS5 = PMD_INTNONE;
			ADx->PINTS6 = PMD_INTNONE;
			ADx->PINTS7 = PMD_INTNONE;
		}

		if (_initdata->pmd_ch == cPMD0)
		{
			/*ADx->PSET0	REG0:Idc----, REG1:-------, REG2:-------, REG3:------- */
			data32 = (0x80 | _initdata->idcch);
			ADx->PSET0 = data32;

			/*ADx->PSET1	REG0:-------, REG1:Idc----, REG2:-------, REG3:------- */
			data32 = (((0x80 | _initdata->idcch) << 8U));
			ADx->PSET1 = data32;

			/*ADx->PSET2	REG0:-------, REG1:-------, REG2:-------, REG3:Vdc---- */
			data32 = ((0x80 | _initdata->vdcch) << 24U);
			ADx->PSET2 = data32;
		}
		else if (_initdata->pmd_ch == cPMD1)
		{
			/*ADx->PSET0	REG0:Idc----, REG1:-------, REG2:-------, REG3:------- */
			data32 = (0x80 | _initdata->idcch);
			ADx->PSET0 = data32;

			/*ADx->PSET1	REG0:-------, REG1:Idc----, REG2:-------, REG3:------- */
			data32 = (((0x80 | _initdata->idcch) << 8U));
			ADx->PSET1 = data32;

			/*ADx->PSET2	REG0:-------, REG1:-------, REG2:-------, REG3:Vdc---- */
			data32 = ((0x80 | _initdata->vdcch) << 24U);
			ADx->PSET2 = data32;
		}
		ADx->PSET3 = 0;
		ADx->PSET4 = 0;
		ADx->PSET5 = 0;
		ADx->PSET6 = 0;
		ADx->PSET7 = 0;
	}
	else if (_initdata->shunt == cPFCshunt)
	{
		if (_initdata->pmd_ch == cPMD2)
		{
			if (_initdata->pints == cPINTS_C)
			{
				ADx->PREGS = cPMD2_PREGS_PFCSHUNT;

				ADx->PSEL3  = TRG_ENABLE(PMD_PROG2);		/* PMD0 TRG3:Prog2 1-shunt */

				ADx->PSEL6  = TRG_ENABLE(PMD_PROG0);		/* PMD1 TRG0:BProg0 1-shunt */
				ADx->PSEL7  = TRG_ENABLE(PMD_PROG1);		/* PMD1 TRG1:BProg1 1-shunt */
				ADx->PSEL8  = TRG_DISABLE(PMD_PROG0);		/* PMD1 TRG2:BProg2 */
				ADx->PSEL9  = TRG_DISABLE(PMD_PROG0);		/* PMD1 TRG3:BProg3 */
				ADx->PSEL10 = TRG_DISABLE(PMD_PROG0);		/* PMD1 TRG4:BProg4 */
				ADx->PSEL11 = TRG_DISABLE(PMD_PROG0);		/* PMD1 TRG5:BProg5 */

				ADx->PINTS0 = PMD_INTADPDB; 					/* End of all prog generate INTADC */
				ADx->PINTS1 = PMD_INTNONE;
				ADx->PINTS2 = PMD_INTNONE;
				ADx->PINTS3 = PMD_INTNONE;
				ADx->PINTS4 = PMD_INTNONE;
				ADx->PINTS5 = PMD_INTNONE;
				ADx->PINTS6 = PMD_INTNONE;
				ADx->PINTS7 = PMD_INTNONE;
			}
				/*ADx->PSET0	REG0:Idc----, REG1:-------, REG2:-------, REG3:------- */
			data32 = (((0x80  | _initdata->idcch)) \
		        | ((0x80 | _initdata->idcch)  << 8U) \
		        | ((0x80 | _initdata->idcch)  << 16U) \
		        | ((0x00 | _initdata->vdcch) << 24U));
			ADx->PSET0 = data32;

			data33 = (((0x80  | _initdata->vdcch)) \
		        | ((0x00 | _initdata->vdcch)  << 8U) \
		        | ((0x00 | _initdata->vdcch)  << 16U) \
		        | ((0x00 | _initdata->vdcch) << 24U));

			ADx->PSET2 = data33;  /* vac 1 sample */
		}
	}
}

/**
 *********************************************************************************************
 * @brief		AMP IP Initialize
 *
 * @return		none
 *********************************************************************************************
 */
void IP_AMP_init(void)
{
#if defined(__AMP)
	CG_SetFsysSupplyPeriphB(CG_FSYS_PERIPH_AMP, ENABLE);
	/* Setting internal amp */
	#if defined(__USE_INAMP_FAN) 
	TSB_AMP->CTLA = AMP_ENABLE(AMPGAIN_FOLD_3_5);
	#endif /* __USE_INAMP */
	#if defined(__USE_INAMP_COMP)
	TSB_AMP->CTLB = AMP_ENABLE(AMPGAIN_FOLD_12_0);
	#endif /* __USE_INAMP */
	TSB_AMP->CTLC = AMP_ENABLE(AMPGAIN_FOLD_3_5);
#endif
}

/**
 ********************************************************************************************
 * @brief		VE_GET_Cal_Vdq
 *
 * @return		none
 ********************************************************************************************
 */
q15_t VE_GET_Cal_Vdq(q15_t vd, q15_t vq)
{
	q31_t w_work1;
	w_work1 = C_E_Sqrt ( ( ( ( vd * vd ) << 1 ) + ( ( vq * vq ) << 1 ) ) );
	return ( C_muljc ( (w_work1 >> 16), ( 256 * 128 * 1732 / 1000 ) ) ) >> 16;

}

/*********************************** END OF FILE ******************************/
