/**
 ****************************************************************************
 * @file     dac_drv.c
 * @brief    DAC IC Control for AD5624 on SIO.
 * @version  V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2021
 *****************************************************************************
 */
#include "ipdefine.h"
#include "mcuip_drv.h"

#define DEFINE_APP
#include "dac_drv.h"
#undef	DEFINE_APP

/**
 * @note	DAC IC Control for AD5624.
 *
 */

/*===================================================================*
  Device local functions prototype.
 *===================================================================*/
static void DacLocalWriteInit(TSB_SC_TypeDef* const SCx);

/*===================================================================*
	  Local Variable Definition
 *===================================================================*/
uint32_t DacNo = DAC_MAX_CH;


/*===================================================================*
	  Const Definition
 *===================================================================*/
/* Command Definition */
#define	DAC_CMD_W_INPUT					0U		/* Write to input register n */
#define	DAC_CMD_UP_DAC					1U		/* Update DAC register n */
#define	DAC_CMD_W_INPUT_UP_ALL			2U		/* Write to input register n, update all(software LDAC) */
#define	DAC_CMD_W_UP_DAC				3U		/* Write to and update DAC channel n */
#define	DAC_CMD_POWER_DOWN				4U		/* Power down DAC (power-up) */
#define	DAC_CMD_RESET					5U		/* Reset */
#define	DAC_CMD_LOAD_LDAC				6U		/* Load LDAC register */

#define	DAC_RESET_MODE					1U		/* Reset mode :  0:DAC and Input shift register */
												/*            :  1:DAC and Input shift and LDAC and Power-down register */


#define	cDAC_POWERRESET					((DAC_CMD_RESET << 19U) | (DAC_RESET_MODE << 4U))
#define	cDAC_LDAC_NORMAL				((DAC_CMD_LOAD_LDAC << 19U) | 0)

/**
 *********************************************************************************************
 * @brief		D/A initialize.
 *
 * @param		TSB_SC_TypeDef * const	SCx		Select the sio channel.
 *
 * @return		none
 *********************************************************************************************
 */
void init_Dac(TSB_SC_TypeDef* const SCx)
{
	SIO_InitTypeDef SIO_DacInit;

	SIO_Enable(SCx);

	/*initialize the SIO struct */
	SIO_DacInit.InputClkEdge = SIO_SCLKS_TXDF_RXDR;
//	SIO_DacInit.InputClkEdge = SIO_SCLKS_TXDR_RXDF;
	SIO_DacInit.IntervalTime = SIO_SINT_TIME_NONE;
	SIO_DacInit.TransferMode = SIO_TRANSFER_HALFDPX_TX;
	SIO_DacInit.TransferDir = SIO_MSB_FRIST;
	SIO_DacInit.Mode = SIO_ENABLE_TX;
	SIO_DacInit.DoubleBuffer = SIO_WBUF_DISABLE;
	SIO_DacInit.BaudRateClock = SIO_BR_CLOCK_T1;
	SIO_DacInit.Divider = SIO_BR_DIVIDER_3;

	SIO_Init(SCx, SIO_CLK_BAUDRATE, &SIO_DacInit);

	/* Setting SIO port. */
	if (SCx == TSB_SC0)
	{
		TSB_PE->FR1 |= ((1 << 2u) | (1 << 0u));
		TSB_PE->CR |= ((1 << 2u) | (1 << 0u));

	}
#if defined(__SC_CH1)
	else if (SCx == TSB_SC1)
	{
		TSB_PA->FR1 |= ((1 << 4u) | (1 << 5u));
		TSB_PA->CR |= ((1 << 4u) | (1 << 5u));
	}
#endif
#if defined(__SC_CH2)
	else if (SCx == TSB_SC2)
	{
		TSB_PD->FR1 |= ((1 << 4u) | (1 << 5u));
		TSB_PD->CR |= ((1 << 4u) | (1 << 5u));
	}
#endif
#if defined(__SC_CH3)
	else if (SCx == TSB_SC3)
	{
		TSB_PF->FR2 |= ((1 << 2u) | (1 << 3u));
		TSB_PF->CR |= ((1 << 2u) | (1 << 3u));
	}
#endif

	/* Setting ~SYNC port */
	PORT_DAC_SYNC_DATA_INI;
	PORT_DAC_SYNC_CR_INI;

	DacLocalWriteInit(SCx);

	NVIC_SetPriority(DAC_IRQ_INTTX, DAC_IRQ_LEVEL); 		/* Interruption level set */
	NVIC_ClearPendingIRQ(DAC_IRQ_INTTX);
	NVIC_EnableIRQ(DAC_IRQ_INTTX);							/* INTTX enable */

}


/**
 *********************************************************************************************
 * @brief		D/A initial output.
 *
 * @param		TSB_SC_TypeDef * const	SCx		Select the sio channel.
 *
 * @return		none
 *********************************************************************************************
 */
static void DacLocalWriteInit(TSB_SC_TypeDef* const SCx)
{
	PORT_DAC_SYNC = 0;										/* begin tx */

	NVIC_ClearPendingIRQ(DAC_IRQ_INTTX);
	SCx->BUF = (uint8_t)((cDAC_POWERRESET >> 16U) & 0x000000ff);
	while (!NVIC_GetPendingIRQ(DAC_IRQ_INTTX));

	NVIC_ClearPendingIRQ(DAC_IRQ_INTTX);
	SCx->BUF = (uint8_t)((cDAC_POWERRESET >> 8U) & 0x000000ff);
	while (!NVIC_GetPendingIRQ(DAC_IRQ_INTTX));

	NVIC_ClearPendingIRQ(DAC_IRQ_INTTX);
	SCx->BUF = (uint8_t)(cDAC_POWERRESET & 0x000000ff);
	while (!NVIC_GetPendingIRQ(DAC_IRQ_INTTX));

	PORT_DAC_SYNC = 1;										/* end tx */

	PORT_DAC_SYNC = 0;										/* begin tx */

	NVIC_ClearPendingIRQ(DAC_IRQ_INTTX);
	SCx->BUF = (uint8_t)((cDAC_LDAC_NORMAL >> 16U) & 0x000000ff);
	while (!NVIC_GetPendingIRQ(DAC_IRQ_INTTX));

	NVIC_ClearPendingIRQ(DAC_IRQ_INTTX);
	SCx->BUF = (uint8_t)((cDAC_LDAC_NORMAL >> 8U) & 0x000000ff);
	while (!NVIC_GetPendingIRQ(DAC_IRQ_INTTX));

	NVIC_ClearPendingIRQ(DAC_IRQ_INTTX);
	SCx->BUF = (uint8_t)(cDAC_LDAC_NORMAL & 0x000000ff);
	while (!NVIC_GetPendingIRQ(DAC_IRQ_INTTX));

	PORT_DAC_SYNC = 1;										/* end tx */
}


/**
 *********************************************************************************************
 * @brief		D/A data output start.
 *
 * @param		TSB_SC_TypeDef * const	SCx		Select the sio channel.
 *
 * @return		none
 *********************************************************************************************
 */
void DacWriteStart(TSB_SC_TypeDef* const SCx)
{
	if (DacNo >= DAC_MAX_CH)
	{
		DacNo = 0;
		DacWriteContinue(SCx);
	}
}


/**
 *********************************************************************************************
 * @brief		D/A data output continue.
 *
 * @param		TSB_SC_TypeDef * const	SCx		Select the sio channel.
 *
 * @return		none
 *********************************************************************************************
 */
void DacWriteContinue(TSB_SC_TypeDef* const SCx)
{
	static uint16_t data = 0;
	static uint16_t seq = 0;

	if (DacNo < DAC_MAX_CH)
	{
		switch (seq)
		{
			case 0:
				data = DacData[DacNo];
				PORT_DAC_SYNC = 1;							/* end tx */
				PORT_DAC_SYNC = 0;							/* begin tx */
				if (DacNo != (DAC_MAX_CH - 1))
				{
					SCx->BUF = (DAC_CMD_W_INPUT << 3U) | DacNo;
				}
				else
				{
					SCx->BUF = (DAC_CMD_W_INPUT_UP_ALL << 3U) | DacNo;
				}
				seq = 1;
				break;
			case 1:
				SCx->BUF = (data >> 8);
				seq = 2;
				break;
			case 2:
				SCx->BUF = data;
				seq = 0;
				++DacNo;
				break;
			default:
				break;
		}
	}
	else
	{
		PORT_DAC_SYNC = 1;									/* end tx */
	}
}

/*********************************** END OF FILE ******************************/
