/**
 ****************************************************************************
 * @file     ipdefine.h
 * @brief    Application Definitions
 * @version  V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2021
 *****************************************************************************
 */
#ifndef _IPDEFINE_H
#define _IPDEFINE_H

#include <arm_math.h>

/* Select MCU */
#if defined(__MCU_M370)
#include "TMPM370.h"
#elif defined(__MCU_M376)
#include "TMPM376.h"
#elif defined(__MCU_M372)
#include "TMPM372.h"
#elif defined(__MCU_M373)
#include "TMPM373.h"
#elif defined(__MCU_M374)
#include "TMPM374.h"
#elif defined(__MCU_M470)
#include "TMPM470.h"
#elif defined(__MCU_M475)
#include "TMPM475.h"
#elif defined(__MCU_M375)
#include "TMPM375.h"
#elif defined(__MCU_M37A)
#include "TMPM37A.h"
#elif defined(__MCU_M4K4)
#include "TMPM4KxA.h"
#elif defined(__MCU_M4KN)
#include "TMPM4Ky.h"
#elif defined(__MCU_M4L2)
#include "TMPM4Lx.h"
#elif defined(__MCU_M4J5)
#include "TMPM4Jx.h"
#endif

#include "sys_macro.h"
#include "D_Para.h"

#include "CheckVdq.h"

#define	__MOTOR_DEBUG						/* Motor debug */
//#define	__MOTOR_DBGOUT_VECTOR_TIME_INT		/* Debug output for vector control process time */
//#define	__MOTOR_DBGOUT_IDET					/* Debug output for current detection status */

#ifdef DEFINE_APP
#define EXTERN
#else
#define EXTERN	extern
#endif


/*===================================================================*
	  MCU Parameter Definition
 *===================================================================*/
/* Setting fc */
#if (defined(__TMPM4J5_H__))
#define	cIMCLK_FRQ					(160)						/* [MHz] */
#elif (defined(__TMPM470_H__) || defined(__TMPM475_H__))
#define	cIMCLK_FRQ					(120)						/* [MHz] */
#elif (defined(__TMPM370_H__) || defined(__TMPM372_H__) || defined(__TMPM373_H__) || defined(__TMPM374_H__) || defined(__TMPM376_H__) || defined(__TMPM4K4A_H__) || defined(__TMPM4L2_H__) || defined(__TMPM4KN_H__) )
#define	cIMCLK_FRQ					(80)						/* [MHz] */
#elif (defined(__TMPM375_H__) || defined(__TMPM37A_H__))
#define	cIMCLK_FRQ					(40)						/* [MHz] */
#endif

/* Setting INTVC name */
#if (defined(__TMPM370_H__) || defined(__TMPM372_H__) || defined(__TMPM373_H__) || defined(__TMPM374_H__) || defined(__TMPM376_H__) || defined(__TMPM375_H__) || defined(__TMPM37A_H__))
#define	INTVCN0_IRQn			INTVCNA_IRQn
#define	INTVCN1_IRQn			INTVCNB_IRQn
#define	INTVCN0_IRQHandler		INTVCNA_IRQHandler
#define	INTVCN1_IRQHandler		INTVCNB_IRQHandler
#endif

/* Setting PMD type */
#if (defined(__TMPM370_H__) || defined(__TMPM372_H__) || defined(__TMPM373_H__) || defined(__TMPM374_H__) || defined(__TMPM376_H__) || defined(__TMPM375_H__) || defined(__TMPM37A_H__) || defined(__TMPM470_H__) || defined(__TMPM475_H__))
#define	__PMD
#elif (defined(__TMPM4L2_H__) || defined(__TMPM4J5_H__))
#define __P_PMD
#elif (defined(__TMPM4K4A_H__) || defined(__TMPM4KN_H__))
#define	__A_PMD
#endif

/* Setting VE type */
#if (defined(__TMPM370_H__) || defined(__TMPM372_H__) || defined(__TMPM373_H__) || defined(__TMPM374_H__) || defined(__TMPM376_H__))
#define	__VE		/* VE */
#elif (defined(__TMPM375_H__) || defined(__TMPM37A_H__))
#define	__VE_P		/* VE Plus */
#elif (defined(__TMPM470_H__) || defined(__TMPM475_H__) || defined(__TMPM4L2_H__) || defined(__TMPM4J5_H__))
#define	__A_VE		/* Advanced VE */
#elif (defined(__TMPM4K4A_H__) || defined(__TMPM4KN_H__))
#define	__A_VE_P	/* Advanced VE Plus */
#endif

/* Setting ADC type */
#if (defined(__TMPM370_H__) || defined(__TMPM372_H__) || defined(__TMPM373_H__) || defined(__TMPM374_H__) || defined(__TMPM376_H__) || defined(__TMPM375_H__) || defined(__TMPM37A_H__))
#define	__ADC
#elif (defined(__TMPM470_H__) || defined(__TMPM475_H__))
#define	__ADC2
#elif (defined(__TMPM4K4A_H__) || defined(__TMPM4KN_H__) || defined(__TMPM4L2_H__) || defined(__TMPM4J5_H__))
#define	__ADCZ
#endif


/* Setting ENC type */
#if (defined(__TMPM370_H__) || defined(__TMPM372_H__) || defined(__TMPM373_H__) || defined(__TMPM374_H__) || defined(__TMPM376_H__) || defined(__TMPM375_H__))
#define	__ENC
#elif (defined(__TMPMKN_H__) || defined(__TMPM4L2_H__) || defined(__TMPM4J5_H__))
#define __A_ENC_32
#elif (defined(__TMPM470_H__) || defined(__TMPM475_H__) || defined(__TMPM4K4A_H__))
#define	__A_ENC
#endif

/* Setting AMP type */
#if (defined(__TMPM370_H__) || defined(__TMPM375_H__) || defined(__TMPM37A_H__) || defined(__TMPM4K4A_H__) || defined(__TMPM4KN_H__))
#define	__AMP
#endif

/* Setting AMP Unit */
#if defined(__TMPM370_H__)
#define	__AMP_UNIT_A
#define	__AMP_UNIT_B
#define	__AMP_UNIT_C
#define	__AMP_UNIT_D
#elif (defined(__TMPM375_H__) || defined(__TMPM37A_H__))
#define	__AMP_UNIT_D
#elif (defined(__TMPM4K4A_H__) || defined(__TMPM4KN_H__))
#define	__AMP_UNIT_A
#define	__AMP_UNIT_B
#define	__AMP_UNIT_C
#endif

/* Setting CMP type */
#if defined(__TMPM370_H__)
#define	__CMP
#endif

/* Setting CMP Unit */
#if defined(__TMPM370_H__)
#define	__CMP_UNIT_A
#define	__CMP_UNIT_B
#define	__CMP_UNIT_C
#define	__CMP_UNIT_D
#endif



/* Setting SIO/UART type */
#if (defined(__TMPM370_H__) || defined(__TMPM372_H__) || defined(__TMPM373_H__) || defined(__TMPM374_H__) || defined(__TMPM376_H__) || defined(__TMPM4K4A_H__) || defined(__TMPM4KN_H__))
#define	__SIO
#define	__UART
#elif (defined(__TMPM375_H__) || defined(__TMPM37A_H__))
#define	__SIO2
#define	__UART2
#elif (defined(__TMPM470_H__) || defined(__TMPM475_H__))
#define	__SIO2
#define	__UART2
#endif


/* Setting PMD Unit */
#if (defined(__TMPM4KN_H__))
#define __PMD_CH0
#define __PMD_CH1
#define __PMD_CH2
#elif (defined(__TMPM370_H__) || defined(__TMPM376_H__) || defined(__TMPM470_H__) || defined(__TMPM475_H__) || defined(__TMPM4K4A_H__))
#define	__PMD_CH0
#define	__PMD_CH1
#elif (defined(__TMPM372_H__) || defined(__TMPM373_H__) || defined(__TMPM374_H__) || defined(__TMPM375_H__) || defined(__TMPM37A_H__))
#define	__PMD_CH1
#endif

/* Setting VE Unit */
#if (defined(__TMPM370_H__) || defined(__TMPM376_H__) || defined(__TMPM470_H__) || defined(__TMPM475_H__))
#define	__VE_CH0
#define	__VE_CH1
#elif (defined(__TMPM372_H__) || defined(__TMPM373_H__) || defined(__TMPM374_H__) || defined(__TMPM375_H__) || defined(__TMPM37A_H__))
#define	__VE_CH1
#elif (defined(__TMPM4K4A_H__) || defined(__TMPM4KN_H__))
#define	__VE_CH0
#endif


/* Setting ADC Unit */
#if (defined(__TMPM4K4A_H__))
#define	__ADC_UNITA
#elif (defined(__TMPM370_H__) || defined(__TMPM376_H__) || defined(__TMPM470_H__) || defined(__TMPM475_H__))
#define	__ADC_UNITA
#define	__ADC_UNITB
#elif (defined(__TMPM372_H__) || defined(__TMPM373_H__) || defined(__TMPM374_H__) || defined(__TMPM375_H__) || defined(__TMPM37A_H__))
#define	__ADC_UNITB
#endif

/* Setting Enc Unit */
#if (defined(__TMPM4KN_H__))
#define __ENC_CH0
#define __ENC_CH1
#define __ENC_CH2
#elif (defined(__TMPM370_H__) || defined(__TMPM376_H__) || defined(__TMPM470_H__) || defined(__TMPM475_H__))
#define	__ENC_CH0
#define	__ENC_CH1
#elif (defined(__TMPM372_H__) || defined(__TMPM373_H__) || defined(__TMPM374_H__))
#define	__ENC_CH1
#elif (defined(__TMPM375_H__) || defined(__TMPM4K4A_H__))
#define	__ENC_CH0
#endif

/* Setting SC Unit */
#if (defined(__TMPM370_H__) || defined(__TMPM376_H__) || defined(__TMPM372_H__) || defined(__TMPM470_H__) || defined(__TMPM475_H__))
#define	__SC_CH0
#define	__SC_CH1
#define	__SC_CH2
#define	__SC_CH3
#elif (defined(__TMPM373_H__) || defined(__TMPM374_H__))
#define	__SC_CH0
#define	__SC_CH1
#define	__SC_CH3
#elif defined(__TMPM375_H__)
#define	__SC_CH0
#define	__SC_CH1
#elif defined(__TMPM37A_H__)
#define	__SC_CH0
#endif



/*===================================================================*
	  Interruption leve define
 *===================================================================*/
/* High   Low */
/*	0 ---- 7  */

#define	INT4KH_LEVEL			5				/* 4kH interval timer interrupt */
#define	INT_VC_LEVEL			3				/* VE interrupt */
#define	INT_ADC_LEVEL			3				/* ADC interrupt */
#define	INT_DAC_LEVEL			6				/* SIO interrupt for Dac */
#define	INT_UART_LEVEL			6				/* UART interrupt */

/* CH of PMD,VE */
#define	cCH0					0
#define	cCH1					1
#define cCH2					2



/*****************************************************************
 * Constant data set
 *****************************************************************/
#if defined(__A_PMD)
  #define PWMRate(x)			((uint16_t)((1 << 24U) / (cIMCLK_FRQ * x)))
  #define DeadTime(x)			((uint8_t)((cIMCLK_FRQ * 1000000U * x) / 1000000U / 4U))
#elif defined(__PMD)
  #define PWMPeriod(x)			((uint16_t)((cIMCLK_FRQ * x) / 2U))		/* triangle wave */
  #define DeadTime(x)			((uint8_t)((cIMCLK_FRQ * 1000000U * x)	/ 1000000U / 8U))
#elif defined(__P_PMD)
  #define PWMPeriod(x)			((uint16_t)((cIMCLK_FRQ * x) / 2U))		/* triangle wave */
  #define DeadTime(x)			((uint8_t)((cIMCLK_FRQ * 1000000U * x)	/ 1000000U / 8U))
#endif

#define ADC_V2HEX(x)			((uint16_t)((uint16_t)((0xfff * x) / 5) << 4U))

/* TMRB */
#define	TMRB_DIV2_PRD(x)		(uint16_t)( (uint32_t)x * cIMCLK_FRQ / 2)
#define	TMRB_DIV8_PRD(x)		(uint16_t)( (uint32_t)x * cIMCLK_FRQ / 8)
#define	TMRB_DIV32_PRD(x)		(uint16_t)( (uint32_t)x * cIMCLK_FRQ / 32)
#define	TMRB_DIV64_PRD(x)		(uint16_t)( (uint32_t)x * cIMCLK_FRQ / 64)
#define	TMRB_DIV128_PRD(x)		(uint16_t)( (uint32_t)x * cIMCLK_FRQ / 128)
#define	TMRB_DIV256_PRD(x)		(uint16_t)( (uint32_t)x * cIMCLK_FRQ / 256)
#define	TMRB_DIV512_PRD(x)		(uint16_t)( (uint32_t)x * cIMCLK_FRQ / 512)

/*****************************************************************
 * Define The Typedef
 *****************************************************************/

typedef enum
{
	cStop		  = 0,
	cBootstrap	  = 1,
	cInitposition = 2,
	cForce		  = 3,
	cChange_up	  = 4,
	cSteady_A	  = 5,
	cEmergency	  = 6
} main_stage_e;

typedef enum
{
	cStep0		  = 0,
	cStep1		  = 1,
	cStep2		  = 2,
	cStep3		  = 3,
	cStepEnd	  = 4
} sub_stage_e;

typedef enum
{
	ciStop				= 0,
	ciBootstrap			= 1,
	ciInitposition_i	= 2,
	ciInitposition_v	= 3,
	ciForce_i			= 4,
	ciForce_v			= 5,
	ciChange_up			= 6,
	ciSteady_A			= 7,
	ciEmergency			= 8
} itr_stage_e;

typedef enum
{
	c1shunt		= 1,
	c3shunt		= 3
} shunt_type_e;

typedef enum
{
	cBoot_i		= 0,
	cBoot_v		= 1
} boot_type_e;

typedef enum
{
	cPHCVMD_SVM			= 0,
	cPHCVMD_I_CLARKE	= 1
} phcvmd_e;

typedef enum
{
	cTRGPOS_3SHUNT				= 0,
	cTRGPOS_1SHUNT_FIRST_HALF	= 1,
	cTRGPOS_1SHUNT_SECOND_HALF	= 2
} trgpos_e;

typedef struct
{
	main_stage_e main;
	sub_stage_e sub;
	itr_stage_e itr;
} stage_t;


#define	c3PHASE		0
#define	c2PHASE		1


typedef struct
{
	uint16_t reserve: 12; 						/* reserve */
	uint16_t spwm: 1; 							/* Shift PWM		0=off					1=on								*/
	uint16_t modul: 1;							/* PWM Moduration	0=3phase modulation 	1=2phase modulation 				*/
	uint16_t encoder: 1;						/* Position detect	0=Current				1=Encoder							*/
	uint16_t onoff: 1;							/* PWM output		0=off					1=on								*/
} command_t;

typedef struct
{
	uint16_t reserve: 9;						/* reserve */
	uint16_t F_vcomm_theta: 1;					/* Omega to Theta	0=command value 		1=Calculate the theta from omega.	*/
	uint16_t F_vcomm_omega: 1;					/* Omega by 		0=command value 		1=Result of Estimation position 	*/
	uint16_t F_vcomm_current: 1;				/* Current by		0=command value 		1=Result of Speed Control			*/
	uint16_t F_vcomm_volt: 1; 					/* Voltage by		0=command value 		1=Result of Current Control 		*/
	uint16_t F_vcomm_Edetect: 1;				/* Position detect	0=off					1=on								*/
	uint16_t F_vcomm_Idetect: 1;				/* Current detect	0=off					1=on								*/
	uint16_t F_vcomm_onoff: 1;					/* Motor output		0=off					1=on								*/
} vectorcmd_t;


typedef union
{
	struct
	{
		uint16_t reserve: 11; 					/* reserve */
		uint16_t Loss_sync: 1;					/* 0:normal, 1: Loss of synchronism		*/
		uint16_t emg_DC: 1;						/* 0:normal, 1: Over Vdc				*/
		uint16_t emg_I: 1;						/* 0:normal, 1: Current detect error	*/
		uint16_t emg_S: 1;						/* 0:normal, 1: Over current(soft)		*/
		uint16_t emg_H: 1;						/* 0:normal, 1: Over current(hard)		*/
	} flg;
	uint16_t	all;
} state_t;

typedef struct
{

	stage_t				stage;					/* Stage */
#if defined(__MOTOR_DEBUG)
	main_stage_e	stage_his[5];
#endif

	struct
	{
		q31_u			omega_com;				/* [Hz/maxHz] Command of Omega(electrical angle) */
		q31_u			omega;					/* [Hz/maxHz] Omega(speed): Electrical angle */
		q15_t			omega_dev;				/* [Hz/maxHz] Omega dev			 			*/

		q31_u			Id_com;					/* [A/maxA] Command of d-axis current */
		q31_u			Iq_com;					/* [A/maxA] Command of q-axis current */
		q15_t			Id_ref;					/* [A/maxA] Referense of d-axis current */
		q15_t			Iq_ref;					/* [A/maxA] Referense of q-axis current */
		q15_t			Id;						/* [A/maxA] d-axis current */
		q15_t			Iq;						/* [A/maxA] q-axis current */

		q31_u			Iq_ref_I; 				/* [A/maxA] Referense of q-axis current (Integral) */

		uint16_t		theta_com;				/* [deg/360] Command of theta of motor */
		uint32_u		theta;					/* [deg/360] Theta(position): Electrical angle */

		q15_t			Vdc;					/* [V/maxV] DC voltage */
		q31_u			Vdc_ave;				/* [V/maxV] DC voltage average */

		q31_u			Vd;						/* [V/maxV] d-axis voltage */
		q31_u			Vq;						/* [V/maxV] q-axis voltage */
		q31_u			Vd_ave;					/* [V/maxV] d-axis voltage */
		q31_u			Vq_ave;					/* [V/maxV] q-axis voltage */

		q15_t			Vdq;					/* [V/maxV] dq-axis voltage root3 */
		q31_u			Vdq_ave;				/* [V/maxV] dq-axis voltage root3 average */
		float			Vdq_per;

		q31_t			Vd_out;					/* [V/maxV] d-axis output voltage for VOLT BOOT */

		q15_t			Ed;						/* [V/maxV] d-axis induced voltage	*/
		q15_t			Eq;						/* [V/maxV] q-axis induced voltage	*/

		q31_t			Ed_I;					/* d-axis induced voltage I */
		q31_t			Ed_PI;					/* d-axis induced voltage PI */

		state_t			state;					/* Motor error state */
		command_t		command;				/* Command instruction */
		vectorcmd_t		vector_cmd;				/* Vector control command */

		q15_t			spwm_threshold;			/* [Hz/maxHz] Shift-PWM threshold */
		uint16_t		chkpls;					/* [---] Check pulse width setting */
		uint8_t			idetect_error;			/* current detect error state */

		q15_t			Ia_raw;					/* [A/maxA] a phase current Raw data */
		q15_t			Ib_raw;					/* [A/maxA] b phase current Raw data */
		q15_t			Ic_raw;					/* [A/maxA] c phase current Raw data */

		q15_t			Ia;						/* [A/maxA] a phase current */
		q15_t			Ib;						/* [A/maxA] b phase current */
		q15_t			Ic;						/* [A/maxA] c phase current */

		q31_u			Iao_ave;				/* [A/maxA] a phase average of zero current */
		q31_u			Ibo_ave;				/* [A/maxA] b phase average of zero current */
		q31_u			Ico_ave;				/* [A/maxA] c phase average of zero current */

		uint8_t			spdprd;					/* Speed pi period counter */
	} drv;

	struct
	{
		q31_u			omega_user;				/* [Hz/maxHz] Target omega by user */

		q15_t			Id_st_user;				/* [A/maxA] Start d-axis current by user */
		q15_t			Iq_st_user;				/* [A/maxA] Start q-axis current by user */
		uint16_t		lambda_user;			/* [deg/360] Init position(Lambda) by user */

		command_t		com_user; 				/* Control command by User order */
		command_t		com_user_1;				/* before step Scom_user */

	} usr;

	struct
	{
		q15_t			omega_min;				/* [Hz/maxHz] min omega */
		q15_t			omega_v2i;				/* [Hz/maxHz] Switching speed from the voltage type control to the current type control */

		q15_t			spwm_threshold;			/* [Hz/maxHz] Shift-PWM threshold */

		q31_t			vd_pos;					/* [V/maxV] d-axis Output Voltage */
		q31_t			spd_coef;				/* [---] speed coefficient */

		q31_u			sp_ud_lim_f; 			/* Limitation of the increase value (Forced Stage) */
		q31_u			sp_up_lim_s; 			/* Limitation of the increase value (Steady Stage) */
		q31_u			sp_dn_lim_s; 			/* Limitation of the decrease value (Steady Stage) */

		struct
		{
			uint16_t	initpos;				/* Time of Positioning State */
			uint16_t	initpos2;				/* Time of Positioning Wait State */
			uint16_t	bootstp;				/* Time of Bootstrap State */
			uint16_t	go_up;					/* Time of Change-up state */
		} time;

		q31_t			iq_lim;					/* [A/maxA] Limitation of Iq */
		q31_t			id_lim;					/* [A/maxA] Limitation of Id */
		q15_t			err_ovc;				/* [A/maxA] Over current */

		struct
		{
			q31_t		kp;						/* [---] Normalize data Pos Kp */
			q31_t		ki;						/* [---] Normalize data Pos Ki */
			int32_t		ctrlprd;				/* [---] Normalize data ctrlprd */
		} pos;

		struct
		{
			q31_t		kp;						/* [---] Normalize data Spd Kp */
			q31_t		ki;						/* [---] Normalize data Spd Ki */
			uint8_t		pi_prd;					/* [cnt] Speed PI period count */
		} spd;

		struct
		{
			q31_t		dkp;					/* [---] Normalize data Current(Id) Kp */
			q31_t		dki;					/* [---] Normalize data Current(Id) Ki */
			q31_t		qkp;					/* [---] Normalize data Current(Iq) Kp */
			q31_t		qki;					/* [---] Normalize data Current(Iq) Ki */
		} crt;
		
		struct
		{
			q31_t		r;						/* [---] Normalize data motor R */
			q31_t		Lq;						/* [---] Normalize data motor Lq */
			q31_t		Ld;						/* [---] Normalize data motor Ld */
		} motor;

		int32_t			delta_lambda;			/* [---] Change_up stage lamda */

		uint16_t		chkpls;					/* [---] Check pulse width setting */
	} para;

	uint32_t			stage_counter;			/* Stage counter */
	shunt_type_e		shunt_type;				/* Shunt type */
	boot_type_e			boot_type;				/* Boot type */

#if defined(__USE_FIELD_WEAKING)
	fldwk_t				Fldwk;					/* Field-weaking */
	spdlim_t			Spdlim;					/* Speed limit */
#endif /* defined(__USE_FIELD_WEAKING) */

#if defined(__USE_DECOUPLING)
	decouple_t			Decouple;				/* Decoupling control */
#endif /* defined(__USE_DECOUPLING) */

#if defined(__USE_DEADTIME)
	deadtime_t			Deadtime;				/* Deadtime control */
#endif /* defined(__USE_DEADTIME) */

} vector_t;



/*****************************************************************
 * Definition of variable
 *****************************************************************/
EXTERN uint16_t		M_Main_Counter; 		/* Main loop counter */

EXTERN vector_t		Motor_ch1;				/* Motor data for ch1 */

EXTERN	uint32_t		errdet_ch0;
EXTERN	uint32_t		errdet_ch1;

/*===================================================================*
  Includes
 *===================================================================*/
#include "ipdrv.h"

/*===================================================================*
  global functions.
 *===================================================================*/
/****************************************************************************************
 * Module	: calc_ave
 *---------------------------------------------------------------------------------------
 * Function : cals Average
 * Input	: ave, val, shift
 * Output	: Average
 * Note 	:
 ****************************************************************************************/
static __INLINE int32_t calc_ave(int32_t ave, int32_t val, int32_t shift)
{
	return (ave + ((val - ave) >> shift));
}


#endif /* _IPDEFINE_H */


/*************************** END OF FILE **************************************/
